/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.fs;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hudi.common.config.HoodieMetadataConfig;
import org.apache.hudi.common.config.SerializableConfiguration;
import org.apache.hudi.common.engine.HoodieEngineContext;
import org.apache.hudi.common.fs.ConsistencyGuardConfig;
import org.apache.hudi.common.fs.FailSafeConsistencyGuard;
import org.apache.hudi.common.fs.HoodieWrapperFileSystem;
import org.apache.hudi.common.fs.NoOpConsistencyGuard;
import org.apache.hudi.common.fs.StorageSchemes;
import org.apache.hudi.common.model.HoodieFileFormat;
import org.apache.hudi.common.model.HoodieLogFile;
import org.apache.hudi.common.table.HoodieTableConfig;
import org.apache.hudi.common.table.view.FileSystemViewStorageConfig;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.common.util.collection.ImmutablePair;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.exception.InvalidHoodiePathException;
import org.apache.hudi.hadoop.CachingPath;
import org.apache.hudi.metadata.HoodieTableMetadata;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public class FSUtils {
    private static final Logger LOG = LogManager.getLogger(FSUtils.class);
    public static final Pattern LOG_FILE_PATTERN = Pattern.compile("\\.(.*)_(.*)\\.(.*)\\.([0-9]*)(_(([0-9]*)-([0-9]*)-([0-9]*)))?");
    private static final int MAX_ATTEMPTS_RECOVER_LEASE = 10;
    private static final long MIN_CLEAN_TO_KEEP = 10L;
    private static final long MIN_ROLLBACK_TO_KEEP = 10L;
    private static final String HOODIE_ENV_PROPS_PREFIX = "HOODIE_ENV_";
    private static final PathFilter ALLOW_ALL_FILTER = file -> true;

    public static Configuration prepareHadoopConf(Configuration conf) {
        for (Map.Entry<String, String> prop : System.getenv().entrySet()) {
            if (!prop.getKey().startsWith(HOODIE_ENV_PROPS_PREFIX)) continue;
            LOG.info((Object)("Picking up value for hoodie env var :" + prop.getKey()));
            conf.set(prop.getKey().replace(HOODIE_ENV_PROPS_PREFIX, "").replaceAll("_DOT_", "."), prop.getValue());
        }
        return conf;
    }

    public static FileSystem getFs(String pathStr, Configuration conf) {
        return FSUtils.getFs(new Path(pathStr), conf);
    }

    public static FileSystem getFs(Path path, Configuration conf) {
        FileSystem fs;
        FSUtils.prepareHadoopConf(conf);
        try {
            fs = path.getFileSystem(conf);
        }
        catch (IOException e) {
            throw new HoodieIOException("Failed to get instance of " + FileSystem.class.getName(), e);
        }
        return fs;
    }

    public static FileSystem getFs(String pathStr, Configuration conf, boolean localByDefault) {
        if (localByDefault) {
            return FSUtils.getFs(FSUtils.addSchemeIfLocalPath(pathStr), conf);
        }
        return FSUtils.getFs(pathStr, conf);
    }

    public static boolean isTableExists(String path, FileSystem fs) throws IOException {
        return fs.exists(new Path(path + "/" + ".hoodie"));
    }

    public static Path addSchemeIfLocalPath(String path) {
        Path providedPath = new Path(path);
        File localFile = new File(path);
        if (!providedPath.isAbsolute() && localFile.exists()) {
            Path resolvedPath = new Path("file://" + localFile.getAbsolutePath());
            LOG.info((Object)("Resolving file " + path + " to be a local file."));
            return resolvedPath;
        }
        LOG.info((Object)("Resolving file " + path + "to be a remote file."));
        return providedPath;
    }

    public static Path makeQualified(FileSystem fs, Path path) {
        return path.makeQualified(fs.getUri(), fs.getWorkingDirectory());
    }

    public static String makeWriteToken(int taskPartitionId, int stageId, long taskAttemptId) {
        return String.format("%d-%d-%d", taskPartitionId, stageId, taskAttemptId);
    }

    public static String makeBaseFileName(String instantTime, String writeToken, String fileId) {
        return String.format("%s_%s_%s%s", fileId, writeToken, instantTime, HoodieTableConfig.BASE_FILE_FORMAT.defaultValue().getFileExtension());
    }

    public static String makeBaseFileName(String instantTime, String writeToken, String fileId, String fileExtension) {
        return String.format("%s_%s_%s%s", fileId, writeToken, instantTime, fileExtension);
    }

    public static String makeBootstrapIndexFileName(String instantTime, String fileId, String ext) {
        return String.format("%s_%s_%s%s", fileId, "1-0-1", instantTime, ext);
    }

    public static String maskWithoutFileId(String instantTime, int taskPartitionId) {
        return String.format("*_%s_%s%s", taskPartitionId, instantTime, HoodieTableConfig.BASE_FILE_FORMAT.defaultValue().getFileExtension());
    }

    public static String getCommitFromCommitFile(String commitFileName) {
        return commitFileName.split("\\.")[0];
    }

    public static String getCommitTime(String fullFileName) {
        if (FSUtils.isLogFile(fullFileName)) {
            return fullFileName.split("_")[1].split("\\.")[0];
        }
        return fullFileName.split("_")[2].split("\\.")[0];
    }

    public static long getFileSize(FileSystem fs, Path path) throws IOException {
        return fs.getFileStatus(path).getLen();
    }

    public static String getFileId(String fullFileName) {
        return fullFileName.split("_")[0];
    }

    public static List<String> getAllPartitionFoldersThreeLevelsDown(FileSystem fs, String basePath) throws IOException {
        FileStatus[] folders;
        ArrayList<String> datePartitions = new ArrayList<String>();
        PathFilter filter = FSUtils.getExcludeMetaPathFilter();
        for (FileStatus status : folders = fs.globStatus(new Path(basePath + "/*/*/*"), filter)) {
            Path path = status.getPath();
            datePartitions.add(String.format("%s/%s/%s", path.getParent().getParent().getName(), path.getParent().getName(), path.getName()));
        }
        return datePartitions;
    }

    public static String getRelativePartitionPath(Path basePath, Path fullPartitionPath) {
        basePath = CachingPath.getPathWithoutSchemeAndAuthority(basePath);
        String fullPartitionPathStr = (fullPartitionPath = CachingPath.getPathWithoutSchemeAndAuthority(fullPartitionPath)).toString();
        if (!fullPartitionPathStr.startsWith(basePath.toString())) {
            throw new IllegalArgumentException("Partition path does not belong to base-path");
        }
        int partitionStartIndex = fullPartitionPathStr.indexOf(basePath.getName(), basePath.getParent() == null ? 0 : basePath.getParent().toString().length());
        return partitionStartIndex + basePath.getName().length() == fullPartitionPathStr.length() ? "" : fullPartitionPathStr.substring(partitionStartIndex + basePath.getName().length() + 1);
    }

    public static List<String> getAllFoldersWithPartitionMetaFile(FileSystem fs, String basePathStr) throws IOException {
        boolean isMetadataTable = HoodieTableMetadata.isMetadataTable(basePathStr);
        Path basePath = new Path(basePathStr);
        ArrayList<String> partitions = new ArrayList<String>();
        FSUtils.processFiles(fs, basePathStr, locatedFileStatus -> {
            Path filePath = locatedFileStatus.getPath();
            if (filePath.getName().startsWith(".hoodie_partition_metadata")) {
                partitions.add(FSUtils.getRelativePartitionPath(basePath, filePath.getParent()));
            }
            return true;
        }, !isMetadataTable);
        return partitions;
    }

    public static void processFiles(FileSystem fs, String basePathStr, Function<FileStatus, Boolean> consumer, boolean excludeMetaFolder) throws IOException {
        FileStatus[] topLevelStatuses;
        PathFilter pathFilter = excludeMetaFolder ? FSUtils.getExcludeMetaPathFilter() : ALLOW_ALL_FILTER;
        for (FileStatus child : topLevelStatuses = fs.listStatus(new Path(basePathStr))) {
            if (child.isFile()) {
                boolean success = consumer.apply(child);
                if (success) continue;
                throw new HoodieException("Failed to process file-status=" + child);
            }
            if (!pathFilter.accept(child.getPath())) continue;
            RemoteIterator itr = fs.listFiles(child.getPath(), true);
            while (itr.hasNext()) {
                FileStatus status = (FileStatus)itr.next();
                boolean success = consumer.apply(status);
                if (success) continue;
                throw new HoodieException("Failed to process file-status=" + status);
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static List<String> getAllPartitionPaths(HoodieEngineContext engineContext, String basePathStr, boolean useFileListingFromMetadata, boolean assumeDatePartitioning) {
        HoodieMetadataConfig metadataConfig = HoodieMetadataConfig.newBuilder().enable(useFileListingFromMetadata).withAssumeDatePartitioning(assumeDatePartitioning).build();
        try (HoodieTableMetadata tableMetadata = HoodieTableMetadata.create(engineContext, metadataConfig, basePathStr, FileSystemViewStorageConfig.SPILLABLE_DIR.defaultValue());){
            List<String> list = tableMetadata.getAllPartitionPaths();
            return list;
        }
        catch (Exception e) {
            throw new HoodieException("Error fetching partition paths from metadata table", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static List<String> getAllPartitionPaths(HoodieEngineContext engineContext, HoodieMetadataConfig metadataConfig, String basePathStr) {
        try (HoodieTableMetadata tableMetadata = HoodieTableMetadata.create(engineContext, metadataConfig, basePathStr, FileSystemViewStorageConfig.SPILLABLE_DIR.defaultValue());){
            List<String> list = tableMetadata.getAllPartitionPaths();
            return list;
        }
        catch (Exception e) {
            throw new HoodieException("Error fetching partition paths from metadata table", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static Map<String, FileStatus[]> getFilesInPartitions(HoodieEngineContext engineContext, HoodieMetadataConfig metadataConfig, String basePathStr, String[] partitionPaths, String spillableMapPath) {
        try (HoodieTableMetadata tableMetadata = HoodieTableMetadata.create(engineContext, metadataConfig, basePathStr, spillableMapPath, true);){
            Map<String, FileStatus[]> map = tableMetadata.getAllFilesInPartitions(Arrays.asList(partitionPaths));
            return map;
        }
        catch (Exception ex) {
            throw new HoodieException("Error get files in partitions: " + String.join((CharSequence)",", partitionPaths), ex);
        }
    }

    public static String getFileExtension(String fullName) {
        Objects.requireNonNull(fullName);
        String fileName = new File(fullName).getName();
        int dotIndex = fileName.lastIndexOf(46);
        return dotIndex == -1 ? "" : fileName.substring(dotIndex);
    }

    private static PathFilter getExcludeMetaPathFilter() {
        return path -> !path.toString().contains(".hoodie");
    }

    public static String createNewFileIdPfx() {
        return UUID.randomUUID().toString();
    }

    public static String createNewFileId(String idPfx, int id) {
        return String.format("%s-%d", idPfx, id);
    }

    public static String getFileExtensionFromLog(Path logPath) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(logPath.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(logPath, "LogFile");
        }
        return matcher.group(3);
    }

    public static String getFileIdFromLogPath(Path path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path, "LogFile");
        }
        return matcher.group(1);
    }

    public static String getFileIdFromFilePath(Path filePath) {
        if (FSUtils.isLogFile(filePath)) {
            return FSUtils.getFileIdFromLogPath(filePath);
        }
        return FSUtils.getFileId(filePath.getName());
    }

    public static String getBaseCommitTimeFromLogPath(Path path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path, "LogFile");
        }
        return matcher.group(2);
    }

    public static Integer getTaskPartitionIdFromLogPath(Path path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path, "LogFile");
        }
        String val = matcher.group(7);
        return val == null ? null : Integer.valueOf(Integer.parseInt(val));
    }

    public static String getWriteTokenFromLogPath(Path path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path, "LogFile");
        }
        return matcher.group(6);
    }

    public static Integer getStageIdFromLogPath(Path path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path, "LogFile");
        }
        String val = matcher.group(8);
        return val == null ? null : Integer.valueOf(Integer.parseInt(val));
    }

    public static Integer getTaskAttemptIdFromLogPath(Path path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path, "LogFile");
        }
        String val = matcher.group(9);
        return val == null ? null : Integer.valueOf(Integer.parseInt(val));
    }

    public static int getFileVersionFromLog(Path logPath) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(logPath.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(logPath, "LogFile");
        }
        return Integer.parseInt(matcher.group(4));
    }

    public static String makeLogFileName(String fileId, String logFileExtension, String baseCommitTime, int version, String writeToken) {
        String suffix = writeToken == null ? String.format("%s_%s%s.%d", fileId, baseCommitTime, logFileExtension, version) : String.format("%s_%s%s.%d_%s", fileId, baseCommitTime, logFileExtension, version, writeToken);
        return "." + suffix;
    }

    public static boolean isBaseFile(Path path) {
        String extension = FSUtils.getFileExtension(path.getName());
        return HoodieFileFormat.BASE_FILE_EXTENSIONS.contains(extension);
    }

    public static boolean isLogFile(Path logPath) {
        return FSUtils.isLogFile(logPath.getName());
    }

    public static boolean isLogFile(String fileName) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(fileName);
        return matcher.find() && fileName.contains(".log");
    }

    public static boolean isDataFile(Path path) {
        return FSUtils.isBaseFile(path) || FSUtils.isLogFile(path);
    }

    public static FileStatus[] getAllDataFilesInPartition(FileSystem fs, Path partitionPath) throws IOException {
        Set validFileExtensions = Arrays.stream(HoodieFileFormat.values()).map(HoodieFileFormat::getFileExtension).collect(Collectors.toCollection(HashSet::new));
        String logFileExtension = HoodieFileFormat.HOODIE_LOG.getFileExtension();
        try {
            return (FileStatus[])Arrays.stream(fs.listStatus(partitionPath, path -> {
                String extension = FSUtils.getFileExtension(path.getName());
                return validFileExtensions.contains(extension) || path.getName().contains(logFileExtension);
            })).filter(FileStatus::isFile).toArray(FileStatus[]::new);
        }
        catch (IOException e) {
            if (!fs.exists(partitionPath)) {
                return new FileStatus[0];
            }
            throw e;
        }
    }

    public static Option<HoodieLogFile> getLatestLogFile(FileSystem fs, Path partitionPath, String fileId, String logFileExtension, String baseCommitTime) throws IOException {
        return FSUtils.getLatestLogFile(FSUtils.getAllLogFiles(fs, partitionPath, fileId, logFileExtension, baseCommitTime));
    }

    public static Stream<HoodieLogFile> getAllLogFiles(FileSystem fs, Path partitionPath, String fileId, String logFileExtension, String baseCommitTime) throws IOException {
        try {
            PathFilter pathFilter = path -> path.getName().startsWith("." + fileId) && path.getName().contains(logFileExtension);
            return Arrays.stream(fs.listStatus(partitionPath, pathFilter)).map(HoodieLogFile::new).filter(s -> s.getBaseCommitTime().equals(baseCommitTime));
        }
        catch (FileNotFoundException e) {
            return Stream.of(new HoodieLogFile[0]);
        }
    }

    public static Option<Pair<Integer, String>> getLatestLogVersion(FileSystem fs, Path partitionPath, String fileId, String logFileExtension, String baseCommitTime) throws IOException {
        Option<HoodieLogFile> latestLogFile = FSUtils.getLatestLogFile(FSUtils.getAllLogFiles(fs, partitionPath, fileId, logFileExtension, baseCommitTime));
        if (latestLogFile.isPresent()) {
            return Option.of(Pair.of(latestLogFile.get().getLogVersion(), FSUtils.getWriteTokenFromLogPath(latestLogFile.get().getPath())));
        }
        return Option.empty();
    }

    public static int computeNextLogVersion(FileSystem fs, Path partitionPath, String fileId, String logFileExtension, String baseCommitTime) throws IOException {
        Option<Pair<Integer, String>> currentVersionWithWriteToken = FSUtils.getLatestLogVersion(fs, partitionPath, fileId, logFileExtension, baseCommitTime);
        return currentVersionWithWriteToken.isPresent() ? currentVersionWithWriteToken.get().getKey() + 1 : HoodieLogFile.LOGFILE_BASE_VERSION;
    }

    public static int getDefaultBufferSize(FileSystem fs) {
        return fs.getConf().getInt("io.file.buffer.size", 4096);
    }

    public static Short getDefaultReplication(FileSystem fs, Path path) {
        return fs.getDefaultReplication(path);
    }

    public static boolean recoverDFSFileLease(DistributedFileSystem dfs, Path p) throws IOException, InterruptedException {
        LOG.info((Object)("Recover lease on dfs file " + p));
        boolean recovered = false;
        for (int nbAttempt = 0; nbAttempt < 10; ++nbAttempt) {
            LOG.info((Object)("Attempt " + nbAttempt + " to recover lease on dfs file " + p));
            recovered = dfs.recoverLease(p);
            if (recovered) break;
            Thread.sleep(1000L);
        }
        return recovered;
    }

    public static void createPathIfNotExists(FileSystem fs, Path partitionPath) throws IOException {
        if (!fs.exists(partitionPath)) {
            fs.mkdirs(partitionPath);
        }
    }

    public static Long getSizeInMB(long sizeInBytes) {
        return sizeInBytes / 0x100000L;
    }

    public static Path getPartitionPath(String basePath, String partitionPath) {
        if (StringUtils.isNullOrEmpty(partitionPath)) {
            return new Path(basePath);
        }
        String properPartitionPath = partitionPath.startsWith("/") ? partitionPath.substring(1) : partitionPath;
        return FSUtils.getPartitionPath(new CachingPath(basePath), properPartitionPath);
    }

    public static Path getPartitionPath(Path basePath, String partitionPath) {
        return StringUtils.isNullOrEmpty(partitionPath) ? basePath : new CachingPath(basePath, partitionPath);
    }

    public static String getFileName(String filePathWithPartition, String partition) {
        int offset = StringUtils.isNullOrEmpty(partition) ? (filePathWithPartition.startsWith("/") ? 1 : 0) : partition.length() + 1;
        return filePathWithPartition.substring(offset);
    }

    public static String getDFSFullPartitionPath(FileSystem fs, Path fullPartitionPath) {
        return fs.getUri() + fullPartitionPath.toUri().getRawPath();
    }

    public static boolean isGCSFileSystem(FileSystem fs) {
        return fs.getScheme().equals(StorageSchemes.GCS.getScheme());
    }

    public static boolean isCHDFileSystem(FileSystem fs) {
        return StorageSchemes.CHDFS.getScheme().equals(fs.getScheme());
    }

    public static Configuration registerFileSystem(Path file, Configuration conf) {
        Configuration returnConf = new Configuration(conf);
        String scheme = FSUtils.getFs(file.toString(), conf).getScheme();
        returnConf.set("fs." + HoodieWrapperFileSystem.getHoodieScheme(scheme) + ".impl", HoodieWrapperFileSystem.class.getName());
        return returnConf;
    }

    public static HoodieWrapperFileSystem getFs(String path, SerializableConfiguration hadoopConf, ConsistencyGuardConfig consistencyGuardConfig) {
        FileSystem fileSystem = FSUtils.getFs(path, hadoopConf.newCopy());
        return new HoodieWrapperFileSystem(fileSystem, consistencyGuardConfig.isConsistencyCheckEnabled() ? new FailSafeConsistencyGuard(fileSystem, consistencyGuardConfig) : new NoOpConsistencyGuard());
    }

    public static List<FileStatus> getGlobStatusExcludingMetaFolder(FileSystem fs, Path globPath) throws IOException {
        FileStatus[] statuses = fs.globStatus(globPath);
        return Arrays.stream(statuses).filter(fileStatus -> !fileStatus.getPath().toString().contains(".hoodie")).collect(Collectors.toList());
    }

    public static boolean deleteDir(HoodieEngineContext hoodieEngineContext, FileSystem fs, Path dirPath, int parallelism) {
        try {
            if (fs.exists(dirPath)) {
                FSUtils.parallelizeSubPathProcess(hoodieEngineContext, fs, dirPath, parallelism, e -> true, pairOfSubPathAndConf -> FSUtils.deleteSubPath((String)pairOfSubPathAndConf.getKey(), (SerializableConfiguration)pairOfSubPathAndConf.getValue(), true));
                boolean result = fs.delete(dirPath, false);
                LOG.info((Object)("Removed directory at " + dirPath));
                return result;
            }
        }
        catch (IOException ioe) {
            throw new HoodieIOException(ioe.getMessage(), ioe);
        }
        return false;
    }

    public static <T> Map<String, T> parallelizeSubPathProcess(HoodieEngineContext hoodieEngineContext, FileSystem fs, Path dirPath, int parallelism, Predicate<FileStatus> subPathPredicate, SerializableFunction<Pair<String, SerializableConfiguration>, T> pairFunction) {
        Map<Object, Object> result = new HashMap();
        try {
            FileStatus[] fileStatuses = fs.listStatus(dirPath);
            List<String> subPaths = Arrays.stream(fileStatuses).filter(subPathPredicate).map(fileStatus -> fileStatus.getPath().toString()).collect(Collectors.toList());
            result = FSUtils.parallelizeFilesProcess(hoodieEngineContext, fs, parallelism, pairFunction, subPaths);
        }
        catch (IOException ioe) {
            throw new HoodieIOException(ioe.getMessage(), ioe);
        }
        return result;
    }

    public static <T> Map<String, T> parallelizeFilesProcess(HoodieEngineContext hoodieEngineContext, FileSystem fs, int parallelism, SerializableFunction<Pair<String, SerializableConfiguration>, T> pairFunction, List<String> subPaths) {
        Map result = new HashMap();
        if (subPaths.size() > 0) {
            SerializableConfiguration conf = new SerializableConfiguration(fs.getConf());
            int actualParallelism = Math.min(subPaths.size(), parallelism);
            hoodieEngineContext.setJobStatus(FSUtils.class.getSimpleName(), "Parallel listing paths " + String.join((CharSequence)",", subPaths));
            result = hoodieEngineContext.mapToPair(subPaths, subPath -> new ImmutablePair((String)subPath, pairFunction.apply(new ImmutablePair<String, SerializableConfiguration>((String)subPath, conf))), actualParallelism);
        }
        return result;
    }

    public static boolean deleteSubPath(String subPathStr, SerializableConfiguration conf, boolean recursive) {
        try {
            Path subPath = new Path(subPathStr);
            FileSystem fileSystem = subPath.getFileSystem(conf.get());
            return fileSystem.delete(subPath, recursive);
        }
        catch (IOException e) {
            throw new HoodieIOException(e.getMessage(), e);
        }
    }

    public static List<FileStatus> getFileStatusAtLevel(HoodieEngineContext hoodieEngineContext, FileSystem fs, Path rootPath, int expectLevel, int parallelism) {
        List<String> levelPaths = new ArrayList<String>();
        List<Object> result = new ArrayList<FileStatus>();
        levelPaths.add(rootPath.toString());
        for (int i = 0; i <= expectLevel; ++i) {
            result = FSUtils.parallelizeFilesProcess(hoodieEngineContext, fs, parallelism, pairOfSubPathAndConf -> {
                Path path = new Path((String)pairOfSubPathAndConf.getKey());
                try {
                    FileSystem fileSystem = path.getFileSystem(((SerializableConfiguration)pairOfSubPathAndConf.getValue()).get());
                    return Arrays.stream(fileSystem.listStatus(path)).collect(Collectors.toList());
                }
                catch (IOException e) {
                    throw new HoodieIOException("Failed to list " + path, e);
                }
            }, levelPaths).values().stream().flatMap(list -> list.stream()).collect(Collectors.toList());
            if (i >= expectLevel) continue;
            levelPaths = result.stream().filter(FileStatus::isDirectory).map(fileStatus -> fileStatus.getPath().toString()).collect(Collectors.toList());
        }
        return result;
    }

    private static Option<HoodieLogFile> getLatestLogFile(Stream<HoodieLogFile> logFiles) {
        return Option.fromJavaOptional(logFiles.min(HoodieLogFile.getReverseLogFileComparator()));
    }

    public static interface SerializableFunction<T, R>
    extends Function<T, R>,
    Serializable {
    }
}

