/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.raptor.legacy.storage;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import io.airlift.concurrent.MoreFutures;
import io.airlift.concurrent.Threads;
import io.airlift.json.JsonCodec;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.airlift.slice.XxHash64;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import io.trino.memory.context.AggregatedMemoryContext;
import io.trino.orc.FileOrcDataSource;
import io.trino.orc.OrcColumn;
import io.trino.orc.OrcDataSource;
import io.trino.orc.OrcPredicate;
import io.trino.orc.OrcReader;
import io.trino.orc.OrcReaderOptions;
import io.trino.orc.OrcRecordReader;
import io.trino.orc.TupleDomainOrcPredicate;
import io.trino.plugin.base.CatalogName;
import io.trino.plugin.raptor.legacy.RaptorColumnHandle;
import io.trino.plugin.raptor.legacy.RaptorErrorCode;
import io.trino.plugin.raptor.legacy.backup.BackupManager;
import io.trino.plugin.raptor.legacy.backup.BackupStore;
import io.trino.plugin.raptor.legacy.metadata.ColumnInfo;
import io.trino.plugin.raptor.legacy.metadata.ColumnStats;
import io.trino.plugin.raptor.legacy.metadata.ShardDelta;
import io.trino.plugin.raptor.legacy.metadata.ShardInfo;
import io.trino.plugin.raptor.legacy.metadata.ShardRecorder;
import io.trino.plugin.raptor.legacy.storage.OrcFileMetadata;
import io.trino.plugin.raptor.legacy.storage.OrcFileRewriter;
import io.trino.plugin.raptor.legacy.storage.OrcFileWriter;
import io.trino.plugin.raptor.legacy.storage.RaptorPageSource;
import io.trino.plugin.raptor.legacy.storage.ShardRecoveryManager;
import io.trino.plugin.raptor.legacy.storage.ShardRewriter;
import io.trino.plugin.raptor.legacy.storage.ShardStats;
import io.trino.plugin.raptor.legacy.storage.StorageManager;
import io.trino.plugin.raptor.legacy.storage.StorageManagerConfig;
import io.trino.plugin.raptor.legacy.storage.StoragePageSink;
import io.trino.plugin.raptor.legacy.storage.StorageService;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.NodeManager;
import io.trino.spi.Page;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ConnectorPageSource;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.type.ArrayType;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.MapType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.TypeId;
import io.trino.spi.type.TypeManager;
import io.trino.spi.type.TypeSignature;
import io.trino.spi.type.TypeSignatureParameter;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import org.joda.time.DateTimeZone;

public class RaptorStorageManager
implements StorageManager {
    private static final JsonCodec<ShardDelta> SHARD_DELTA_CODEC = JsonCodec.jsonCodec(ShardDelta.class);
    private static final long MAX_ROWS = 1000000000L;
    private static final DataSize HUGE_MAX_READ_BLOCK_SIZE = DataSize.of((long)1L, (DataSize.Unit)DataSize.Unit.PETABYTE);
    private static final JsonCodec<OrcFileMetadata> METADATA_CODEC = JsonCodec.jsonCodec(OrcFileMetadata.class);
    private final String nodeId;
    private final StorageService storageService;
    private final Optional<BackupStore> backupStore;
    private final OrcReaderOptions orcReaderOptions;
    private final BackupManager backupManager;
    private final ShardRecoveryManager recoveryManager;
    private final ShardRecorder shardRecorder;
    private final Duration recoveryTimeout;
    private final long maxShardRows;
    private final DataSize maxShardSize;
    private final DataSize minAvailableSpace;
    private final TypeManager typeManager;
    private final ExecutorService deletionExecutor;
    private final ExecutorService commitExecutor;

    @Inject
    public RaptorStorageManager(NodeManager nodeManager, StorageService storageService, Optional<BackupStore> backupStore, StorageManagerConfig config, CatalogName catalogName, BackupManager backgroundBackupManager, ShardRecoveryManager recoveryManager, ShardRecorder shardRecorder, TypeManager typeManager) {
        this(nodeManager.getCurrentNode().getNodeIdentifier(), storageService, backupStore, config.toOrcReaderOptions(), backgroundBackupManager, recoveryManager, shardRecorder, typeManager, catalogName.toString(), config.getDeletionThreads(), config.getShardRecoveryTimeout(), config.getMaxShardRows(), config.getMaxShardSize(), config.getMinAvailableSpace());
    }

    public RaptorStorageManager(String nodeId, StorageService storageService, Optional<BackupStore> backupStore, OrcReaderOptions orcReaderOptions, BackupManager backgroundBackupManager, ShardRecoveryManager recoveryManager, ShardRecorder shardRecorder, TypeManager typeManager, String connectorId, int deletionThreads, Duration shardRecoveryTimeout, long maxShardRows, DataSize maxShardSize, DataSize minAvailableSpace) {
        this.nodeId = Objects.requireNonNull(nodeId, "nodeId is null");
        this.storageService = Objects.requireNonNull(storageService, "storageService is null");
        this.backupStore = Objects.requireNonNull(backupStore, "backupStore is null");
        this.orcReaderOptions = orcReaderOptions.withMaxReadBlockSize(HUGE_MAX_READ_BLOCK_SIZE);
        this.backupManager = Objects.requireNonNull(backgroundBackupManager, "backgroundBackupManager is null");
        this.recoveryManager = Objects.requireNonNull(recoveryManager, "recoveryManager is null");
        this.recoveryTimeout = Objects.requireNonNull(shardRecoveryTimeout, "shardRecoveryTimeout is null");
        Preconditions.checkArgument((maxShardRows > 0L ? 1 : 0) != 0, (Object)"maxShardRows must be > 0");
        this.maxShardRows = Math.min(maxShardRows, 1000000000L);
        this.maxShardSize = Objects.requireNonNull(maxShardSize, "maxShardSize is null");
        this.minAvailableSpace = Objects.requireNonNull(minAvailableSpace, "minAvailableSpace is null");
        this.shardRecorder = Objects.requireNonNull(shardRecorder, "shardRecorder is null");
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
        this.deletionExecutor = Executors.newFixedThreadPool(deletionThreads, Threads.daemonThreadsNamed((String)("raptor-delete-" + connectorId + "-%s")));
        this.commitExecutor = Executors.newCachedThreadPool(Threads.daemonThreadsNamed((String)("raptor-commit-" + connectorId + "-%s")));
    }

    @PreDestroy
    public void shutdown() {
        this.deletionExecutor.shutdownNow();
        this.commitExecutor.shutdown();
    }

    @Override
    public ConnectorPageSource getPageSource(UUID shardUuid, OptionalInt bucketNumber, List<Long> columnIds, List<Type> columnTypes, TupleDomain<RaptorColumnHandle> effectivePredicate, OrcReaderOptions orcReaderOptions) {
        orcReaderOptions = orcReaderOptions.withMaxReadBlockSize(HUGE_MAX_READ_BLOCK_SIZE);
        OrcDataSource dataSource = this.openShard(shardUuid, orcReaderOptions);
        AggregatedMemoryContext memoryUsage = AggregatedMemoryContext.newSimpleAggregatedMemoryContext();
        try {
            OrcReader reader = (OrcReader)OrcReader.createOrcReader((OrcDataSource)dataSource, (OrcReaderOptions)orcReaderOptions).orElseThrow(() -> new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Data file is empty for shard " + shardUuid));
            Map<Long, OrcColumn> indexMap = RaptorStorageManager.columnIdIndex(reader.getRootColumn().getNestedColumns());
            ArrayList<OrcColumn> fileReadColumn = new ArrayList<OrcColumn>(columnIds.size());
            ArrayList<Type> fileReadTypes = new ArrayList<Type>(columnIds.size());
            ArrayList<RaptorPageSource.ColumnAdaptation> columnAdaptations = new ArrayList<RaptorPageSource.ColumnAdaptation>(columnIds.size());
            for (int i = 0; i < columnIds.size(); ++i) {
                long columnId = columnIds.get(i);
                if (RaptorColumnHandle.isHiddenColumn(columnId)) {
                    columnAdaptations.add(RaptorStorageManager.specialColumnAdaptation(columnId, shardUuid, bucketNumber));
                    continue;
                }
                Type type = RaptorStorageManager.toOrcFileType(columnTypes.get(i), this.typeManager);
                OrcColumn fileColumn = indexMap.get(columnId);
                if (fileColumn == null) {
                    columnAdaptations.add(RaptorPageSource.ColumnAdaptation.nullColumn(type));
                    continue;
                }
                int sourceIndex = fileReadColumn.size();
                columnAdaptations.add(RaptorPageSource.ColumnAdaptation.sourceColumn(sourceIndex));
                fileReadColumn.add(fileColumn);
                fileReadTypes.add(type);
            }
            OrcPredicate predicate = RaptorStorageManager.getPredicate(effectivePredicate, indexMap);
            OrcRecordReader recordReader = reader.createRecordReader(fileReadColumn, fileReadTypes, predicate, DateTimeZone.UTC, memoryUsage, 1, RaptorPageSource::handleException);
            return new RaptorPageSource(recordReader, columnAdaptations, dataSource, memoryUsage);
        }
        catch (IOException | RuntimeException e) {
            throw new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed to create page source for shard " + shardUuid, (Throwable)e);
        }
        finally {
            RaptorStorageManager.closeQuietly((Closeable)dataSource);
        }
    }

    private static RaptorPageSource.ColumnAdaptation specialColumnAdaptation(long columnId, UUID shardUuid, OptionalInt bucketNumber) {
        if (RaptorColumnHandle.isShardRowIdColumn(columnId)) {
            return RaptorPageSource.ColumnAdaptation.rowIdColumn();
        }
        if (RaptorColumnHandle.isShardUuidColumn(columnId)) {
            return RaptorPageSource.ColumnAdaptation.shardUuidColumn(shardUuid);
        }
        if (RaptorColumnHandle.isBucketNumberColumn(columnId)) {
            return RaptorPageSource.ColumnAdaptation.bucketNumberColumn(bucketNumber);
        }
        if (RaptorColumnHandle.isMergeRowIdColumn(columnId)) {
            return RaptorPageSource.ColumnAdaptation.mergeRowIdColumn(bucketNumber, shardUuid);
        }
        throw new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Invalid column ID: " + columnId);
    }

    @Override
    public StoragePageSink createStoragePageSink(long transactionId, OptionalInt bucketNumber, List<Long> columnIds, List<Type> columnTypes, boolean checkSpace) {
        if (checkSpace && this.storageService.getAvailableBytes() < this.minAvailableSpace.toBytes()) {
            throw new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_LOCAL_DISK_FULL, "Local disk is full on node " + this.nodeId);
        }
        return new RaptorStoragePageSink(transactionId, columnIds, columnTypes, bucketNumber);
    }

    @Override
    public ShardRewriter createShardRewriter(long transactionId, OptionalInt bucketNumber, UUID shardUuid) {
        return rowsToDelete -> {
            if (rowsToDelete.isEmpty()) {
                return CompletableFuture.completedFuture(ImmutableList.of());
            }
            return CompletableFuture.supplyAsync(() -> this.rewriteShard(transactionId, bucketNumber, shardUuid, rowsToDelete), this.deletionExecutor);
        };
    }

    private void writeShard(UUID shardUuid) {
        if (this.backupStore.isPresent() && !this.backupStore.get().shardExists(shardUuid)) {
            throw new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Backup does not exist after write");
        }
        File stagingFile = this.storageService.getStagingFile(shardUuid);
        File storageFile = this.storageService.getStorageFile(shardUuid);
        this.storageService.createParents(storageFile);
        try {
            Files.move(stagingFile.toPath(), storageFile.toPath(), StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed to move shard file", (Throwable)e);
        }
    }

    @VisibleForTesting
    OrcDataSource openShard(UUID shardUuid, OrcReaderOptions orcReaderOptions) {
        File file = this.storageService.getStorageFile(shardUuid).getAbsoluteFile();
        if (!file.exists() && this.backupStore.isPresent()) {
            try {
                Future<Void> future = this.recoveryManager.recoverShard(shardUuid);
                future.get(this.recoveryTimeout.toMillis(), TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(e);
            }
            catch (ExecutionException e) {
                if (e.getCause() != null) {
                    Throwables.throwIfInstanceOf((Throwable)e.getCause(), TrinoException.class);
                }
                throw new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_RECOVERY_ERROR, "Error recovering shard " + shardUuid, e.getCause());
            }
            catch (TimeoutException e) {
                throw new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_RECOVERY_TIMEOUT, "Shard is being recovered from backup. Please retry in a few minutes: " + shardUuid);
            }
        }
        try {
            return RaptorStorageManager.fileOrcDataSource(orcReaderOptions, file);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed to open shard file: " + file, (Throwable)e);
        }
    }

    private static FileOrcDataSource fileOrcDataSource(OrcReaderOptions orcReaderOptions, File file) throws FileNotFoundException {
        return new FileOrcDataSource(file, orcReaderOptions);
    }

    private ShardInfo createShardInfo(UUID shardUuid, OptionalInt bucketNumber, File file, Set<String> nodes, long rowCount, long uncompressedSize) {
        return new ShardInfo(shardUuid, bucketNumber, nodes, this.computeShardStats(file), rowCount, file.length(), uncompressedSize, RaptorStorageManager.xxhash64(file));
    }

    private List<ColumnStats> computeShardStats(File file) {
        ImmutableList immutableList;
        block9: {
            FileOrcDataSource dataSource = RaptorStorageManager.fileOrcDataSource(this.orcReaderOptions, file);
            try {
                OrcReader reader = (OrcReader)OrcReader.createOrcReader((OrcDataSource)dataSource, (OrcReaderOptions)this.orcReaderOptions).orElseThrow(() -> new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Data file is empty: " + file));
                ImmutableList.Builder list = ImmutableList.builder();
                for (ColumnInfo info : RaptorStorageManager.getColumnInfo(this.typeManager, reader)) {
                    ShardStats.computeColumnStats(reader, info.getColumnId(), info.getType(), this.typeManager).ifPresent(arg_0 -> ((ImmutableList.Builder)list).add(arg_0));
                }
                immutableList = list.build();
                if (dataSource == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (dataSource != null) {
                        try {
                            dataSource.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed to read file: " + file, (Throwable)e);
                }
            }
            dataSource.close();
        }
        return immutableList;
    }

    @VisibleForTesting
    Collection<Slice> rewriteShard(long transactionId, OptionalInt bucketNumber, UUID shardUuid, BitSet rowsToDelete) {
        File output;
        if (rowsToDelete.isEmpty()) {
            return ImmutableList.of();
        }
        UUID newShardUuid = UUID.randomUUID();
        File input = this.storageService.getStorageFile(shardUuid);
        OrcFileRewriter.OrcFileInfo info = RaptorStorageManager.rewriteFile(this.typeManager, input, output = this.storageService.getStagingFile(newShardUuid), rowsToDelete);
        long rowCount = info.getRowCount();
        if (rowCount == 0L) {
            return RaptorStorageManager.shardDelta(shardUuid, Optional.empty());
        }
        this.shardRecorder.recordCreatedShard(transactionId, newShardUuid);
        MoreFutures.getFutureValue(this.backupManager.submit(newShardUuid, output));
        ImmutableSet nodes = ImmutableSet.of((Object)this.nodeId);
        long uncompressedSize = info.getUncompressedSize();
        ShardInfo shard = this.createShardInfo(newShardUuid, bucketNumber, output, (Set<String>)nodes, rowCount, uncompressedSize);
        this.writeShard(newShardUuid);
        return RaptorStorageManager.shardDelta(shardUuid, Optional.of(shard));
    }

    private static Collection<Slice> shardDelta(UUID oldShardUuid, Optional<ShardInfo> shardInfo) {
        List newShards = (List)shardInfo.map(ImmutableList::of).orElse(ImmutableList.of());
        ShardDelta delta = new ShardDelta((List<UUID>)ImmutableList.of((Object)oldShardUuid), newShards);
        return ImmutableList.of((Object)Slices.wrappedBuffer((byte[])SHARD_DELTA_CODEC.toJsonBytes((Object)delta)));
    }

    private static OrcFileRewriter.OrcFileInfo rewriteFile(TypeManager typeManager, File input, File output, BitSet rowsToDelete) {
        try {
            return OrcFileRewriter.rewrite(typeManager, input, output, rowsToDelete);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed to rewrite shard file: " + input, (Throwable)e);
        }
    }

    static List<ColumnInfo> getColumnInfo(TypeManager typeManager, OrcReader reader) {
        return RaptorStorageManager.getColumnInfoFromOrcUserMetadata(typeManager, RaptorStorageManager.getOrcFileMetadata(reader));
    }

    static long xxhash64(File file) {
        long l;
        FileInputStream in = new FileInputStream(file);
        try {
            l = XxHash64.hash((InputStream)in);
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((InputStream)in).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed to read file: " + file, (Throwable)e);
            }
        }
        ((InputStream)in).close();
        return l;
    }

    private static OrcFileMetadata getOrcFileMetadata(OrcReader reader) {
        return Optional.ofNullable((Slice)reader.getFooter().getUserMetadata().get("metadata")).map(slice -> (OrcFileMetadata)METADATA_CODEC.fromJson(slice.getBytes())).orElseThrow(() -> new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "No Raptor metadata in file"));
    }

    private static List<ColumnInfo> getColumnInfoFromOrcUserMetadata(TypeManager typeManager, OrcFileMetadata orcFileMetadata) {
        return orcFileMetadata.getColumnTypes().entrySet().stream().sorted(Map.Entry.comparingByKey()).map(entry -> new ColumnInfo((Long)entry.getKey(), typeManager.getType((TypeId)entry.getValue()))).collect(Collectors.toList());
    }

    static Type toOrcFileType(Type raptorType, TypeManager typeManager) {
        if (raptorType == BooleanType.BOOLEAN || raptorType == TinyintType.TINYINT || raptorType == SmallintType.SMALLINT || raptorType == IntegerType.INTEGER || raptorType == BigintType.BIGINT || raptorType == RealType.REAL || raptorType == DoubleType.DOUBLE || raptorType == DateType.DATE || raptorType instanceof DecimalType || raptorType instanceof VarcharType || raptorType instanceof VarbinaryType) {
            return raptorType;
        }
        if (raptorType.equals(TimestampType.TIMESTAMP_MILLIS)) {
            return BigintType.BIGINT;
        }
        if (raptorType instanceof ArrayType) {
            Type elementType = RaptorStorageManager.toOrcFileType(((ArrayType)raptorType).getElementType(), typeManager);
            return new ArrayType(elementType);
        }
        if (raptorType instanceof MapType) {
            TypeSignature keyType = RaptorStorageManager.toOrcFileType(((MapType)raptorType).getKeyType(), typeManager).getTypeSignature();
            TypeSignature valueType = RaptorStorageManager.toOrcFileType(((MapType)raptorType).getValueType(), typeManager).getTypeSignature();
            return typeManager.getParameterizedType("map", (List)ImmutableList.of((Object)TypeSignatureParameter.typeParameter((TypeSignature)keyType), (Object)TypeSignatureParameter.typeParameter((TypeSignature)valueType)));
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Unsupported type: " + raptorType);
    }

    private static OrcPredicate getPredicate(TupleDomain<RaptorColumnHandle> effectivePredicate, Map<Long, OrcColumn> indexMap) {
        TupleDomainOrcPredicate.TupleDomainOrcPredicateBuilder predicateBuilder = TupleDomainOrcPredicate.builder();
        ((Map)effectivePredicate.getDomains().get()).forEach((columnHandle, value) -> {
            OrcColumn fileColumn = (OrcColumn)indexMap.get(columnHandle.getColumnId());
            if (fileColumn != null) {
                predicateBuilder.addColumn(fileColumn.getColumnId(), value);
            }
        });
        return predicateBuilder.build();
    }

    private static Map<Long, OrcColumn> columnIdIndex(List<OrcColumn> columns) {
        return Maps.uniqueIndex(columns, column -> Long.valueOf(column.getColumnName()));
    }

    private static void closeQuietly(Closeable closeable) {
        try {
            closeable.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private class RaptorStoragePageSink
    implements StoragePageSink {
        private final long transactionId;
        private final List<Long> columnIds;
        private final List<Type> columnTypes;
        private final OptionalInt bucketNumber;
        private final List<File> stagingFiles = new ArrayList<File>();
        private final List<ShardInfo> shards = new ArrayList<ShardInfo>();
        private final List<CompletableFuture<?>> futures = new ArrayList();
        private boolean committed;
        private OrcFileWriter writer;
        private UUID shardUuid;

        public RaptorStoragePageSink(long transactionId, List<Long> columnIds, List<Type> columnTypes, OptionalInt bucketNumber) {
            this.transactionId = transactionId;
            this.columnIds = ImmutableList.copyOf((Collection)Objects.requireNonNull(columnIds, "columnIds is null"));
            this.columnTypes = ImmutableList.copyOf((Collection)Objects.requireNonNull(columnTypes, "columnTypes is null"));
            this.bucketNumber = Objects.requireNonNull(bucketNumber, "bucketNumber is null");
        }

        @Override
        public void appendPages(List<Page> pages) {
            this.createWriterIfNecessary();
            this.writer.appendPages(pages);
        }

        @Override
        public void appendPages(List<Page> inputPages, int[] pageIndexes, int[] positionIndexes) {
            this.createWriterIfNecessary();
            this.writer.appendPages(inputPages, pageIndexes, positionIndexes);
        }

        @Override
        public boolean isFull() {
            if (this.writer == null) {
                return false;
            }
            return this.writer.getRowCount() >= RaptorStorageManager.this.maxShardRows || this.writer.getUncompressedSize() >= RaptorStorageManager.this.maxShardSize.toBytes();
        }

        @Override
        public void flush() {
            if (this.writer != null) {
                this.writer.close();
                RaptorStorageManager.this.shardRecorder.recordCreatedShard(this.transactionId, this.shardUuid);
                File stagingFile = RaptorStorageManager.this.storageService.getStagingFile(this.shardUuid);
                this.futures.add(RaptorStorageManager.this.backupManager.submit(this.shardUuid, stagingFile));
                ImmutableSet nodes = ImmutableSet.of((Object)RaptorStorageManager.this.nodeId);
                long rowCount = this.writer.getRowCount();
                long uncompressedSize = this.writer.getUncompressedSize();
                this.shards.add(RaptorStorageManager.this.createShardInfo(this.shardUuid, this.bucketNumber, stagingFile, (Set<String>)nodes, rowCount, uncompressedSize));
                this.writer = null;
                this.shardUuid = null;
            }
        }

        @Override
        public CompletableFuture<List<ShardInfo>> commit() {
            Preconditions.checkState((!this.committed ? 1 : 0) != 0, (Object)"already committed");
            this.committed = true;
            this.flush();
            return MoreFutures.allAsList(this.futures).thenApplyAsync(ignored -> {
                for (ShardInfo shard : this.shards) {
                    RaptorStorageManager.this.writeShard(shard.getShardUuid());
                }
                return ImmutableList.copyOf(this.shards);
            }, (Executor)RaptorStorageManager.this.commitExecutor);
        }

        @Override
        public void rollback() {
            try {
                if (this.writer != null) {
                    this.writer.close();
                    this.writer = null;
                }
            }
            finally {
                for (File file : this.stagingFiles) {
                    file.delete();
                }
                this.futures.forEach(future -> future.cancel(true));
                RaptorStorageManager.this.backupStore.ifPresent(backupStore -> {
                    for (ShardInfo shard : this.shards) {
                        backupStore.deleteShard(shard.getShardUuid());
                    }
                });
            }
        }

        private void createWriterIfNecessary() {
            if (this.writer == null) {
                this.shardUuid = UUID.randomUUID();
                File stagingFile = RaptorStorageManager.this.storageService.getStagingFile(this.shardUuid);
                RaptorStorageManager.this.storageService.createParents(stagingFile);
                this.stagingFiles.add(stagingFile);
                this.writer = new OrcFileWriter(RaptorStorageManager.this.typeManager, this.columnIds, this.columnTypes, stagingFile);
            }
        }
    }
}

