/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.Verify;
import com.google.common.base.VerifyException;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.SettableFuture;
import com.google.errorprone.annotations.FormatMethod;
import io.airlift.concurrent.MoreFutures;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.execution.ScheduledSplit;
import io.trino.execution.SplitAssignment;
import io.trino.metadata.Split;
import io.trino.operator.DriverContext;
import io.trino.operator.OperationTimer;
import io.trino.operator.Operator;
import io.trino.operator.OperatorContext;
import io.trino.operator.SourceOperator;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.sql.planner.plan.PlanNodeId;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import javax.annotation.concurrent.GuardedBy;

public class Driver
implements Closeable {
    private static final Logger log = Logger.get(Driver.class);
    private static final Duration UNLIMITED_DURATION = new Duration(9.223372036854776E18, TimeUnit.NANOSECONDS);
    private final DriverContext driverContext;
    private final List<Operator> activeOperators;
    private final List<Operator> allOperators;
    private final Optional<SourceOperator> sourceOperator;
    private final AtomicReference<SplitAssignment> pendingSplitAssignmentUpdates = new AtomicReference();
    private final Map<Operator, ListenableFuture<Void>> revokingOperators = new HashMap<Operator, ListenableFuture<Void>>();
    private final AtomicReference<State> state = new AtomicReference<State>(State.ALIVE);
    private final DriverLock exclusiveLock = new DriverLock();
    @GuardedBy(value="exclusiveLock")
    private SplitAssignment currentSplitAssignment;
    private final AtomicReference<SettableFuture<Void>> driverBlockedFuture = new AtomicReference();

    public static Driver createDriver(DriverContext driverContext, List<Operator> operators) {
        Objects.requireNonNull(driverContext, "driverContext is null");
        Objects.requireNonNull(operators, "operators is null");
        Driver driver = new Driver(driverContext, operators);
        driver.initialize();
        return driver;
    }

    @VisibleForTesting
    public static Driver createDriver(DriverContext driverContext, Operator firstOperator, Operator ... otherOperators) {
        Objects.requireNonNull(driverContext, "driverContext is null");
        Objects.requireNonNull(firstOperator, "firstOperator is null");
        Objects.requireNonNull(otherOperators, "otherOperators is null");
        ImmutableList operators = ImmutableList.builder().add((Object)firstOperator).add((Object[])otherOperators).build();
        return Driver.createDriver(driverContext, (List<Operator>)operators);
    }

    private Driver(DriverContext driverContext, List<Operator> operators) {
        this.driverContext = Objects.requireNonNull(driverContext, "driverContext is null");
        this.allOperators = ImmutableList.copyOf((Collection)Objects.requireNonNull(operators, "operators is null"));
        Preconditions.checkArgument((this.allOperators.size() > 1 ? 1 : 0) != 0, (Object)"At least two operators are required");
        this.activeOperators = new ArrayList<Operator>(operators);
        Preconditions.checkArgument((!operators.isEmpty() ? 1 : 0) != 0, (Object)"There must be at least one operator");
        Optional<SourceOperator> sourceOperator = Optional.empty();
        for (Operator operator2 : operators) {
            if (!(operator2 instanceof SourceOperator)) continue;
            Preconditions.checkArgument((boolean)sourceOperator.isEmpty(), (Object)"There must be at most one SourceOperator");
            sourceOperator = Optional.of((SourceOperator)operator2);
        }
        this.sourceOperator = sourceOperator;
        this.currentSplitAssignment = sourceOperator.map(operator -> new SplitAssignment(operator.getSourceId(), (Set<ScheduledSplit>)ImmutableSet.of(), false)).orElse(null);
        SettableFuture future = SettableFuture.create();
        future.set(null);
        this.driverBlockedFuture.set((SettableFuture<Void>)future);
    }

    private void initialize() {
        this.activeOperators.stream().map(Operator::getOperatorContext).forEach(operatorContext -> operatorContext.setMemoryRevocationRequestListener(() -> this.driverBlockedFuture.get().set(null)));
    }

    public DriverContext getDriverContext() {
        return this.driverContext;
    }

    public Optional<PlanNodeId> getSourceId() {
        return this.sourceOperator.map(SourceOperator::getSourceId);
    }

    @Override
    public void close() {
        if (!this.state.compareAndSet(State.ALIVE, State.NEED_DESTRUCTION)) {
            return;
        }
        this.driverContext.getYieldSignal().yieldImmediatelyForTermination();
        this.exclusiveLock.interruptCurrentOwner();
        this.tryWithLockUninterruptibly(() -> Boolean.TRUE);
    }

    public boolean isFinished() {
        this.checkLockNotHeld("Cannot check finished status while holding the driver lock");
        Optional<Boolean> result = this.tryWithLockUninterruptibly(this::isFinishedInternal);
        return result.orElseGet(() -> this.state.get() != State.ALIVE || this.driverContext.isDone());
    }

    @GuardedBy(value="exclusiveLock")
    private boolean isFinishedInternal() {
        boolean finished;
        this.checkLockHeld("Lock must be held to call isFinishedInternal");
        boolean bl = finished = this.state.get() != State.ALIVE || this.driverContext.isDone() || this.activeOperators.isEmpty() || this.activeOperators.get(this.activeOperators.size() - 1).isFinished();
        if (finished) {
            this.state.compareAndSet(State.ALIVE, State.NEED_DESTRUCTION);
        }
        return finished;
    }

    public void updateSplitAssignment(SplitAssignment splitAssignment) {
        this.checkLockNotHeld("Cannot update assignments while holding the driver lock");
        Preconditions.checkArgument((this.sourceOperator.isPresent() && this.sourceOperator.get().getSourceId().equals(splitAssignment.getPlanNodeId()) ? 1 : 0) != 0, (Object)"splitAssignment is for a plan node that is different from this Driver's source node");
        this.pendingSplitAssignmentUpdates.updateAndGet(current -> current == null ? splitAssignment : current.update(splitAssignment));
        this.tryWithLockUninterruptibly(() -> Boolean.TRUE);
    }

    @GuardedBy(value="exclusiveLock")
    private void processNewSources() {
        this.checkLockHeld("Lock must be held to call processNewSources");
        if (this.state.get() != State.ALIVE) {
            return;
        }
        SplitAssignment splitAssignment = this.pendingSplitAssignmentUpdates.getAndSet(null);
        if (splitAssignment == null) {
            return;
        }
        SplitAssignment newAssignment = this.currentSplitAssignment.update(splitAssignment);
        if (newAssignment == this.currentSplitAssignment) {
            return;
        }
        Sets.SetView newSplits = Sets.difference(newAssignment.getSplits(), this.currentSplitAssignment.getSplits());
        SourceOperator sourceOperator = this.sourceOperator.orElseThrow(VerifyException::new);
        for (ScheduledSplit newSplit : newSplits) {
            Split split = newSplit.getSplit();
            sourceOperator.addSplit(split);
        }
        if (newAssignment.isNoMoreSplits()) {
            sourceOperator.noMoreSplits();
        }
        this.currentSplitAssignment = newAssignment;
    }

    public ListenableFuture<Void> processForDuration(Duration duration) {
        return this.process(duration, Integer.MAX_VALUE);
    }

    public ListenableFuture<Void> processForNumberOfIterations(int maxIterations) {
        return this.process(UNLIMITED_DURATION, maxIterations);
    }

    public ListenableFuture<Void> processUntilBlocked() {
        return this.process(UNLIMITED_DURATION, Integer.MAX_VALUE);
    }

    @VisibleForTesting
    public ListenableFuture<Void> process(Duration maxRuntime, int maxIterations) {
        this.checkLockNotHeld("Cannot process for a duration while holding the driver lock");
        Objects.requireNonNull(maxRuntime, "maxRuntime is null");
        Preconditions.checkArgument((maxIterations > 0 ? 1 : 0) != 0, (Object)"maxIterations must be greater than zero");
        SettableFuture<Void> blockedFuture = this.driverBlockedFuture.get();
        if (!blockedFuture.isDone()) {
            return blockedFuture;
        }
        long maxRuntimeInNanos = maxRuntime.roundTo(TimeUnit.NANOSECONDS);
        Optional<ListenableFuture> result = this.tryWithLock(100L, TimeUnit.MILLISECONDS, true, () -> {
            OperationTimer operationTimer = this.createTimer();
            this.driverContext.startProcessTimer();
            this.driverContext.getYieldSignal().setWithDelay(maxRuntimeInNanos, this.driverContext.getYieldExecutor());
            try {
                long start = System.nanoTime();
                int iterations = 0;
                while (!this.isFinishedInternal()) {
                    ListenableFuture<Void> future = this.processInternal(operationTimer);
                    ++iterations;
                    if (!future.isDone()) {
                        ListenableFuture<Void> listenableFuture = this.updateDriverBlockedFuture(future);
                        return listenableFuture;
                    }
                    if (System.nanoTime() - start < maxRuntimeInNanos) {
                        if (iterations < maxIterations) continue;
                    }
                    break;
                }
            }
            catch (Throwable t) {
                List<StackTraceElement> interrupterStack = this.exclusiveLock.getInterrupterStack();
                if (interrupterStack == null) {
                    this.driverContext.failed(t);
                    throw t;
                }
                Exception exception = new Exception("Interrupted By");
                exception.setStackTrace((StackTraceElement[])interrupterStack.toArray(StackTraceElement[]::new));
                TrinoException newException = new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Driver was interrupted", (Throwable)exception);
                newException.addSuppressed(t);
                this.driverContext.failed(newException);
                throw newException;
            }
            finally {
                this.driverContext.getYieldSignal().reset();
                this.driverContext.recordProcessed(operationTimer);
            }
            return Operator.NOT_BLOCKED;
        });
        return result.orElse(Operator.NOT_BLOCKED);
    }

    private OperationTimer createTimer() {
        return new OperationTimer(this.driverContext.isCpuTimerEnabled(), this.driverContext.isCpuTimerEnabled() && this.driverContext.isPerOperatorCpuTimerEnabled());
    }

    private ListenableFuture<Void> updateDriverBlockedFuture(ListenableFuture<Void> sourceBlockedFuture) {
        SettableFuture newDriverBlockedFuture = SettableFuture.create();
        this.driverBlockedFuture.set((SettableFuture<Void>)newDriverBlockedFuture);
        sourceBlockedFuture.addListener(() -> newDriverBlockedFuture.set(null), MoreExecutors.directExecutor());
        boolean memoryRevokingRequested = this.activeOperators.stream().filter(operator -> !this.revokingOperators.containsKey(operator)).map(Operator::getOperatorContext).anyMatch(OperatorContext::isMemoryRevokingRequested);
        if (memoryRevokingRequested) {
            newDriverBlockedFuture.set(null);
        }
        return newDriverBlockedFuture;
    }

    @GuardedBy(value="exclusiveLock")
    private ListenableFuture<Void> processInternal(OperationTimer operationTimer) {
        this.checkLockHeld("Lock must be held to call processInternal");
        this.handleMemoryRevoke();
        this.processNewSources();
        if (!this.activeOperators.isEmpty() && this.activeOperators.size() != this.allOperators.size()) {
            Operator rootOperator = this.activeOperators.get(0);
            rootOperator.finish();
            rootOperator.getOperatorContext().recordFinish(operationTimer);
        }
        boolean movedPage = false;
        for (int i = 0; i < this.activeOperators.size() - 1 && !this.driverContext.isDone(); ++i) {
            Operator current = this.activeOperators.get(i);
            Operator next = this.activeOperators.get(i + 1);
            if (this.getBlockedFuture(current).isPresent()) continue;
            if (!current.isFinished() && this.getBlockedFuture(next).isEmpty() && next.needsInput()) {
                Page page = current.getOutput();
                current.getOperatorContext().recordGetOutput(operationTimer, page);
                if (page != null && page.getPositionCount() != 0) {
                    next.addInput(page);
                    next.getOperatorContext().recordAddInput(operationTimer, page);
                    movedPage = true;
                }
                if (current instanceof SourceOperator) {
                    movedPage = true;
                }
            }
            if (!current.isFinished()) continue;
            next.finish();
            next.getOperatorContext().recordFinish(operationTimer);
        }
        for (int index = this.activeOperators.size() - 1; index >= 0; --index) {
            if (!this.activeOperators.get(index).isFinished()) continue;
            List<Operator> finishedOperators = this.activeOperators.subList(0, index + 1);
            Iterator<Operator> throwable = this.closeAndDestroyOperators(finishedOperators);
            finishedOperators.clear();
            if (throwable != null) {
                Throwables.throwIfUnchecked(throwable);
                throw new RuntimeException((Throwable)((Object)throwable));
            }
            if (this.activeOperators.isEmpty()) break;
            Operator newRootOperator = this.activeOperators.get(0);
            newRootOperator.finish();
            newRootOperator.getOperatorContext().recordFinish(operationTimer);
            break;
        }
        if (!movedPage) {
            ArrayList<Operator> blockedOperators = new ArrayList<Operator>();
            ArrayList<ListenableFuture<Void>> blockedFutures = new ArrayList<ListenableFuture<Void>>();
            for (Operator operator : this.activeOperators) {
                Optional<ListenableFuture<Void>> blocked = this.getBlockedFuture(operator);
                if (!blocked.isPresent()) continue;
                blockedOperators.add(operator);
                blockedFutures.add(blocked.get());
            }
            if (!blockedFutures.isEmpty()) {
                for (Operator operator : this.activeOperators) {
                    operator.getOperatorContext().getFinishedFuture().ifPresent(blockedFutures::add);
                }
                ListenableFuture<Void> blocked = Driver.firstFinishedFuture(blockedFutures);
                this.driverContext.recordBlocked(blocked);
                for (Operator operator : blockedOperators) {
                    operator.getOperatorContext().recordBlocked(blocked);
                }
                return blocked;
            }
        }
        return Operator.NOT_BLOCKED;
    }

    @GuardedBy(value="exclusiveLock")
    private void handleMemoryRevoke() {
        for (int i = 0; i < this.activeOperators.size() && !this.driverContext.isDone(); ++i) {
            Operator operator = this.activeOperators.get(i);
            if (this.revokingOperators.containsKey(operator)) {
                this.checkOperatorFinishedRevoking(operator);
                continue;
            }
            if (!operator.getOperatorContext().isMemoryRevokingRequested()) continue;
            ListenableFuture<Void> future = operator.startMemoryRevoke();
            this.revokingOperators.put(operator, future);
            this.checkOperatorFinishedRevoking(operator);
        }
    }

    @GuardedBy(value="exclusiveLock")
    private void checkOperatorFinishedRevoking(Operator operator) {
        ListenableFuture<Void> future = this.revokingOperators.get(operator);
        if (future.isDone()) {
            MoreFutures.getFutureValue(future);
            this.revokingOperators.remove(operator);
            operator.finishMemoryRevoke();
            operator.getOperatorContext().resetMemoryRevokingRequested();
        }
    }

    @GuardedBy(value="exclusiveLock")
    private void destroyIfNecessary() {
        this.checkLockHeld("Lock must be held to call destroyIfNecessary");
        if (!this.state.compareAndSet(State.NEED_DESTRUCTION, State.DESTROYED)) {
            return;
        }
        Throwable inFlightException = null;
        try {
            inFlightException = this.closeAndDestroyOperators(this.activeOperators);
            if (this.driverContext.getMemoryUsage() > 0L) {
                log.error("Driver still has memory reserved after freeing all operator memory.");
            }
            if (this.driverContext.getRevocableMemoryUsage() > 0L) {
                log.error("Driver still has revocable memory reserved after freeing all operator memory. Freeing it.");
            }
            this.driverContext.finished();
        }
        catch (Throwable t) {
            inFlightException = Driver.addSuppressedException(inFlightException, t, "Error destroying driver for task %s", this.driverContext.getTaskId());
        }
        if (inFlightException != null) {
            Throwables.throwIfUnchecked((Throwable)inFlightException);
            throw new RuntimeException(inFlightException);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Throwable closeAndDestroyOperators(List<Operator> operators) {
        boolean wasInterrupted = Thread.interrupted();
        Throwable inFlightException = null;
        try {
            for (Operator operator : operators) {
                try {
                    operator.close();
                }
                catch (InterruptedException t) {
                    wasInterrupted = true;
                }
                catch (Throwable t) {
                    inFlightException = Driver.addSuppressedException(inFlightException, t, "Error closing operator %s for task %s", operator.getOperatorContext().getOperatorId(), this.driverContext.getTaskId());
                }
                try {
                    operator.getOperatorContext().destroy();
                }
                catch (Throwable t) {
                    inFlightException = Driver.addSuppressedException(inFlightException, t, "Error freeing all allocated memory for operator %s for task %s", operator.getOperatorContext().getOperatorId(), this.driverContext.getTaskId());
                }
            }
        }
        finally {
            if (wasInterrupted) {
                Thread.currentThread().interrupt();
            }
        }
        return inFlightException;
    }

    private Optional<ListenableFuture<Void>> getBlockedFuture(Operator operator) {
        ListenableFuture<Void> blocked = this.revokingOperators.get(operator);
        if (blocked != null) {
            return Optional.of(blocked);
        }
        blocked = operator.isBlocked();
        if (!blocked.isDone()) {
            return Optional.of(blocked);
        }
        blocked = operator.getOperatorContext().isWaitingForMemory();
        if (!blocked.isDone()) {
            return Optional.of(blocked);
        }
        blocked = operator.getOperatorContext().isWaitingForRevocableMemory();
        if (!blocked.isDone()) {
            return Optional.of(blocked);
        }
        return Optional.empty();
    }

    @FormatMethod
    private static Throwable addSuppressedException(Throwable inFlightException, Throwable newException, String message, Object ... args) {
        if (newException instanceof Error) {
            if (inFlightException == null) {
                inFlightException = newException;
            } else if (inFlightException != newException) {
                inFlightException.addSuppressed(newException);
            }
        } else {
            log.error(newException, message, args);
        }
        return inFlightException;
    }

    private synchronized void checkLockNotHeld(String message) {
        Preconditions.checkState((!this.exclusiveLock.isHeldByCurrentThread() ? 1 : 0) != 0, (Object)message);
    }

    @GuardedBy(value="exclusiveLock")
    private synchronized void checkLockHeld(String message) {
        Preconditions.checkState((boolean)this.exclusiveLock.isHeldByCurrentThread(), (Object)message);
    }

    private static ListenableFuture<Void> firstFinishedFuture(List<ListenableFuture<Void>> futures) {
        if (futures.size() == 1) {
            return futures.get(0);
        }
        SettableFuture result = SettableFuture.create();
        for (ListenableFuture<Void> future : futures) {
            future.addListener(() -> result.set(null), MoreExecutors.directExecutor());
        }
        return result;
    }

    private <T> Optional<T> tryWithLockUninterruptibly(Supplier<T> task) {
        return this.tryWithLock(0L, TimeUnit.MILLISECONDS, false, task);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T> Optional<T> tryWithLock(long timeout, TimeUnit unit, boolean interruptOnClose, Supplier<T> task) {
        this.checkLockNotHeld("Lock cannot be reacquired");
        boolean acquired = false;
        try {
            acquired = this.exclusiveLock.tryLock(timeout, unit, interruptOnClose);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        if (!acquired) {
            return Optional.empty();
        }
        Object result = null;
        Throwable failure = null;
        try {
            result = task.get();
            this.processNewSources();
            this.destroyIfNecessary();
        }
        catch (Throwable t) {
            failure = t;
        }
        finally {
            this.exclusiveLock.unlock();
        }
        while ((this.pendingSplitAssignmentUpdates.get() != null && this.state.get() == State.ALIVE || this.state.get() == State.NEED_DESTRUCTION) && this.exclusiveLock.tryLock(interruptOnClose)) {
            try {
                block20: {
                    try {
                        this.processNewSources();
                    }
                    catch (Throwable t) {
                        if (failure == null) {
                            failure = t;
                        }
                        if (failure == t) break block20;
                        failure.addSuppressed(t);
                    }
                }
                try {
                    this.destroyIfNecessary();
                }
                catch (Throwable t) {
                    if (failure == null) {
                        failure = t;
                        continue;
                    }
                    if (failure == t) continue;
                    failure.addSuppressed(t);
                }
            }
            finally {
                this.exclusiveLock.unlock();
            }
        }
        if (failure != null) {
            Throwables.throwIfUnchecked((Throwable)failure);
            throw new AssertionError((Object)failure);
        }
        Verify.verify((result != null ? 1 : 0) != 0, (String)"result is null", (Object[])new Object[0]);
        return Optional.of(result);
    }

    private static enum State {
        ALIVE,
        NEED_DESTRUCTION,
        DESTROYED;

    }

    private static class DriverLock {
        private final ReentrantLock lock = new ReentrantLock();
        @GuardedBy(value="this")
        private Thread currentOwner;
        @GuardedBy(value="this")
        private boolean currentOwnerInterruptionAllowed;
        @GuardedBy(value="this")
        private List<StackTraceElement> interrupterStack;

        private DriverLock() {
        }

        public boolean isHeldByCurrentThread() {
            return this.lock.isHeldByCurrentThread();
        }

        public boolean tryLock(boolean currentThreadInterruptionAllowed) {
            Preconditions.checkState((!this.lock.isHeldByCurrentThread() ? 1 : 0) != 0, (Object)"Lock is not reentrant");
            boolean acquired = this.lock.tryLock();
            if (acquired) {
                this.setOwner(currentThreadInterruptionAllowed);
            }
            return acquired;
        }

        public boolean tryLock(long timeout, TimeUnit unit, boolean currentThreadInterruptionAllowed) throws InterruptedException {
            Preconditions.checkState((!this.lock.isHeldByCurrentThread() ? 1 : 0) != 0, (Object)"Lock is not reentrant");
            boolean acquired = this.lock.tryLock(timeout, unit);
            if (acquired) {
                this.setOwner(currentThreadInterruptionAllowed);
            }
            return acquired;
        }

        private synchronized void setOwner(boolean interruptionAllowed) {
            Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread(), (Object)"Current thread does not hold lock");
            this.currentOwner = Thread.currentThread();
            this.currentOwnerInterruptionAllowed = interruptionAllowed;
        }

        public synchronized void unlock() {
            Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread(), (Object)"Current thread does not hold lock");
            this.currentOwner = null;
            this.currentOwnerInterruptionAllowed = false;
            this.lock.unlock();
        }

        public synchronized List<StackTraceElement> getInterrupterStack() {
            return this.interrupterStack;
        }

        public synchronized void interruptCurrentOwner() {
            if (!this.currentOwnerInterruptionAllowed) {
                return;
            }
            if (this.interrupterStack == null) {
                this.interrupterStack = ImmutableList.copyOf((Object[])Thread.currentThread().getStackTrace());
            }
            if (this.currentOwner != null) {
                this.currentOwner.interrupt();
            }
        }
    }
}

