/*
 * Decompiled with CFR 0.152.
 */
package io.trino.sql.planner.planprinter;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import io.airlift.units.DataSize;
import io.trino.cost.PlanNodeStatsAndCostSummary;
import io.trino.plugin.base.metrics.TDigestHistogram;
import io.trino.spi.metrics.Metric;
import io.trino.spi.metrics.Metrics;
import io.trino.sql.planner.planprinter.BasicOperatorStats;
import io.trino.sql.planner.planprinter.NodeRepresentation;
import io.trino.sql.planner.planprinter.PlanNodeStats;
import io.trino.sql.planner.planprinter.PlanRepresentation;
import io.trino.sql.planner.planprinter.Renderer;
import io.trino.sql.planner.planprinter.WindowOperatorStats;
import io.trino.sql.planner.planprinter.WindowPlanNodeStats;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.stream.Collectors;

public class TextRenderer
implements Renderer<String> {
    private final boolean verbose;
    private final int level;

    public TextRenderer(boolean verbose, int level) {
        this.verbose = verbose;
        this.level = level;
    }

    @Override
    public String render(PlanRepresentation plan) {
        StringBuilder output = new StringBuilder();
        NodeRepresentation root = plan.getRoot();
        boolean hasChildren = TextRenderer.hasChildren(root, plan);
        return this.writeTextOutput(output, plan, Indent.newInstance(this.level, hasChildren), root);
    }

    private String writeTextOutput(StringBuilder output, PlanRepresentation plan, Indent indent, NodeRepresentation node) {
        String stats;
        List<PlanNodeStatsAndCostSummary> estimates;
        output.append(indent.nodeIndent()).append(node.getName()).append(node.getDescriptor().entrySet().stream().filter(entry -> !((String)entry.getValue()).isEmpty() && !((String)entry.getValue()).equals("[]")).map(entry -> (String)entry.getKey() + " = " + (String)entry.getValue()).collect(Collectors.joining(", ", "[", "]"))).append("\n");
        String columns = node.getOutputs().stream().map(s -> s.getSymbol() + ":" + s.getType()).collect(Collectors.joining(", "));
        output.append(TextRenderer.indentMultilineString("Layout: [" + columns + "]\n", indent.detailIndent()));
        String reorderJoinStatsAndCost = this.printReorderJoinStatsAndCost(node);
        if (!reorderJoinStatsAndCost.isEmpty()) {
            output.append(TextRenderer.indentMultilineString(reorderJoinStatsAndCost, indent.detailIndent()));
        }
        if (!(estimates = node.getEstimates(plan.getTypes())).isEmpty()) {
            output.append(TextRenderer.indentMultilineString(this.printEstimates(estimates), indent.detailIndent()));
        }
        if (!(stats = this.printStats(plan, node)).isEmpty()) {
            output.append(TextRenderer.indentMultilineString(stats, indent.detailIndent()));
        }
        if (!node.getDetails().isEmpty()) {
            String details = TextRenderer.indentMultilineString(Joiner.on((String)"\n").join(node.getDetails()), indent.detailIndent());
            output.append(details);
            if (!details.endsWith("\n")) {
                output.append('\n');
            }
        }
        List children = node.getChildren().stream().map(plan::getNode).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
        Iterator iterator = children.iterator();
        while (iterator.hasNext()) {
            NodeRepresentation child = (NodeRepresentation)iterator.next();
            this.writeTextOutput(output, plan, indent.forChild(!iterator.hasNext(), TextRenderer.hasChildren(child, plan)), child);
        }
        return output.toString();
    }

    private String printStats(PlanRepresentation plan, NodeRepresentation node) {
        StringBuilder output = new StringBuilder();
        if (node.getStats().isEmpty() || !plan.getTotalCpuTime().isPresent() || !plan.getTotalScheduledTime().isPresent() || !plan.getTotalBlockedTime().isPresent()) {
            return "";
        }
        PlanNodeStats nodeStats = node.getStats().get();
        double scheduledTimeFraction = 100.0 * (double)nodeStats.getPlanNodeScheduledTime().toMillis() / (double)plan.getTotalScheduledTime().get().toMillis();
        double cpuTimeFraction = 100.0 * (double)nodeStats.getPlanNodeCpuTime().toMillis() / (double)plan.getTotalCpuTime().get().toMillis();
        double blockedTimeFraction = 100.0 * (double)nodeStats.getPlanNodeBlockedTime().toMillis() / (double)plan.getTotalBlockedTime().get().toMillis();
        output.append(String.format("CPU: %s (%s%%), Scheduled: %s (%s%%), Blocked: %s (%s%%)", nodeStats.getPlanNodeCpuTime().convertToMostSuccinctTimeUnit(), TextRenderer.formatDouble(cpuTimeFraction), nodeStats.getPlanNodeScheduledTime().convertToMostSuccinctTimeUnit(), TextRenderer.formatDouble(scheduledTimeFraction), nodeStats.getPlanNodeBlockedTime().convertToMostSuccinctTimeUnit(), TextRenderer.formatDouble(blockedTimeFraction)));
        output.append(String.format(", Output: %s (%s)", TextRenderer.formatPositions(nodeStats.getPlanNodeOutputPositions()), nodeStats.getPlanNodeOutputDataSize().toString()));
        if (nodeStats.getPlanNodeSpilledDataSize().toBytes() > 0L) {
            output.append(String.format(", Spilled: %s", nodeStats.getPlanNodeSpilledDataSize()));
        }
        output.append("\n");
        this.printMetrics(output, "connector metrics:", BasicOperatorStats::getConnectorMetrics, nodeStats);
        this.printMetrics(output, "metrics:", BasicOperatorStats::getMetrics, nodeStats);
        this.printDistributions(output, nodeStats);
        if (nodeStats instanceof WindowPlanNodeStats) {
            this.printWindowOperatorStats(output, ((WindowPlanNodeStats)nodeStats).getWindowOperatorStats());
        }
        return output.toString();
    }

    private void printMetrics(StringBuilder output, String label, Function<BasicOperatorStats, Metrics> metricsGetter, PlanNodeStats stats) {
        if (!this.verbose) {
            return;
        }
        Map<String, String> translatedOperatorTypes = TextRenderer.translateOperatorTypes(stats.getOperatorTypes());
        for (String operator : translatedOperatorTypes.keySet()) {
            String translatedOperatorType = translatedOperatorTypes.get(operator);
            Map metrics = metricsGetter.apply(stats.getOperatorStats().get(operator)).getMetrics();
            if ((metrics = (Map)metrics.entrySet().stream().filter(entry -> {
                Object patt7429$temp = entry.getValue();
                if (!(patt7429$temp instanceof TDigestHistogram)) {
                    return true;
                }
                TDigestHistogram histogram = (TDigestHistogram)patt7429$temp;
                return histogram.getMin() != 0.0 || histogram.getMax() != 0.0;
            }).collect(ImmutableMap.toImmutableMap(Map.Entry::getKey, Map.Entry::getValue))).isEmpty()) continue;
            output.append(translatedOperatorType + label).append("\n");
            TreeMap<String, Metric> sortedMap = new TreeMap<String, Metric>(metrics);
            sortedMap.forEach((name, metric) -> output.append(String.format("  '%s' = %s\n", name, metric)));
        }
    }

    private void printDistributions(StringBuilder output, PlanNodeStats stats) {
        Map<String, Double> inputAverages = stats.getOperatorInputPositionsAverages();
        Map<String, Double> inputStdDevs = stats.getOperatorInputPositionsStdDevs();
        Map<String, String> translatedOperatorTypes = TextRenderer.translateOperatorTypes(stats.getOperatorTypes());
        for (String operator : translatedOperatorTypes.keySet()) {
            String translatedOperatorType = translatedOperatorTypes.get(operator);
            double inputAverage = inputAverages.get(operator);
            output.append(translatedOperatorType);
            output.append(String.format(Locale.US, "Input avg.: %s rows, Input std.dev.: %s%%\n", TextRenderer.formatDouble(inputAverage), TextRenderer.formatDouble(100.0 * inputStdDevs.get(operator) / inputAverage)));
        }
    }

    private void printWindowOperatorStats(StringBuilder output, WindowOperatorStats stats) {
        if (!this.verbose) {
            return;
        }
        output.append(String.format("Active Drivers: [ %d / %d ]\n", stats.getActiveDrivers(), stats.getTotalDrivers()));
        output.append(String.format("Index size: std.dev.: %s bytes, %s rows\n", TextRenderer.formatDouble(stats.getIndexSizeStdDev()), TextRenderer.formatDouble(stats.getIndexPositionsStdDev())));
        output.append(String.format("Index count per driver: std.dev.: %s\n", TextRenderer.formatDouble(stats.getIndexCountPerDriverStdDev())));
        output.append(String.format("Rows per driver: std.dev.: %s\n", TextRenderer.formatDouble(stats.getRowsPerDriverStdDev())));
        output.append(String.format("Size of partition: std.dev.: %s\n", TextRenderer.formatDouble(stats.getPartitionRowsStdDev())));
    }

    private static Map<String, String> translateOperatorTypes(Set<String> operators) {
        if (operators.size() == 1) {
            return ImmutableMap.of((Object)((String)Iterables.getOnlyElement(operators)), (Object)"");
        }
        if (operators.contains("LookupJoinOperator") && operators.contains("HashBuilderOperator")) {
            return ImmutableMap.of((Object)"LookupJoinOperator", (Object)"Left (probe) ", (Object)"HashBuilderOperator", (Object)"Right (build) ");
        }
        return ImmutableMap.of();
    }

    private String printReorderJoinStatsAndCost(NodeRepresentation node) {
        if (this.verbose && node.getReorderJoinStatsAndCost().isPresent()) {
            return String.format("Reorder joins cost : %s\n", this.formatPlanNodeStatsAndCostSummary(node.getReorderJoinStatsAndCost().get()));
        }
        return "";
    }

    private String printEstimates(List<PlanNodeStatsAndCostSummary> estimates) {
        return estimates.stream().map(this::formatPlanNodeStatsAndCostSummary).collect(Collectors.joining("/", "Estimates: ", "\n"));
    }

    private String formatPlanNodeStatsAndCostSummary(PlanNodeStatsAndCostSummary stats) {
        Objects.requireNonNull(stats, "stats is null");
        return String.format("{rows: %s (%s), cpu: %s, memory: %s, network: %s}", TextRenderer.formatAsLong(stats.getOutputRowCount()), TextRenderer.formatAsDataSize(stats.getOutputSizeInBytes()), TextRenderer.formatAsCpuCost(stats.getCpuCost()), TextRenderer.formatAsDataSize(stats.getMemoryCost()), TextRenderer.formatAsDataSize(stats.getNetworkCost()));
    }

    private static boolean hasChildren(NodeRepresentation node, PlanRepresentation plan) {
        return node.getChildren().stream().map(plan::getNode).anyMatch(Optional::isPresent);
    }

    private static String formatAsLong(double value) {
        if (Double.isFinite(value)) {
            return String.format(Locale.US, "%d", Math.round(value));
        }
        return "?";
    }

    private static String formatAsCpuCost(double value) {
        return TextRenderer.formatAsDataSize(value).replaceAll("B$", "");
    }

    private static String formatAsDataSize(double value) {
        if (Double.isNaN(value)) {
            return "?";
        }
        if (value == Double.POSITIVE_INFINITY) {
            return "+\u221e";
        }
        if (value == Double.NEGATIVE_INFINITY) {
            return "-\u221e";
        }
        return DataSize.succinctBytes((long)Math.round(value)).toString();
    }

    static String formatDouble(double value) {
        if (Double.isFinite(value)) {
            return String.format(Locale.US, "%.2f", value);
        }
        return "?";
    }

    static String formatPositions(long positions) {
        String noun = positions == 1L ? "row" : "rows";
        return positions + " " + noun;
    }

    static String indentString(int indent) {
        return "    ".repeat(indent);
    }

    private static String indentMultilineString(String string, String indent) {
        return string.replaceAll("(?m)^", indent);
    }

    private static class Indent {
        private static final String VERTICAL_LINE = "\u2502";
        private static final String LAST_NODE = "\u2514\u2500";
        private static final String INTERMEDIATE_NODE = "\u251c\u2500";
        private final String firstLinePrefix;
        private final String nextLinesPrefix;
        private final boolean hasChildren;

        public static Indent newInstance(int level, boolean hasChildren) {
            String indent = TextRenderer.indentString(level);
            return new Indent(indent, indent, hasChildren);
        }

        private Indent(String firstLinePrefix, String nextLinesPrefix, boolean hasChildren) {
            this.firstLinePrefix = firstLinePrefix;
            this.nextLinesPrefix = nextLinesPrefix;
            this.hasChildren = hasChildren;
        }

        public Indent forChild(boolean last, boolean hasChildren) {
            String next;
            String first;
            if (last) {
                first = Indent.pad(LAST_NODE, 3);
                next = Indent.pad("", 3);
            } else {
                first = Indent.pad(INTERMEDIATE_NODE, 3);
                next = Indent.pad(VERTICAL_LINE, 3);
            }
            return new Indent(this.nextLinesPrefix + first, this.nextLinesPrefix + next, hasChildren);
        }

        public String nodeIndent() {
            return this.firstLinePrefix;
        }

        public String detailIndent() {
            String indent = "";
            if (this.hasChildren) {
                indent = VERTICAL_LINE;
            }
            return this.nextLinesPrefix + Indent.pad(indent, 4);
        }

        private static String pad(String text, int length) {
            Preconditions.checkArgument((text.length() <= length ? 1 : 0) != 0, (Object)"text is longer that length");
            return text + " ".repeat(length - text.length());
        }
    }
}

