/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.io.Closer;
import io.airlift.log.Logger;
import io.airlift.slice.SizeOf;
import io.airlift.units.DataSize;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoInputFile;
import io.trino.orc.OrcDataSink;
import io.trino.orc.OrcDataSource;
import io.trino.orc.OrcDataSourceId;
import io.trino.orc.OrcReaderOptions;
import io.trino.plugin.hive.FileFormatDataSourceStats;
import io.trino.plugin.hive.FileWriter;
import io.trino.plugin.hive.HiveErrorCode;
import io.trino.plugin.hive.orc.HdfsOrcDataSource;
import io.trino.plugin.hive.util.MergingPageIterator;
import io.trino.plugin.hive.util.SortBuffer;
import io.trino.plugin.hive.util.TempFileReader;
import io.trino.plugin.hive.util.TempFileWriter;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.PageSorter;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.SortOrder;
import io.trino.spi.type.Type;
import io.trino.spi.type.TypeOperators;
import java.io.Closeable;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.stream.IntStream;
import org.apache.hadoop.fs.Path;

public class SortingFileWriter
implements FileWriter {
    private static final Logger log = Logger.get(SortingFileWriter.class);
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(SortingFileWriter.class);
    private final TrinoFileSystem fileSystem;
    private final Path tempFilePrefix;
    private final int maxOpenTempFiles;
    private final List<Type> types;
    private final List<Integer> sortFields;
    private final List<SortOrder> sortOrders;
    private final FileWriter outputWriter;
    private final SortBuffer sortBuffer;
    private final TempFileSinkFactory tempFileSinkFactory;
    private final Queue<TempFile> tempFiles = new PriorityQueue<TempFile>(Comparator.comparing(TempFile::getSize));
    private final AtomicLong nextFileId = new AtomicLong();
    private final TypeOperators typeOperators;

    public SortingFileWriter(TrinoFileSystem fileSystem, Path tempFilePrefix, FileWriter outputWriter, DataSize maxMemory, int maxOpenTempFiles, List<Type> types, List<Integer> sortFields, List<SortOrder> sortOrders, PageSorter pageSorter, TypeOperators typeOperators, TempFileSinkFactory tempFileSinkFactory) {
        Preconditions.checkArgument((maxOpenTempFiles >= 2 ? 1 : 0) != 0, (Object)"maxOpenTempFiles must be at least two");
        this.fileSystem = Objects.requireNonNull(fileSystem, "fileSystem is null");
        this.tempFilePrefix = Objects.requireNonNull(tempFilePrefix, "tempFilePrefix is null");
        this.maxOpenTempFiles = maxOpenTempFiles;
        this.types = ImmutableList.copyOf((Collection)Objects.requireNonNull(types, "types is null"));
        this.sortFields = ImmutableList.copyOf((Collection)Objects.requireNonNull(sortFields, "sortFields is null"));
        this.sortOrders = ImmutableList.copyOf((Collection)Objects.requireNonNull(sortOrders, "sortOrders is null"));
        this.outputWriter = Objects.requireNonNull(outputWriter, "outputWriter is null");
        this.sortBuffer = new SortBuffer(maxMemory, types, sortFields, sortOrders, pageSorter);
        this.tempFileSinkFactory = tempFileSinkFactory;
        this.typeOperators = Objects.requireNonNull(typeOperators, "typeOperators is null");
    }

    @Override
    public long getWrittenBytes() {
        return this.outputWriter.getWrittenBytes();
    }

    @Override
    public long getMemoryUsage() {
        return (long)INSTANCE_SIZE + this.sortBuffer.getRetainedBytes();
    }

    @Override
    public void appendRows(Page page) {
        if (!this.sortBuffer.canAdd(page)) {
            this.flushToTempFile();
        }
        this.sortBuffer.add(page);
    }

    @Override
    public Closeable commit() {
        Closeable rollbackAction = SortingFileWriter.createRollbackAction(this.fileSystem, this.tempFiles);
        if (!this.sortBuffer.isEmpty()) {
            if (this.tempFiles.isEmpty()) {
                this.sortBuffer.flushTo(this.outputWriter::appendRows);
                this.outputWriter.commit();
                return rollbackAction;
            }
            this.flushToTempFile();
        }
        try {
            this.writeSorted();
            this.outputWriter.commit();
        }
        catch (UncheckedIOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_WRITER_CLOSE_ERROR, "Error committing write to Hive", (Throwable)e);
        }
        return rollbackAction;
    }

    @Override
    public void rollback() {
        Closeable rollbackAction = SortingFileWriter.createRollbackAction(this.fileSystem, this.tempFiles);
        try (Closer closer = Closer.create();){
            closer.register(this.outputWriter::rollback);
            closer.register(rollbackAction);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static Closeable createRollbackAction(TrinoFileSystem fileSystem, Queue<TempFile> tempFiles) {
        return () -> {
            for (TempFile file : tempFiles) {
                SortingFileWriter.cleanupFile(fileSystem, file.getPath());
            }
        };
    }

    @Override
    public long getValidationCpuNanos() {
        return this.outputWriter.getValidationCpuNanos();
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("tempFilePrefix", (Object)this.tempFilePrefix).add("outputWriter", (Object)this.outputWriter).toString();
    }

    @Override
    public Optional<Runnable> getVerificationTask() {
        return this.outputWriter.getVerificationTask();
    }

    private void flushToTempFile() {
        this.writeTempFile(writer -> this.sortBuffer.flushTo(writer::writePage));
    }

    private void writeSorted() {
        this.combineFiles();
        this.mergeFiles(this.tempFiles, this.outputWriter::appendRows);
    }

    private void combineFiles() {
        while (this.tempFiles.size() > this.maxOpenTempFiles) {
            int count = Math.min(this.maxOpenTempFiles, this.tempFiles.size() - (this.maxOpenTempFiles - 1));
            List smallestFiles = (List)IntStream.range(0, count).mapToObj(i -> this.tempFiles.poll()).collect(ImmutableList.toImmutableList());
            this.writeTempFile(writer -> this.mergeFiles(smallestFiles, writer::writePage));
        }
    }

    private void mergeFiles(Iterable<TempFile> files, Consumer<Page> consumer) {
        try (Closer closer = Closer.create();){
            ArrayList<Iterator<Page>> iterators = new ArrayList<Iterator<Page>>();
            for (TempFile tempFile : files) {
                String file = tempFile.getPath();
                TrinoInputFile inputFile = this.fileSystem.newInputFile(file);
                HdfsOrcDataSource dataSource = new HdfsOrcDataSource(new OrcDataSourceId(file), inputFile.length(), new OrcReaderOptions(), inputFile, new FileFormatDataSourceStats());
                closer.register((Closeable)((Object)dataSource));
                iterators.add((Iterator<Page>)((Object)new TempFileReader(this.types, (OrcDataSource)dataSource)));
            }
            new MergingPageIterator(iterators, this.types, this.sortFields, this.sortOrders, this.typeOperators).forEachRemaining(consumer);
            for (TempFile tempFile : files) {
                this.fileSystem.deleteFile(tempFile.getPath());
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void writeTempFile(Consumer<TempFileWriter> consumer) {
        String tempFile = this.getTempFileName();
        try (TempFileWriter writer = new TempFileWriter(this.types, this.tempFileSinkFactory.createSink(this.fileSystem, tempFile));){
            consumer.accept(writer);
            writer.close();
            this.tempFiles.add(new TempFile(tempFile, writer.getWrittenBytes()));
        }
        catch (IOException | UncheckedIOException e) {
            SortingFileWriter.cleanupFile(this.fileSystem, tempFile);
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_WRITER_DATA_ERROR, "Failed to write temporary file: " + tempFile, (Throwable)e);
        }
    }

    private static void cleanupFile(TrinoFileSystem fileSystem, String file) {
        try {
            fileSystem.deleteFile(file);
        }
        catch (IOException e) {
            log.warn((Throwable)e, "Failed to delete temporary file: %s", new Object[]{file});
        }
    }

    private String getTempFileName() {
        return this.tempFilePrefix + "." + this.nextFileId.getAndIncrement();
    }

    public static interface TempFileSinkFactory {
        public OrcDataSink createSink(TrinoFileSystem var1, String var2) throws IOException;
    }

    private static class TempFile {
        private final String path;
        private final long size;

        public TempFile(String path, long size) {
            Preconditions.checkArgument((size >= 0L ? 1 : 0) != 0, (Object)"size is negative");
            this.path = Objects.requireNonNull(path, "path is null");
            this.size = size;
        }

        public String getPath() {
            return this.path;
        }

        public long getSize() {
            return this.size;
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("path", (Object)this.path).add("size", this.size).toString();
        }
    }
}

