/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.s3select;

import com.amazonaws.AbortedException;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CompressionType;
import com.amazonaws.services.s3.model.ExpressionType;
import com.amazonaws.services.s3.model.InputSerialization;
import com.amazonaws.services.s3.model.OutputSerialization;
import com.amazonaws.services.s3.model.ScanRange;
import com.amazonaws.services.s3.model.SelectObjectContentRequest;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.google.common.io.Closer;
import io.airlift.units.Duration;
import io.trino.plugin.hive.s3.HiveS3Config;
import io.trino.plugin.hive.s3.TrinoS3FileSystem;
import io.trino.plugin.hive.s3select.TrinoS3ClientFactory;
import io.trino.plugin.hive.s3select.TrinoS3SelectClient;
import io.trino.plugin.hive.util.RetryDriver;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.TimeUnit;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.LongWritable;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.io.compress.BZip2Codec;
import org.apache.hadoop.io.compress.CompressionCodec;
import org.apache.hadoop.io.compress.CompressionCodecFactory;
import org.apache.hadoop.io.compress.GzipCodec;
import org.apache.hadoop.mapred.RecordReader;
import org.apache.hadoop.util.LineReader;

@ThreadSafe
public abstract class S3SelectLineRecordReader
implements RecordReader<LongWritable, Text> {
    private InputStream selectObjectContent;
    private long processedRecords;
    private long recordsFromS3;
    private long position;
    private LineReader reader;
    private boolean isFirstLine;
    private static final Duration BACKOFF_MIN_SLEEP = new Duration(1.0, TimeUnit.SECONDS);
    private final TrinoS3SelectClient selectClient;
    private final long start;
    private final long end;
    private final int maxAttempts;
    private final Duration maxBackoffTime;
    private final Duration maxRetryTime;
    private final Closer closer = Closer.create();
    private final SelectObjectContentRequest selectObjectContentRequest;
    private final CompressionCodecFactory compressionCodecFactory;
    private final String lineDelimiter;
    private final Properties schema;
    private final CompressionType compressionType;

    public S3SelectLineRecordReader(Configuration configuration, Path path, long start, long length, Properties schema, String ionSqlQuery, TrinoS3ClientFactory s3ClientFactory) {
        Objects.requireNonNull(configuration, "configuration is null");
        Objects.requireNonNull(schema, "schema is null");
        Objects.requireNonNull(path, "path is null");
        Objects.requireNonNull(ionSqlQuery, "ionSqlQuery is null");
        Objects.requireNonNull(s3ClientFactory, "s3ClientFactory is null");
        this.lineDelimiter = schema.getProperty("line.delim", "\n");
        this.processedRecords = 0L;
        this.recordsFromS3 = 0L;
        this.position = this.start = start;
        this.end = this.start + length;
        this.isFirstLine = true;
        this.compressionCodecFactory = new CompressionCodecFactory(configuration);
        this.compressionType = this.getCompressionType(path);
        this.schema = schema;
        this.selectObjectContentRequest = this.buildSelectObjectRequest(ionSqlQuery, path);
        HiveS3Config defaults = new HiveS3Config();
        this.maxAttempts = configuration.getInt("trino.s3.max-client-retries", defaults.getS3MaxClientRetries()) + 1;
        this.maxBackoffTime = Duration.valueOf((String)configuration.get("trino.s3.max-backoff-time", defaults.getS3MaxBackoffTime().toString()));
        this.maxRetryTime = Duration.valueOf((String)configuration.get("trino.s3.max-retry-time", defaults.getS3MaxRetryTime().toString()));
        this.selectClient = new TrinoS3SelectClient(configuration, s3ClientFactory);
        this.closer.register((Closeable)this.selectClient);
    }

    protected abstract InputSerialization buildInputSerialization();

    protected abstract OutputSerialization buildOutputSerialization();

    protected abstract boolean shouldEnableScanRange();

    protected Properties getSchema() {
        return this.schema;
    }

    protected CompressionType getCompressionType() {
        return this.compressionType;
    }

    public SelectObjectContentRequest buildSelectObjectRequest(String query, Path path) {
        SelectObjectContentRequest selectObjectRequest = new SelectObjectContentRequest();
        URI uri = path.toUri();
        selectObjectRequest.setBucketName(TrinoS3FileSystem.extractBucketName(uri));
        selectObjectRequest.setKey(TrinoS3FileSystem.keyFromPath(path));
        selectObjectRequest.setExpression(query);
        selectObjectRequest.setExpressionType(ExpressionType.SQL);
        InputSerialization selectObjectInputSerialization = this.buildInputSerialization();
        selectObjectRequest.setInputSerialization(selectObjectInputSerialization);
        OutputSerialization selectObjectOutputSerialization = this.buildOutputSerialization();
        selectObjectRequest.setOutputSerialization(selectObjectOutputSerialization);
        if (this.shouldEnableScanRange()) {
            ScanRange scanRange = new ScanRange();
            scanRange.setStart(this.getStart());
            scanRange.setEnd(this.getEnd());
            selectObjectRequest.setScanRange(scanRange);
        }
        return selectObjectRequest;
    }

    protected CompressionType getCompressionType(Path path) {
        CompressionCodec codec = this.compressionCodecFactory.getCodec(path);
        if (codec == null) {
            return CompressionType.NONE;
        }
        if (codec instanceof GzipCodec) {
            return CompressionType.GZIP;
        }
        if (codec instanceof BZip2Codec) {
            return CompressionType.BZIP2;
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Compression extension not supported for S3 Select: " + path);
    }

    private int readLine(Text value) throws IOException {
        try {
            return RetryDriver.retry().maxAttempts(this.maxAttempts).exponentialBackoff(BACKOFF_MIN_SLEEP, this.maxBackoffTime, this.maxRetryTime, 2.0).stopOn(InterruptedException.class, UnrecoverableS3OperationException.class, AbortedException.class).run("readRecordsContentStream", () -> {
                if (this.isFirstLine) {
                    this.recordsFromS3 = 0L;
                    this.selectObjectContent = this.selectClient.getRecordsContent(this.selectObjectContentRequest);
                    this.closer.register((Closeable)this.selectObjectContent);
                    this.reader = new LineReader(this.selectObjectContent, this.lineDelimiter.getBytes(StandardCharsets.UTF_8));
                    this.closer.register((Closeable)this.reader);
                    this.isFirstLine = false;
                }
                try {
                    return this.reader.readLine(value);
                }
                catch (RuntimeException e) {
                    this.isFirstLine = true;
                    this.recordsFromS3 = 0L;
                    if (e instanceof AmazonS3Exception) {
                        switch (((AmazonS3Exception)((Object)((Object)e))).getStatusCode()) {
                            case 400: 
                            case 403: 
                            case 404: {
                                throw new UnrecoverableS3OperationException(this.selectClient.getBucketName(), this.selectClient.getKeyName(), e);
                            }
                        }
                    }
                    throw e;
                }
            });
        }
        catch (AbortedException | InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedIOException();
        }
        catch (Exception e) {
            Throwables.throwIfInstanceOf((Throwable)e, IOException.class);
            Throwables.throwIfUnchecked((Throwable)e);
            throw new RuntimeException(e);
        }
    }

    public synchronized boolean next(LongWritable key, Text value) throws IOException {
        int bytes;
        do {
            if ((bytes = this.readLine(value)) <= 0) {
                if (!this.selectClient.isRequestComplete()) {
                    throw new IOException("S3 Select request was incomplete as End Event was not received");
                }
                return false;
            }
            ++this.recordsFromS3;
        } while (this.recordsFromS3 <= this.processedRecords);
        this.position += (long)bytes;
        ++this.processedRecords;
        key.set(this.processedRecords);
        return true;
    }

    public LongWritable createKey() {
        return new LongWritable();
    }

    public Text createValue() {
        return new Text();
    }

    public long getPos() {
        return this.position;
    }

    public void close() throws IOException {
        this.closer.close();
    }

    public float getProgress() {
        return (float)(this.position - this.start) / (float)(this.end - this.start);
    }

    protected long getStart() {
        return this.start;
    }

    protected long getEnd() {
        return this.end;
    }

    protected String getLineDelimiter() {
        return this.lineDelimiter;
    }

    @VisibleForTesting
    static class UnrecoverableS3OperationException
    extends RuntimeException {
        public UnrecoverableS3OperationException(String bucket, String key, Throwable cause) {
            super(String.format("%s (Bucket: %s, Key: %s)", cause, bucket, key));
        }
    }
}

