/*
 * Decompiled with CFR 0.152.
 */
package io.trino.hive.formats.line.sequence;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.Closer;
import com.google.errorprone.annotations.FormatMethod;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceInput;
import io.airlift.slice.Slices;
import io.trino.filesystem.TrinoInputFile;
import io.trino.hive.formats.FileCorruptionException;
import io.trino.hive.formats.ReadWriteUtils;
import io.trino.hive.formats.TrinoDataInputStream;
import io.trino.hive.formats.compression.Codec;
import io.trino.hive.formats.compression.CompressionKind;
import io.trino.hive.formats.compression.ValueDecompressor;
import io.trino.hive.formats.line.LineBuffer;
import io.trino.hive.formats.line.LineReader;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Map;
import java.util.Objects;

public final class SequenceFileReader
implements LineReader {
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(SequenceFileReader.class);
    private static final Slice SEQUENCE_FILE_MAGIC = Slices.utf8Slice((String)"SEQ");
    private static final byte SEQUENCE_FILE_VERSION = 6;
    private static final int MAX_METADATA_ENTRIES = 500000;
    private static final int MAX_METADATA_STRING_LENGTH = 0x100000;
    private final String location;
    private final TrinoDataInputStream input;
    private final String keyClassName;
    private final String valueClassName;
    private final Map<String, String> metadata;
    private final long syncFirst;
    private final long syncSecond;
    private long rowsRead;
    private final ValueReader valueReader;
    private boolean closed;
    private final Closer closer = Closer.create();

    public SequenceFileReader(TrinoInputFile inputFile, long offset, long length) throws IOException {
        try {
            ValueDecompressor decompressor;
            Objects.requireNonNull(inputFile, "inputFile is null");
            this.location = inputFile.location();
            this.input = new TrinoDataInputStream(inputFile.newStream());
            this.closer.register((Closeable)this.input);
            SequenceFileReader.verify(offset >= 0L, "offset is negative", new Object[0]);
            SequenceFileReader.verify(offset < inputFile.length(), "offset is greater than data size", new Object[0]);
            SequenceFileReader.verify(length >= 1L, "length must be at least 1", new Object[0]);
            long end = offset + length;
            long fileSize = inputFile.length();
            SequenceFileReader.verify(end <= fileSize, "offset plus length is greater than data size", new Object[0]);
            Slice magic = this.input.readSlice(SEQUENCE_FILE_MAGIC.length());
            SequenceFileReader.verify(SEQUENCE_FILE_MAGIC.equals((Object)magic), "File %s is not a Sequence File", this.location);
            byte version = this.input.readByte();
            SequenceFileReader.verify(version == 6, "Sequence File %s version %s is not supported", this.location, version);
            this.keyClassName = this.readLengthPrefixedString(this.input).toStringUtf8();
            this.valueClassName = this.readLengthPrefixedString(this.input).toStringUtf8();
            SequenceFileReader.verify("org.apache.hadoop.io.Text".equals(this.valueClassName), "Sequence File %s value class %s is not supported", this.location, this.valueClassName);
            boolean compressed = this.input.readBoolean();
            boolean blockCompression = this.input.readBoolean();
            SequenceFileReader.verify(!blockCompression || compressed, "Uncompressed Sequence File %s has block compression enabled", this.location);
            if (compressed) {
                String codecClassName = this.readLengthPrefixedString(this.input).toStringUtf8();
                CompressionKind compressionKind = CompressionKind.fromHadoopClassName(codecClassName);
                Preconditions.checkArgument((compressionKind != CompressionKind.LZOP ? 1 : 0) != 0, (Object)"LZOP cannot be used with SequenceFile. LZO compression can be used, but LZ4 is preferred.");
                Codec codecFromHadoopClassName = compressionKind.createCodec();
                decompressor = codecFromHadoopClassName.createValueDecompressor();
            } else {
                decompressor = null;
            }
            int metadataEntries = Integer.reverseBytes(this.input.readInt());
            SequenceFileReader.verify(metadataEntries >= 0, "Invalid metadata entry count %s in Sequence File %s", metadataEntries, this.location);
            SequenceFileReader.verify(metadataEntries <= 500000, "Too many metadata entries (%s) in Sequence File %s", metadataEntries, this.location);
            ImmutableMap.Builder metadataBuilder = ImmutableMap.builder();
            for (int i = 0; i < metadataEntries; ++i) {
                String key = this.readLengthPrefixedString(this.input).toStringUtf8();
                String value = this.readLengthPrefixedString(this.input).toStringUtf8();
                metadataBuilder.put((Object)key, (Object)value);
            }
            this.metadata = metadataBuilder.buildOrThrow();
            this.syncFirst = this.input.readLong();
            this.syncSecond = this.input.readLong();
            if (offset != 0L) {
                long startOfSyncSequence = ReadWriteUtils.findFirstSyncPosition(inputFile, offset, length, this.syncFirst, this.syncSecond);
                if (startOfSyncSequence < 0L) {
                    this.close();
                    this.valueReader = null;
                    return;
                }
                this.input.seek(startOfSyncSequence);
            }
            this.valueReader = blockCompression ? new BlockCompressedValueReader(this.location, fileSize, this.input, decompressor, end, this.syncFirst, this.syncSecond) : new SingleValueReader(this.location, fileSize, this.input, decompressor, end, this.syncFirst, this.syncSecond);
        }
        catch (Throwable e) {
            Closer closer = this.closer;
            try {
                throw e;
            }
            catch (Throwable throwable) {
                if (closer != null) {
                    try {
                        closer.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
        }
    }

    public String getFileLocation() {
        return this.location;
    }

    public String getKeyClassName() {
        return this.keyClassName;
    }

    public String getValueClassName() {
        return this.valueClassName;
    }

    public Map<String, String> getMetadata() {
        return this.metadata;
    }

    @Override
    public long getBytesRead() {
        return this.input.getReadBytes();
    }

    public long getRowsRead() {
        return this.rowsRead;
    }

    @Override
    public long getReadTimeNanos() {
        return this.input.getReadTimeNanos();
    }

    public Slice getSync() {
        Slice sync = Slices.allocate((int)16);
        sync.setLong(0, this.syncFirst);
        sync.setLong(8, this.syncSecond);
        return sync;
    }

    @Override
    public long getRetainedSize() {
        return (long)INSTANCE_SIZE + this.input.getRetainedSize() + this.valueReader.getRetainedSize();
    }

    @Override
    public boolean isClosed() {
        return this.closed;
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        this.closer.close();
    }

    @Override
    public boolean readLine(LineBuffer lineBuffer) throws IOException {
        boolean readLine;
        lineBuffer.reset();
        if (this.closed) {
            return false;
        }
        try {
            readLine = this.valueReader.readLine(lineBuffer);
        }
        catch (Throwable e) {
            SequenceFileReader ignored = this;
            try {
                throw e;
            }
            catch (Throwable throwable) {
                if (ignored != null) {
                    try {
                        ignored.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
        }
        if (readLine) {
            ++this.rowsRead;
        } else {
            this.close();
        }
        return readLine;
    }

    private Slice readLengthPrefixedString(TrinoDataInputStream in) throws IOException {
        int length = Math.toIntExact(ReadWriteUtils.readVInt(in));
        SequenceFileReader.verify(length <= 0x100000, "Metadata string value is too long (%s) in Sequence File %s", length, this.location);
        return in.readSlice(length);
    }

    @FormatMethod
    private static void verify(boolean expression, String messageFormat, Object ... args) throws FileCorruptionException {
        if (!expression) {
            throw new FileCorruptionException(messageFormat, args);
        }
    }

    private static interface ValueReader {
        public long getRetainedSize();

        public boolean readLine(LineBuffer var1) throws IOException;
    }

    private static class BlockCompressedValueReader
    implements ValueReader {
        private static final int INSTANCE_SIZE = SizeOf.instanceSize(BlockCompressedValueReader.class);
        private final String location;
        private final long fileSize;
        private final TrinoDataInputStream input;
        private final long end;
        private final long syncFirst;
        private final long syncSecond;
        private final ReadBuffer valueBuffer;
        private final ReadBuffer lengthsBuffer;
        private ValuesBlock valuesBlock = ValuesBlock.EMPTY_VALUES_BLOCK;

        public BlockCompressedValueReader(String location, long fileSize, TrinoDataInputStream input, ValueDecompressor decompressor, long end, long syncFirst, long syncSecond) {
            this.location = Objects.requireNonNull(location, "location is null");
            this.fileSize = fileSize;
            this.input = Objects.requireNonNull(input, "input is null");
            Objects.requireNonNull(decompressor, "decompressor is null");
            this.end = end;
            this.syncFirst = syncFirst;
            this.syncSecond = syncSecond;
            this.valueBuffer = new ReadBuffer(input, decompressor);
            this.lengthsBuffer = new ReadBuffer(input, decompressor);
        }

        @Override
        public long getRetainedSize() {
            return (long)INSTANCE_SIZE + this.input.getRetainedSize() + this.valueBuffer.getRetainedSize() + this.lengthsBuffer.getRetainedSize() + this.valuesBlock.getRetainedSize();
        }

        @Override
        public boolean readLine(LineBuffer lineBuffer) throws IOException {
            if (this.valuesBlock.readLine(lineBuffer)) {
                return true;
            }
            this.valuesBlock = ValuesBlock.EMPTY_VALUES_BLOCK;
            if (this.fileSize - this.input.getPos() == 0L) {
                return false;
            }
            SequenceFileReader.verify(this.fileSize - this.input.getPos() >= 4L, "SequenceFile truncated %s", this.location);
            SequenceFileReader.verify(-1 == this.input.readInt(), "Invalid sync in SequenceFile %s", this.location);
            SequenceFileReader.verify(this.fileSize - this.input.getPos() >= 20L, "SequenceFile truncated %s", this.location);
            if (this.input.getPos() - 4L >= this.end) {
                return false;
            }
            SequenceFileReader.verify(this.syncFirst == this.input.readLong() && this.syncSecond == this.input.readLong(), "Invalid sync in SequenceFile %s", this.location);
            int blockRecordCount = Math.toIntExact(ReadWriteUtils.readVInt(this.input));
            SequenceFileReader.verify(blockRecordCount > 0, "Invalid block record count %s", blockRecordCount);
            int keyLengthsLength = this.readVIntLength("key lengths");
            this.input.skipNBytes(keyLengthsLength);
            int keysLength = this.readVIntLength("keys");
            this.input.skipNBytes(keysLength);
            int valueLengthsLength = this.readVIntLength("value lengths");
            Slice valueLengths = this.lengthsBuffer.readBlock(valueLengthsLength);
            int valueLength = this.readVIntLength("values");
            Slice values = this.valueBuffer.readBlock(valueLength);
            this.valuesBlock = new ValuesBlock(blockRecordCount, (SliceInput)valueLengths.getInput(), (SliceInput)values.getInput());
            return this.valuesBlock.readLine(lineBuffer);
        }

        private int readVIntLength(String name) throws IOException {
            int blockSize = Math.toIntExact(ReadWriteUtils.readVInt(this.input));
            SequenceFileReader.verify(blockSize >= 0, "Invalid SequenceFile %s: %s block size is negative", this.location, name);
            return blockSize;
        }

        private static class ValuesBlock {
            private static final int INSTANCE_SIZE = SizeOf.instanceSize(ValuesBlock.class);
            public static final ValuesBlock EMPTY_VALUES_BLOCK = new ValuesBlock(0, (SliceInput)Slices.EMPTY_SLICE.getInput(), (SliceInput)Slices.EMPTY_SLICE.getInput());
            private final SliceInput lengthsInput;
            private final SliceInput dataInput;
            private int remainingRows;

            public ValuesBlock(int rowCount, SliceInput lengthsInput, SliceInput dataInput) {
                Preconditions.checkArgument((rowCount >= 0 ? 1 : 0) != 0, (Object)"rowCount is negative");
                this.remainingRows = rowCount;
                this.lengthsInput = Objects.requireNonNull(lengthsInput, "lengthsInput is null");
                this.dataInput = Objects.requireNonNull(dataInput, "dataInput is null");
            }

            public long getRetainedSize() {
                return (long)INSTANCE_SIZE + this.lengthsInput.getRetainedSize() + this.dataInput.getRetainedSize();
            }

            public boolean readLine(LineBuffer lineBuffer) throws IOException {
                if (this.remainingRows <= 0) {
                    return false;
                }
                --this.remainingRows;
                int valueLength = Math.toIntExact(ReadWriteUtils.readVInt(this.lengthsInput));
                SequenceFileReader.verify(valueLength >= 0, "Value length is negative", new Object[0]);
                long expectedValueEnd = this.dataInput.position() + (long)valueLength;
                int textLength = (int)ReadWriteUtils.readVInt(this.dataInput);
                SequenceFileReader.verify(textLength >= 0 && textLength <= valueLength, "Invalid text length: %s", textLength);
                lineBuffer.write((InputStream)this.dataInput, textLength);
                SequenceFileReader.verify(expectedValueEnd == this.dataInput.position(), "Raw value larger than text value", new Object[0]);
                return true;
            }
        }
    }

    private static class SingleValueReader
    implements ValueReader {
        private static final int INSTANCE_SIZE = SizeOf.instanceSize(SingleValueReader.class);
        private final String location;
        private final long fileSize;
        private final TrinoDataInputStream input;
        private final ValueDecompressor decompressor;
        private final long end;
        private final long syncFirst;
        private final long syncSecond;
        private final DynamicSliceOutput compressedBuffer = new DynamicSliceOutput(0);
        private final DynamicSliceOutput uncompressedBuffer = new DynamicSliceOutput(0);

        public SingleValueReader(String location, long fileSize, TrinoDataInputStream input, ValueDecompressor decompressor, long end, long syncFirst, long syncSecond) {
            this.location = Objects.requireNonNull(location, "location is null");
            this.fileSize = fileSize;
            this.input = Objects.requireNonNull(input, "input is null");
            this.decompressor = decompressor;
            this.end = end;
            this.syncFirst = syncFirst;
            this.syncSecond = syncSecond;
        }

        @Override
        public long getRetainedSize() {
            return (long)INSTANCE_SIZE + this.input.getRetainedSize() + this.compressedBuffer.getRetainedSize() + this.uncompressedBuffer.getRetainedSize();
        }

        @Override
        public boolean readLine(LineBuffer lineBuffer) throws IOException {
            if (this.fileSize - this.input.getPos() == 0L) {
                return false;
            }
            SequenceFileReader.verify(this.fileSize - this.input.getPos() >= 4L, "SequenceFile truncated %s", this.location);
            int recordLength = Integer.reverseBytes(this.input.readInt());
            if (recordLength == -1) {
                SequenceFileReader.verify(this.fileSize - this.input.getPos() >= 16L, "SequenceFile truncated %s", this.location);
                if (this.input.getPos() - 4L >= this.end) {
                    return false;
                }
                SequenceFileReader.verify(this.syncFirst == this.input.readLong() && this.syncSecond == this.input.readLong(), "Invalid sync in SequenceFile %s", this.location);
                if (this.fileSize - this.input.getPos() == 0L) {
                    return false;
                }
                SequenceFileReader.verify(this.fileSize - this.input.getPos() >= 4L, "SequenceFile truncated %s", this.location);
                recordLength = Integer.reverseBytes(this.input.readInt());
            }
            SequenceFileReader.verify(recordLength > 0, "Invalid block record count %s", recordLength);
            int keyLength = Integer.reverseBytes(this.input.readInt());
            SequenceFileReader.verify(keyLength >= 0, "Invalid SequenceFile %s: key length size is negative", this.location);
            this.input.skipNBytes(keyLength);
            int compressedValueLength = recordLength - keyLength;
            if (this.decompressor == null) {
                long expectedValueEnd = this.input.getPos() + (long)compressedValueLength;
                int textLength = Math.toIntExact(ReadWriteUtils.readVInt(this.input));
                this.input.readFully(lineBuffer, textLength);
                SequenceFileReader.verify(expectedValueEnd == this.input.getPos(), "Raw value larger than text value", new Object[0]);
                return true;
            }
            this.compressedBuffer.reset();
            this.input.readFully((OutputStream)this.compressedBuffer, compressedValueLength);
            Slice compressed = this.compressedBuffer.slice();
            this.uncompressedBuffer.reset();
            this.decompressor.decompress(compressed, (OutputStream)this.uncompressedBuffer);
            BasicSliceInput uncompressed = this.uncompressedBuffer.slice().getInput();
            int textLength = Math.toIntExact(ReadWriteUtils.readVInt((SliceInput)uncompressed));
            lineBuffer.write((InputStream)uncompressed, textLength);
            SequenceFileReader.verify(uncompressed.available() == 0, "Raw value larger than text value", new Object[0]);
            return true;
        }
    }

    private static class ReadBuffer {
        private static final int INSTANCE_SIZE = SizeOf.instanceSize(ReadBuffer.class);
        private final TrinoDataInputStream input;
        private final ValueDecompressor decompressor;
        private final DynamicSliceOutput compressedBuffer = new DynamicSliceOutput(0);
        private final DynamicSliceOutput uncompressedBuffer = new DynamicSliceOutput(0);

        public ReadBuffer(TrinoDataInputStream input, ValueDecompressor decompressor) {
            this.input = Objects.requireNonNull(input, "input is null");
            this.decompressor = decompressor;
        }

        public long getRetainedSize() {
            return (long)INSTANCE_SIZE + this.input.getRetainedSize() + this.compressedBuffer.getRetainedSize() + this.uncompressedBuffer.getRetainedSize();
        }

        public Slice readBlock(int length) throws IOException {
            this.compressedBuffer.reset();
            this.input.readFully((OutputStream)this.compressedBuffer, length);
            Slice compressed = this.compressedBuffer.slice();
            if (this.decompressor == null) {
                return compressed;
            }
            this.uncompressedBuffer.reset();
            this.decompressor.decompress(compressed, (OutputStream)this.uncompressedBuffer);
            return this.uncompressedBuffer.slice();
        }
    }
}

