/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import io.trino.hadoop.$internal.com.google.common.annotations.VisibleForTesting;
import io.trino.hadoop.$internal.com.google.common.base.Preconditions;
import io.trino.hadoop.$internal.com.google.common.collect.Lists;
import io.trino.hadoop.$internal.org.apache.commons.lang3.StringUtils;
import io.trino.hadoop.$internal.org.slf4j.Logger;
import io.trino.hadoop.$internal.org.slf4j.LoggerFactory;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.NavigableMap;
import java.util.NavigableSet;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.fs.BatchedRemoteIterator;
import org.apache.hadoop.hdfs.protocol.OpenFileEntry;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
public class LeaseManager {
    public static final Logger LOG = LoggerFactory.getLogger(LeaseManager.class.getName());
    private final FSNamesystem fsnamesystem;
    private long softLimit = 60000L;
    private long hardLimit = 3600000L;
    static final int INODE_FILTER_WORKER_COUNT_MAX = 4;
    static final int INODE_FILTER_WORKER_TASK_MIN = 512;
    private long lastHolderUpdateTime;
    private String internalLeaseHolder;
    private final SortedMap<String, Lease> leases = new TreeMap<String, Lease>();
    private final NavigableSet<Lease> sortedLeases = new TreeSet<Lease>(new Comparator<Lease>(){

        @Override
        public int compare(Lease o1, Lease o2) {
            if (o1.getLastUpdate() != o2.getLastUpdate()) {
                return Long.signum(o1.getLastUpdate() - o2.getLastUpdate());
            }
            return o1.holder.compareTo(o2.holder);
        }
    });
    private final TreeMap<Long, Lease> leasesById = new TreeMap();
    private Daemon lmthread;
    private volatile boolean shouldRunMonitor;

    LeaseManager(FSNamesystem fsnamesystem) {
        this.fsnamesystem = fsnamesystem;
        this.updateInternalLeaseHolder();
    }

    private void updateInternalLeaseHolder() {
        this.lastHolderUpdateTime = Time.monotonicNow();
        this.internalLeaseHolder = "HDFS_NameNode-" + Time.formatTime(Time.now());
    }

    String getInternalLeaseHolder() {
        long elapsed = Time.monotonicNow() - this.lastHolderUpdateTime;
        if (elapsed > this.hardLimit) {
            this.updateInternalLeaseHolder();
        }
        return this.internalLeaseHolder;
    }

    Lease getLease(String holder) {
        return (Lease)this.leases.get(holder);
    }

    synchronized long getNumUnderConstructionBlocks() {
        assert (this.fsnamesystem.hasReadLock()) : "The FSNamesystem read lock wasn'tacquired before counting under construction blocks";
        long numUCBlocks = 0L;
        for (Long id : this.getINodeIdWithLeases()) {
            INode inode = this.fsnamesystem.getFSDirectory().getInode(id);
            if (inode == null) {
                LOG.warn("Failed to find inode {} in getNumUnderConstructionBlocks().", (Object)id);
                continue;
            }
            INodeFile cons = inode.asFile();
            if (!cons.isUnderConstruction()) {
                LOG.warn("The file {} is not under construction but has lease.", (Object)cons.getFullPathName());
                continue;
            }
            BlockInfo[] blocks = cons.getBlocks();
            if (blocks == null) continue;
            for (BlockInfo b : blocks) {
                if (b.isComplete()) continue;
                ++numUCBlocks;
            }
        }
        LOG.info("Number of blocks under construction: {}", (Object)numUCBlocks);
        return numUCBlocks;
    }

    Collection<Long> getINodeIdWithLeases() {
        return this.leasesById.keySet();
    }

    @VisibleForTesting
    Set<INodesInPath> getINodeWithLeases() throws IOException {
        return this.getINodeWithLeases(null);
    }

    private synchronized INode[] getINodesWithLease() {
        ArrayList<INode> inodes = new ArrayList<INode>(this.leasesById.size());
        for (long inodeId : this.leasesById.keySet()) {
            INode currentINode = this.fsnamesystem.getFSDirectory().getInode(inodeId);
            if (currentINode == null || !currentINode.isFile() || this.fsnamesystem.isFileDeleted(currentINode.asFile())) continue;
            inodes.add(currentINode);
        }
        return inodes.toArray(new INode[0]);
    }

    public Set<INodesInPath> getINodeWithLeases(final INodeDirectory ancestorDir) throws IOException {
        assert (this.fsnamesystem.hasReadLock());
        long startTimeMs = Time.monotonicNow();
        HashSet<INodesInPath> iipSet = new HashSet<INodesInPath>();
        final INode[] inodes = this.getINodesWithLease();
        final int inodeCount = inodes.length;
        if (inodeCount == 0) {
            return iipSet;
        }
        ArrayList<Future<List<INodesInPath>>> futureList = Lists.newArrayList();
        final int workerCount = Math.min(4, (inodeCount - 1) / 512 + 1);
        ExecutorService inodeFilterService = Executors.newFixedThreadPool(workerCount);
        int workerIdx = 0;
        while (workerIdx < workerCount) {
            final int n = workerIdx++;
            Callable<List<INodesInPath>> c = new Callable<List<INodesInPath>>(){

                @Override
                public List<INodesInPath> call() {
                    ArrayList<INodesInPath> iNodesInPaths = Lists.newArrayList();
                    for (int idx = n; idx < inodeCount; idx += workerCount) {
                        INode inode = inodes[idx];
                        if (!inode.isFile()) continue;
                        INodesInPath inodesInPath = INodesInPath.fromINode(LeaseManager.this.fsnamesystem.getFSDirectory().getRoot(), inode.asFile());
                        if (ancestorDir != null && !inodesInPath.isDescendant(ancestorDir)) continue;
                        iNodesInPaths.add(inodesInPath);
                    }
                    return iNodesInPaths;
                }
            };
            futureList.add(inodeFilterService.submit(c));
        }
        inodeFilterService.shutdown();
        for (Future future : futureList) {
            try {
                iipSet.addAll((Collection)future.get());
            }
            catch (Exception e) {
                throw new IOException("Failed to get files with active leases", e);
            }
        }
        long endTimeMs = Time.monotonicNow();
        if (endTimeMs - startTimeMs > 1000L) {
            LOG.info("Took {} ms to collect {} open files with leases {}", endTimeMs - startTimeMs, iipSet.size(), ancestorDir != null ? " under " + ancestorDir.getFullPathName() : ".");
        }
        return iipSet;
    }

    public BatchedRemoteIterator.BatchedListEntries<OpenFileEntry> getUnderConstructionFiles(long prevId) throws IOException {
        return this.getUnderConstructionFiles(prevId, "/");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public BatchedRemoteIterator.BatchedListEntries<OpenFileEntry> getUnderConstructionFiles(long prevId, String path) throws IOException {
        NavigableMap<Long, Lease> remainingLeases;
        assert (this.fsnamesystem.hasReadLock());
        LeaseManager leaseManager = this;
        synchronized (leaseManager) {
            remainingLeases = this.leasesById.tailMap(prevId, false);
        }
        Set inodeIds = remainingLeases.keySet();
        int numResponses = Math.min(this.fsnamesystem.getMaxListOpenFilesResponses(), inodeIds.size());
        ArrayList<OpenFileEntry> openFileEntries = Lists.newArrayListWithExpectedSize(numResponses);
        int count = 0;
        String fullPathName = null;
        for (Long inodeId : inodeIds) {
            INodeFile inodeFile = this.fsnamesystem.getFSDirectory().getInode(inodeId).asFile();
            if (!inodeFile.isUnderConstruction()) {
                LOG.warn("The file {} is not under construction but has lease.", (Object)inodeFile.getFullPathName());
                continue;
            }
            fullPathName = inodeFile.getFullPathName();
            if (StringUtils.isEmpty(path) || fullPathName.startsWith(path)) {
                openFileEntries.add(new OpenFileEntry(inodeFile.getId(), fullPathName, inodeFile.getFileUnderConstructionFeature().getClientName(), inodeFile.getFileUnderConstructionFeature().getClientMachine()));
                ++count;
            }
            if (count < numResponses) continue;
            break;
        }
        boolean hasMore = numResponses < remainingLeases.size();
        return new BatchedRemoteIterator.BatchedListEntries<OpenFileEntry>(openFileEntries, hasMore);
    }

    public synchronized Lease getLease(INodeFile src) {
        return this.leasesById.get(src.getId());
    }

    @VisibleForTesting
    public synchronized int countLease() {
        return this.sortedLeases.size();
    }

    synchronized long countPath() {
        return this.leasesById.size();
    }

    synchronized Lease addLease(String holder, long inodeId) {
        Lease lease = this.getLease(holder);
        if (lease == null) {
            lease = new Lease(holder);
            this.leases.put(holder, lease);
            this.sortedLeases.add(lease);
        } else {
            this.renewLease(lease);
        }
        this.leasesById.put(inodeId, lease);
        lease.files.add(inodeId);
        return lease;
    }

    synchronized void removeLease(long inodeId) {
        Lease lease = this.leasesById.get(inodeId);
        if (lease != null) {
            this.removeLease(lease, inodeId);
        }
    }

    private synchronized void removeLease(Lease lease, long inodeId) {
        this.leasesById.remove(inodeId);
        if (!lease.removeFile(inodeId)) {
            LOG.debug("inode {} not found in lease.files (={})", (Object)inodeId, (Object)lease);
        }
        if (!lease.hasFiles()) {
            this.leases.remove(lease.holder);
            if (!this.sortedLeases.remove(lease)) {
                LOG.error("{} not found in sortedLeases", (Object)lease);
            }
        }
    }

    synchronized void removeLease(String holder, INodeFile src) {
        Lease lease = this.getLease(holder);
        if (lease != null) {
            this.removeLease(lease, src.getId());
        } else {
            LOG.warn("Removing non-existent lease! holder={} src={}", (Object)holder, (Object)src.getFullPathName());
        }
    }

    synchronized void removeAllLeases() {
        this.sortedLeases.clear();
        this.leasesById.clear();
        this.leases.clear();
    }

    synchronized Lease reassignLease(Lease lease, INodeFile src, String newHolder) {
        assert (newHolder != null) : "new lease holder is null";
        if (lease != null) {
            this.removeLease(lease, src.getId());
        }
        return this.addLease(newHolder, src.getId());
    }

    synchronized void renewLease(String holder) {
        this.renewLease(this.getLease(holder));
    }

    synchronized void renewLease(Lease lease) {
        if (lease != null) {
            this.sortedLeases.remove(lease);
            lease.renew();
            this.sortedLeases.add(lease);
        }
    }

    synchronized void renewAllLeases() {
        for (Lease l : this.leases.values()) {
            this.renewLease(l);
        }
    }

    public void setLeasePeriod(long softLimit, long hardLimit) {
        this.softLimit = softLimit;
        this.hardLimit = hardLimit;
    }

    @VisibleForTesting
    synchronized boolean checkLeases() {
        boolean needSync = false;
        assert (this.fsnamesystem.hasWriteLock());
        long start = Time.monotonicNow();
        while (!this.sortedLeases.isEmpty() && ((Lease)this.sortedLeases.first()).expiredHardLimit() && !this.isMaxLockHoldToReleaseLease(start)) {
            Lease leaseToCheck = (Lease)this.sortedLeases.first();
            LOG.info("{} has expired hard limit", (Object)leaseToCheck);
            ArrayList<Long> removing = new ArrayList<Long>();
            Collection files = leaseToCheck.getFiles();
            Long[] leaseINodeIds = files.toArray(new Long[files.size()]);
            FSDirectory fsd = this.fsnamesystem.getFSDirectory();
            String p = null;
            String newHolder = this.getInternalLeaseHolder();
            for (Long id : leaseINodeIds) {
                try {
                    INodesInPath iip = INodesInPath.fromINode(fsd.getInode(id));
                    p = iip.getPath();
                    if (!p.startsWith("/")) {
                        throw new IOException("Invalid path in the lease " + p);
                    }
                    INodeFile lastINode = iip.getLastINode().asFile();
                    if (this.fsnamesystem.isFileDeleted(lastINode)) {
                        this.removeLease(lastINode.getId());
                        continue;
                    }
                    boolean completed = false;
                    try {
                        completed = this.fsnamesystem.internalReleaseLease(leaseToCheck, p, iip, newHolder);
                    }
                    catch (IOException e) {
                        LOG.warn("Cannot release the path {} in the lease {}. It will be retried.", p, leaseToCheck, e);
                        continue;
                    }
                    if (LOG.isDebugEnabled()) {
                        if (completed) {
                            LOG.debug("Lease recovery for inode {} is complete. File closed.", (Object)id);
                        } else {
                            LOG.debug("Started block recovery {} lease {}", (Object)p, (Object)leaseToCheck);
                        }
                    }
                    if (!needSync && !completed) {
                        needSync = true;
                    }
                }
                catch (IOException e) {
                    LOG.warn("Removing lease with an invalid path: {},{}", p, leaseToCheck, e);
                    removing.add(id);
                }
                if (!this.isMaxLockHoldToReleaseLease(start)) continue;
                LOG.debug("Breaking out of checkLeases after {} ms.", (Object)this.fsnamesystem.getMaxLockHoldToReleaseLeaseMs());
                break;
            }
            for (Long id : removing) {
                this.removeLease(leaseToCheck, id);
            }
        }
        return needSync;
    }

    private boolean isMaxLockHoldToReleaseLease(long start) {
        return Time.monotonicNow() - start > this.fsnamesystem.getMaxLockHoldToReleaseLeaseMs();
    }

    public synchronized String toString() {
        return this.getClass().getSimpleName() + "= {\n leases=" + this.leases + "\n sortedLeases=" + this.sortedLeases + "\n leasesById=" + this.leasesById + "\n}";
    }

    void startMonitor() {
        Preconditions.checkState(this.lmthread == null, "Lease Monitor already running");
        this.shouldRunMonitor = true;
        this.lmthread = new Daemon(new Monitor());
        this.lmthread.start();
    }

    void stopMonitor() {
        if (this.lmthread != null) {
            this.shouldRunMonitor = false;
            try {
                this.lmthread.interrupt();
                this.lmthread.join(3000L);
            }
            catch (InterruptedException ie) {
                LOG.warn("Encountered exception ", ie);
            }
            this.lmthread = null;
        }
    }

    @VisibleForTesting
    public void triggerMonitorCheckNow() {
        Preconditions.checkState(this.lmthread != null, "Lease monitor is not running");
        this.lmthread.interrupt();
    }

    @VisibleForTesting
    public void runLeaseChecks() {
        this.checkLeases();
    }

    class Monitor
    implements Runnable {
        final String name = this.getClass().getSimpleName();

        Monitor() {
        }

        @Override
        public void run() {
            while (LeaseManager.this.shouldRunMonitor && LeaseManager.this.fsnamesystem.isRunning()) {
                boolean needSync = false;
                try {
                    LeaseManager.this.fsnamesystem.writeLockInterruptibly();
                    try {
                        if (!LeaseManager.this.fsnamesystem.isInSafeMode()) {
                            needSync = LeaseManager.this.checkLeases();
                        }
                    }
                    finally {
                        LeaseManager.this.fsnamesystem.writeUnlock("leaseManager");
                        if (needSync) {
                            LeaseManager.this.fsnamesystem.getEditLog().logSync();
                        }
                    }
                    Thread.sleep(LeaseManager.this.fsnamesystem.getLeaseRecheckIntervalMs());
                }
                catch (InterruptedException ie) {
                    LOG.debug("{} is interrupted", (Object)this.name, (Object)ie);
                }
                catch (Throwable e) {
                    LOG.warn("Unexpected throwable: ", e);
                }
            }
        }
    }

    class Lease {
        private final String holder;
        private long lastUpdate;
        private final HashSet<Long> files = new HashSet();

        private Lease(String holder) {
            this.holder = holder;
            this.renew();
        }

        private void renew() {
            this.lastUpdate = Time.monotonicNow();
        }

        public boolean expiredHardLimit() {
            return Time.monotonicNow() - this.lastUpdate > LeaseManager.this.hardLimit;
        }

        public boolean expiredSoftLimit() {
            return Time.monotonicNow() - this.lastUpdate > LeaseManager.this.softLimit;
        }

        boolean hasFiles() {
            return !this.files.isEmpty();
        }

        boolean removeFile(long inodeId) {
            return this.files.remove(inodeId);
        }

        public String toString() {
            return "[Lease.  Holder: " + this.holder + ", pending creates: " + this.files.size() + "]";
        }

        public int hashCode() {
            return this.holder.hashCode();
        }

        private Collection<Long> getFiles() {
            return Collections.unmodifiableCollection(this.files);
        }

        String getHolder() {
            return this.holder;
        }

        @VisibleForTesting
        long getLastUpdate() {
            return this.lastUpdate;
        }
    }
}

