/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg;

import com.google.common.annotations.VisibleForTesting;
import io.airlift.slice.Murmur3Hash32;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceUtf8;
import io.airlift.slice.Slices;
import io.trino.plugin.iceberg.util.Timestamps;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.block.RunLengthEncodedBlock;
import io.trino.spi.predicate.Utils;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.FixedWidthType;
import io.trino.spi.type.Int128;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.LongTimestampWithTimeZone;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.Type;
import io.trino.spi.type.TypeUtils;
import io.trino.spi.type.UuidType;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.LongUnaryOperator;
import java.util.function.ToLongFunction;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import org.apache.iceberg.PartitionField;
import org.joda.time.DateTimeField;
import org.joda.time.chrono.ISOChronology;

public final class PartitionTransforms {
    private static final Pattern BUCKET_PATTERN = Pattern.compile("bucket\\[(\\d+)]");
    private static final Pattern TRUNCATE_PATTERN = Pattern.compile("truncate\\[(\\d+)]");
    private static final DateTimeField YEAR_FIELD = ISOChronology.getInstanceUTC().year();
    private static final DateTimeField MONTH_FIELD = ISOChronology.getInstanceUTC().monthOfYear();

    private PartitionTransforms() {
    }

    public static ColumnTransform getColumnTransform(PartitionField field, Type sourceType) {
        String transform;
        switch (transform = field.transform().toString()) {
            case "identity": {
                return PartitionTransforms.identity(sourceType);
            }
            case "year": {
                if (sourceType.equals(DateType.DATE)) {
                    return PartitionTransforms.yearsFromDate();
                }
                if (sourceType.equals(TimestampType.TIMESTAMP_MICROS)) {
                    return PartitionTransforms.yearsFromTimestamp();
                }
                if (sourceType.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS)) {
                    return PartitionTransforms.yearsFromTimestampWithTimeZone();
                }
                throw new UnsupportedOperationException("Unsupported type for 'year': " + field);
            }
            case "month": {
                if (sourceType.equals(DateType.DATE)) {
                    return PartitionTransforms.monthsFromDate();
                }
                if (sourceType.equals(TimestampType.TIMESTAMP_MICROS)) {
                    return PartitionTransforms.monthsFromTimestamp();
                }
                if (sourceType.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS)) {
                    return PartitionTransforms.monthsFromTimestampWithTimeZone();
                }
                throw new UnsupportedOperationException("Unsupported type for 'month': " + field);
            }
            case "day": {
                if (sourceType.equals(DateType.DATE)) {
                    return PartitionTransforms.daysFromDate();
                }
                if (sourceType.equals(TimestampType.TIMESTAMP_MICROS)) {
                    return PartitionTransforms.daysFromTimestamp();
                }
                if (sourceType.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS)) {
                    return PartitionTransforms.daysFromTimestampWithTimeZone();
                }
                throw new UnsupportedOperationException("Unsupported type for 'day': " + field);
            }
            case "hour": {
                if (sourceType.equals(TimestampType.TIMESTAMP_MICROS)) {
                    return PartitionTransforms.hoursFromTimestamp();
                }
                if (sourceType.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS)) {
                    return PartitionTransforms.hoursFromTimestampWithTimeZone();
                }
                throw new UnsupportedOperationException("Unsupported type for 'hour': " + field);
            }
            case "void": {
                return PartitionTransforms.voidTransform(sourceType);
            }
        }
        Matcher matcher = BUCKET_PATTERN.matcher(transform);
        if (matcher.matches()) {
            int count = Integer.parseInt(matcher.group(1));
            return PartitionTransforms.bucket(sourceType, count);
        }
        matcher = TRUNCATE_PATTERN.matcher(transform);
        if (matcher.matches()) {
            int width = Integer.parseInt(matcher.group(1));
            if (sourceType.equals(IntegerType.INTEGER)) {
                return PartitionTransforms.truncateInteger(width);
            }
            if (sourceType.equals(BigintType.BIGINT)) {
                return PartitionTransforms.truncateBigint(width);
            }
            if (sourceType instanceof DecimalType) {
                DecimalType decimalType = (DecimalType)sourceType;
                if (decimalType.isShort()) {
                    return PartitionTransforms.truncateShortDecimal(sourceType, width, decimalType);
                }
                return PartitionTransforms.truncateLongDecimal(sourceType, width, decimalType);
            }
            if (sourceType instanceof VarcharType) {
                return PartitionTransforms.truncateVarchar(width);
            }
            if (sourceType.equals(VarbinaryType.VARBINARY)) {
                return PartitionTransforms.truncateVarbinary(width);
            }
            throw new UnsupportedOperationException("Unsupported type for 'truncate': " + field);
        }
        throw new UnsupportedOperationException("Unsupported partition transform: " + field);
    }

    private static ColumnTransform identity(Type type) {
        return new ColumnTransform(type, false, true, Function.identity(), ValueTransform.identity(type));
    }

    @VisibleForTesting
    static ColumnTransform bucket(Type type, int count) {
        Hasher hasher = PartitionTransforms.getBucketingHash(type);
        return new ColumnTransform((Type)IntegerType.INTEGER, false, false, block -> PartitionTransforms.bucketBlock(block, count, hasher), (block, position) -> {
            if (block.isNull(position)) {
                return null;
            }
            int hash = hasher.hash(block, position);
            int bucket = (hash & Integer.MAX_VALUE) % count;
            return (long)bucket;
        });
    }

    private static Hasher getBucketingHash(Type type) {
        if (type.equals(IntegerType.INTEGER)) {
            return PartitionTransforms::hashInteger;
        }
        if (type.equals(BigintType.BIGINT)) {
            return PartitionTransforms::hashBigint;
        }
        if (type instanceof DecimalType) {
            DecimalType decimalType = (DecimalType)type;
            if (decimalType.isShort()) {
                return PartitionTransforms.hashShortDecimal(decimalType);
            }
            return PartitionTransforms.hashLongDecimal(decimalType);
        }
        if (type.equals(DateType.DATE)) {
            return PartitionTransforms::hashDate;
        }
        if (type.equals(TimeType.TIME_MICROS)) {
            return PartitionTransforms::hashTime;
        }
        if (type.equals(TimestampType.TIMESTAMP_MICROS)) {
            return PartitionTransforms::hashTimestamp;
        }
        if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS)) {
            return PartitionTransforms::hashTimestampWithTimeZone;
        }
        if (type instanceof VarcharType) {
            return PartitionTransforms::hashVarchar;
        }
        if (type.equals(VarbinaryType.VARBINARY)) {
            return PartitionTransforms::hashVarbinary;
        }
        if (type.equals(UuidType.UUID)) {
            return PartitionTransforms::hashUuid;
        }
        throw new UnsupportedOperationException("Unsupported type for 'bucket': " + type);
    }

    private static ColumnTransform yearsFromDate() {
        LongUnaryOperator transform = value -> PartitionTransforms.epochYear(TimeUnit.DAYS.toMillis(value));
        return new ColumnTransform((Type)IntegerType.INTEGER, false, true, block -> PartitionTransforms.transformBlock((Type)DateType.DATE, (FixedWidthType)IntegerType.INTEGER, block, transform), ValueTransform.from((Type)DateType.DATE, transform));
    }

    private static ColumnTransform monthsFromDate() {
        LongUnaryOperator transform = value -> PartitionTransforms.epochMonth(TimeUnit.DAYS.toMillis(value));
        return new ColumnTransform((Type)IntegerType.INTEGER, false, true, block -> PartitionTransforms.transformBlock((Type)DateType.DATE, (FixedWidthType)IntegerType.INTEGER, block, transform), ValueTransform.from((Type)DateType.DATE, transform));
    }

    private static ColumnTransform daysFromDate() {
        LongUnaryOperator transform = LongUnaryOperator.identity();
        return new ColumnTransform((Type)IntegerType.INTEGER, false, true, block -> PartitionTransforms.transformBlock((Type)DateType.DATE, (FixedWidthType)IntegerType.INTEGER, block, transform), ValueTransform.from((Type)DateType.DATE, transform));
    }

    private static ColumnTransform yearsFromTimestamp() {
        LongUnaryOperator transform = epochMicros -> PartitionTransforms.epochYear(Math.floorDiv(epochMicros, 1000));
        return new ColumnTransform((Type)IntegerType.INTEGER, false, true, block -> PartitionTransforms.transformBlock((Type)TimestampType.TIMESTAMP_MICROS, (FixedWidthType)IntegerType.INTEGER, block, transform), ValueTransform.from((Type)TimestampType.TIMESTAMP_MICROS, transform));
    }

    private static ColumnTransform monthsFromTimestamp() {
        LongUnaryOperator transform = epochMicros -> PartitionTransforms.epochMonth(Math.floorDiv(epochMicros, 1000));
        return new ColumnTransform((Type)IntegerType.INTEGER, false, true, block -> PartitionTransforms.transformBlock((Type)TimestampType.TIMESTAMP_MICROS, (FixedWidthType)IntegerType.INTEGER, block, transform), ValueTransform.from((Type)TimestampType.TIMESTAMP_MICROS, transform));
    }

    private static ColumnTransform daysFromTimestamp() {
        LongUnaryOperator transform = epochMicros -> PartitionTransforms.epochDay(Math.floorDiv(epochMicros, 1000));
        return new ColumnTransform((Type)IntegerType.INTEGER, false, true, block -> PartitionTransforms.transformBlock((Type)TimestampType.TIMESTAMP_MICROS, (FixedWidthType)IntegerType.INTEGER, block, transform), ValueTransform.from((Type)TimestampType.TIMESTAMP_MICROS, transform));
    }

    private static ColumnTransform hoursFromTimestamp() {
        LongUnaryOperator transform = epochMicros -> PartitionTransforms.epochHour(Math.floorDiv(epochMicros, 1000));
        return new ColumnTransform((Type)IntegerType.INTEGER, false, true, block -> PartitionTransforms.transformBlock((Type)TimestampType.TIMESTAMP_MICROS, (FixedWidthType)IntegerType.INTEGER, block, transform), ValueTransform.from((Type)TimestampType.TIMESTAMP_MICROS, transform));
    }

    private static ColumnTransform yearsFromTimestampWithTimeZone() {
        ToLongFunction<LongTimestampWithTimeZone> transform = value -> PartitionTransforms.epochYear(value.getEpochMillis());
        return new ColumnTransform((Type)IntegerType.INTEGER, false, true, block -> PartitionTransforms.extractTimestampWithTimeZone(block, transform), ValueTransform.fromTimestampTzTransform(transform));
    }

    private static ColumnTransform monthsFromTimestampWithTimeZone() {
        ToLongFunction<LongTimestampWithTimeZone> transform = value -> PartitionTransforms.epochMonth(value.getEpochMillis());
        return new ColumnTransform((Type)IntegerType.INTEGER, false, true, block -> PartitionTransforms.extractTimestampWithTimeZone(block, transform), ValueTransform.fromTimestampTzTransform(transform));
    }

    private static ColumnTransform daysFromTimestampWithTimeZone() {
        ToLongFunction<LongTimestampWithTimeZone> transform = value -> PartitionTransforms.epochDay(value.getEpochMillis());
        return new ColumnTransform((Type)IntegerType.INTEGER, false, true, block -> PartitionTransforms.extractTimestampWithTimeZone(block, transform), ValueTransform.fromTimestampTzTransform(transform));
    }

    private static ColumnTransform hoursFromTimestampWithTimeZone() {
        ToLongFunction<LongTimestampWithTimeZone> transform = value -> PartitionTransforms.epochHour(value.getEpochMillis());
        return new ColumnTransform((Type)IntegerType.INTEGER, false, true, block -> PartitionTransforms.extractTimestampWithTimeZone(block, transform), ValueTransform.fromTimestampTzTransform(transform));
    }

    private static Block extractTimestampWithTimeZone(Block block, ToLongFunction<LongTimestampWithTimeZone> function) {
        BlockBuilder builder = IntegerType.INTEGER.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            LongTimestampWithTimeZone value = Timestamps.getTimestampTz(block, position);
            IntegerType.INTEGER.writeLong(builder, function.applyAsLong(value));
        }
        return builder.build();
    }

    private static int hashInteger(Block block, int position) {
        return PartitionTransforms.bucketHash(IntegerType.INTEGER.getLong(block, position));
    }

    private static int hashBigint(Block block, int position) {
        return PartitionTransforms.bucketHash(BigintType.BIGINT.getLong(block, position));
    }

    private static Hasher hashShortDecimal(DecimalType decimal) {
        return (block, position) -> {
            BigDecimal value = Decimals.readBigDecimal((DecimalType)decimal, (Block)block, (int)position);
            return PartitionTransforms.bucketHash(Slices.wrappedBuffer((byte[])value.unscaledValue().toByteArray()));
        };
    }

    private static Hasher hashLongDecimal(DecimalType decimal) {
        return (block, position) -> {
            BigDecimal value = Decimals.readBigDecimal((DecimalType)decimal, (Block)block, (int)position);
            return PartitionTransforms.bucketHash(Slices.wrappedBuffer((byte[])value.unscaledValue().toByteArray()));
        };
    }

    private static int hashDate(Block block, int position) {
        return PartitionTransforms.bucketHash(DateType.DATE.getLong(block, position));
    }

    private static int hashTime(Block block, int position) {
        long picos = TimeType.TIME_MICROS.getLong(block, position);
        return PartitionTransforms.bucketHash(picos / 1000000L);
    }

    private static int hashTimestamp(Block block, int position) {
        return PartitionTransforms.bucketHash(TimestampType.TIMESTAMP_MICROS.getLong(block, position));
    }

    private static int hashTimestampWithTimeZone(Block block, int position) {
        return PartitionTransforms.bucketHash(Timestamps.timestampTzToMicros(Timestamps.getTimestampTz(block, position)));
    }

    private static int hashVarchar(Block block, int position) {
        return PartitionTransforms.bucketHash(VarcharType.VARCHAR.getSlice(block, position));
    }

    private static int hashVarbinary(Block block, int position) {
        return PartitionTransforms.bucketHash(VarbinaryType.VARBINARY.getSlice(block, position));
    }

    private static int hashUuid(Block block, int position) {
        return PartitionTransforms.bucketHash(UuidType.UUID.getSlice(block, position));
    }

    private static Block bucketBlock(Block block, int count, Hasher hasher) {
        BlockBuilder builder = IntegerType.INTEGER.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            int hash = hasher.hash(block, position);
            int bucket = (hash & Integer.MAX_VALUE) % count;
            IntegerType.INTEGER.writeLong(builder, (long)bucket);
        }
        return builder.build();
    }

    private static int bucketHash(long value) {
        return Murmur3Hash32.hash((long)value);
    }

    private static int bucketHash(Slice value) {
        return Murmur3Hash32.hash((Slice)value);
    }

    private static ColumnTransform truncateInteger(int width) {
        return new ColumnTransform((Type)IntegerType.INTEGER, false, true, block -> PartitionTransforms.truncateInteger(block, width), (block, position) -> {
            if (block.isNull(position)) {
                return null;
            }
            return PartitionTransforms.truncateInteger(block, position, width);
        });
    }

    private static Block truncateInteger(Block block, int width) {
        BlockBuilder builder = IntegerType.INTEGER.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            IntegerType.INTEGER.writeLong(builder, PartitionTransforms.truncateInteger(block, position, width));
        }
        return builder.build();
    }

    private static long truncateInteger(Block block, int position, int width) {
        long value = IntegerType.INTEGER.getLong(block, position);
        return value - (value % (long)width + (long)width) % (long)width;
    }

    private static ColumnTransform truncateBigint(int width) {
        return new ColumnTransform((Type)BigintType.BIGINT, false, true, block -> PartitionTransforms.truncateBigint(block, width), (block, position) -> {
            if (block.isNull(position)) {
                return null;
            }
            return PartitionTransforms.truncateBigint(block, position, width);
        });
    }

    private static Block truncateBigint(Block block, int width) {
        BlockBuilder builder = BigintType.BIGINT.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            BigintType.BIGINT.writeLong(builder, PartitionTransforms.truncateBigint(block, position, width));
        }
        return builder.build();
    }

    private static long truncateBigint(Block block, int position, int width) {
        long value = BigintType.BIGINT.getLong(block, position);
        return value - (value % (long)width + (long)width) % (long)width;
    }

    private static ColumnTransform truncateShortDecimal(Type type, int width, DecimalType decimal) {
        BigInteger unscaledWidth = BigInteger.valueOf(width);
        return new ColumnTransform(type, false, true, block -> PartitionTransforms.truncateShortDecimal(decimal, block, unscaledWidth), (block, position) -> {
            if (block.isNull(position)) {
                return null;
            }
            return PartitionTransforms.truncateShortDecimal(decimal, block, position, unscaledWidth);
        });
    }

    private static Block truncateShortDecimal(DecimalType type, Block block, BigInteger unscaledWidth) {
        BlockBuilder builder = type.createBlockBuilder(null, block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            type.writeLong(builder, PartitionTransforms.truncateShortDecimal(type, block, position, unscaledWidth));
        }
        return builder.build();
    }

    private static long truncateShortDecimal(DecimalType type, Block block, int position, BigInteger unscaledWidth) {
        BigDecimal value = Decimals.readBigDecimal((DecimalType)type, (Block)block, (int)position);
        BigDecimal truncated = PartitionTransforms.truncateDecimal(value, unscaledWidth);
        return Decimals.encodeShortScaledValue((BigDecimal)truncated, (int)type.getScale());
    }

    private static ColumnTransform truncateLongDecimal(Type type, int width, DecimalType decimal) {
        BigInteger unscaledWidth = BigInteger.valueOf(width);
        return new ColumnTransform(type, false, true, block -> PartitionTransforms.truncateLongDecimal(decimal, block, unscaledWidth), (block, position) -> {
            if (block.isNull(position)) {
                return null;
            }
            return PartitionTransforms.truncateLongDecimal(decimal, block, position, unscaledWidth);
        });
    }

    private static Block truncateLongDecimal(DecimalType type, Block block, BigInteger unscaledWidth) {
        BlockBuilder builder = type.createBlockBuilder(null, block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            type.writeObject(builder, (Object)PartitionTransforms.truncateLongDecimal(type, block, position, unscaledWidth));
        }
        return builder.build();
    }

    private static Int128 truncateLongDecimal(DecimalType type, Block block, int position, BigInteger unscaledWidth) {
        BigDecimal value = Decimals.readBigDecimal((DecimalType)type, (Block)block, (int)position);
        BigDecimal truncated = PartitionTransforms.truncateDecimal(value, unscaledWidth);
        return Decimals.encodeScaledValue((BigDecimal)truncated, (int)type.getScale());
    }

    private static BigDecimal truncateDecimal(BigDecimal value, BigInteger unscaledWidth) {
        BigDecimal remainder = new BigDecimal(value.unscaledValue().remainder(unscaledWidth).add(unscaledWidth).remainder(unscaledWidth), value.scale());
        return value.subtract(remainder);
    }

    private static ColumnTransform truncateVarchar(int width) {
        return new ColumnTransform((Type)VarcharType.VARCHAR, false, true, block -> PartitionTransforms.truncateVarchar(block, width), (block, position) -> {
            if (block.isNull(position)) {
                return null;
            }
            return PartitionTransforms.truncateVarchar(VarcharType.VARCHAR.getSlice(block, position), width);
        });
    }

    private static Block truncateVarchar(Block block, int width) {
        BlockBuilder builder = VarcharType.VARCHAR.createBlockBuilder(null, block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            Slice value = VarcharType.VARCHAR.getSlice(block, position);
            VarcharType.VARCHAR.writeSlice(builder, PartitionTransforms.truncateVarchar(value, width));
        }
        return builder.build();
    }

    private static Slice truncateVarchar(Slice value, int max) {
        if (value.length() <= max) {
            return value;
        }
        int end = SliceUtf8.offsetOfCodePoint((Slice)value, (int)0, (int)max);
        if (end < 0) {
            return value;
        }
        return value.slice(0, end);
    }

    private static ColumnTransform truncateVarbinary(int width) {
        return new ColumnTransform((Type)VarbinaryType.VARBINARY, false, true, block -> PartitionTransforms.truncateVarbinary(block, width), (block, position) -> {
            if (block.isNull(position)) {
                return null;
            }
            return PartitionTransforms.truncateVarbinary(VarbinaryType.VARBINARY.getSlice(block, position), width);
        });
    }

    private static Block truncateVarbinary(Block block, int width) {
        BlockBuilder builder = VarbinaryType.VARBINARY.createBlockBuilder(null, block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            Slice value = VarbinaryType.VARBINARY.getSlice(block, position);
            VarbinaryType.VARBINARY.writeSlice(builder, PartitionTransforms.truncateVarbinary(value, width));
        }
        return builder.build();
    }

    private static Slice truncateVarbinary(Slice value, int width) {
        if (value.length() <= width) {
            return value;
        }
        return value.slice(0, width);
    }

    private static ColumnTransform voidTransform(Type type) {
        Block nullBlock = Utils.nativeValueToBlock((Type)type, null);
        return new ColumnTransform(type, true, true, block -> RunLengthEncodedBlock.create((Block)nullBlock, (int)block.getPositionCount()), (block, position) -> null);
    }

    private static Block transformBlock(Type sourceType, FixedWidthType resultType, Block block, LongUnaryOperator function) {
        BlockBuilder builder = resultType.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            long value = sourceType.getLong(block, position);
            resultType.writeLong(builder, function.applyAsLong(value));
        }
        return builder.build();
    }

    @VisibleForTesting
    static long epochYear(long epochMilli) {
        return (long)YEAR_FIELD.get(epochMilli) - 1970L;
    }

    @VisibleForTesting
    static long epochMonth(long epochMilli) {
        long year = PartitionTransforms.epochYear(epochMilli);
        int month = MONTH_FIELD.get(epochMilli) - 1;
        return year * 12L + (long)month;
    }

    @VisibleForTesting
    static long epochDay(long epochMilli) {
        return Math.floorDiv(epochMilli, 86400000);
    }

    @VisibleForTesting
    static long epochHour(long epochMilli) {
        return Math.floorDiv(epochMilli, 3600000);
    }

    public static class ColumnTransform {
        private final Type type;
        private final boolean preservesNonNull;
        private final boolean monotonic;
        private final Function<Block, Block> blockTransform;
        private final ValueTransform valueTransform;

        public ColumnTransform(Type type, boolean preservesNonNull, boolean monotonic, Function<Block, Block> blockTransform, ValueTransform valueTransform) {
            this.type = Objects.requireNonNull(type, "type is null");
            this.preservesNonNull = preservesNonNull;
            this.monotonic = monotonic;
            this.blockTransform = Objects.requireNonNull(blockTransform, "transform is null");
            this.valueTransform = Objects.requireNonNull(valueTransform, "valueTransform is null");
        }

        public Type getType() {
            return this.type;
        }

        public boolean preservesNonNull() {
            return this.preservesNonNull;
        }

        public boolean isMonotonic() {
            return this.monotonic;
        }

        public Function<Block, Block> getBlockTransform() {
            return this.blockTransform;
        }

        public ValueTransform getValueTransform() {
            return this.valueTransform;
        }
    }

    public static interface ValueTransform {
        public static ValueTransform identity(Type type) {
            return (block, position) -> TypeUtils.readNativeValue((Type)type, (Block)block, (int)position);
        }

        public static ValueTransform from(Type sourceType, LongUnaryOperator transform) {
            return (block, position) -> {
                if (block.isNull(position)) {
                    return null;
                }
                return transform.applyAsLong(sourceType.getLong(block, position));
            };
        }

        public static ValueTransform fromTimestampTzTransform(ToLongFunction<LongTimestampWithTimeZone> transform) {
            return (block, position) -> {
                if (block.isNull(position)) {
                    return null;
                }
                return transform.applyAsLong(Timestamps.getTimestampTz(block, position));
            };
        }

        @Nullable
        public Object apply(Block var1, int var2);
    }

    private static interface Hasher {
        public int hash(Block var1, int var2);
    }
}

