/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.data.manager.realtime;

import com.google.common.annotations.VisibleForTesting;
import java.time.Clock;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.apache.pinot.common.metrics.AbstractMetrics;
import org.apache.pinot.common.metrics.ServerGauge;
import org.apache.pinot.common.metrics.ServerMetrics;
import org.apache.pinot.core.data.manager.realtime.RealtimeTableDataManager;
import org.apache.pinot.spi.stream.LongMsgOffset;
import org.apache.pinot.spi.stream.StreamPartitionMsgOffset;
import org.apache.pinot.spi.utils.builder.TableNameBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IngestionDelayTracker {
    private static final int SCHEDULED_EXECUTOR_THREAD_TICK_INTERVAL_MS = 300000;
    private static final int PARTITION_TIMEOUT_MS = 600000;
    private static final int INITIAL_SCHEDULED_EXECUTOR_THREAD_DELAY_MS = 100;
    private static final Logger _logger = LoggerFactory.getLogger((String)IngestionDelayTracker.class.getSimpleName());
    private final Map<Integer, IngestionTimestamps> _partitionToIngestionTimestampsMap = new ConcurrentHashMap<Integer, IngestionTimestamps>();
    private final Map<Integer, IngestionOffsets> _partitionToOffsetMap = new ConcurrentHashMap<Integer, IngestionOffsets>();
    private final Map<Integer, Long> _partitionsMarkedForVerification = new ConcurrentHashMap<Integer, Long>();
    final int _scheduledExecutorThreadTickIntervalMs;
    private final ScheduledExecutorService _scheduledExecutor = Executors.newScheduledThreadPool(2);
    private final ServerMetrics _serverMetrics;
    private final String _tableNameWithType;
    private final String _metricName;
    private final RealtimeTableDataManager _realTimeTableDataManager;
    private final Supplier<Boolean> _isServerReadyToServeQueries;
    private Clock _clock;

    public IngestionDelayTracker(ServerMetrics serverMetrics, final String tableNameWithType, RealtimeTableDataManager realtimeTableDataManager, int scheduledExecutorThreadTickIntervalMs, Supplier<Boolean> isServerReadyToServeQueries) throws RuntimeException {
        this._serverMetrics = serverMetrics;
        this._tableNameWithType = tableNameWithType;
        this._metricName = tableNameWithType;
        this._realTimeTableDataManager = realtimeTableDataManager;
        this._clock = Clock.systemUTC();
        this._isServerReadyToServeQueries = isServerReadyToServeQueries;
        if (scheduledExecutorThreadTickIntervalMs <= 0) {
            throw new RuntimeException(String.format("Illegal timer timeout argument, expected > 0, got=%d for table=%s", scheduledExecutorThreadTickIntervalMs, this._tableNameWithType));
        }
        this._scheduledExecutorThreadTickIntervalMs = scheduledExecutorThreadTickIntervalMs;
        ThreadFactory threadFactory = new ThreadFactory(){
            private final ThreadFactory _defaultFactory = Executors.defaultThreadFactory();

            @Override
            public Thread newThread(Runnable r) {
                Thread thread = this._defaultFactory.newThread(r);
                thread.setName("IngestionDelayTimerThread-" + TableNameBuilder.extractRawTableName((String)tableNameWithType));
                return thread;
            }
        };
        ((ScheduledThreadPoolExecutor)this._scheduledExecutor).setThreadFactory(threadFactory);
        this._scheduledExecutor.scheduleWithFixedDelay(this::timeoutInactivePartitions, 100L, this._scheduledExecutorThreadTickIntervalMs, TimeUnit.MILLISECONDS);
    }

    public IngestionDelayTracker(ServerMetrics serverMetrics, String tableNameWithType, RealtimeTableDataManager tableDataManager, Supplier<Boolean> isServerReadyToServeQueries) {
        this(serverMetrics, tableNameWithType, tableDataManager, 300000, isServerReadyToServeQueries);
    }

    private long getIngestionDelayMs(long ingestionTimeMs) {
        if (ingestionTimeMs < 0L) {
            return 0L;
        }
        long agedIngestionDelayMs = this._clock.millis() - ingestionTimeMs;
        agedIngestionDelayMs = Math.max(agedIngestionDelayMs, 0L);
        return agedIngestionDelayMs;
    }

    private long getPartitionOffsetLag(IngestionOffsets offset) {
        if (offset == null) {
            return 0L;
        }
        StreamPartitionMsgOffset currentOffset = offset._offset;
        StreamPartitionMsgOffset latestOffset = offset._latestOffset;
        if (!(currentOffset instanceof LongMsgOffset) || !(latestOffset instanceof LongMsgOffset)) {
            return 0L;
        }
        return ((LongMsgOffset)latestOffset).getOffset() - ((LongMsgOffset)currentOffset).getOffset();
    }

    private void removePartitionId(int partitionGroupId) {
        this._partitionToIngestionTimestampsMap.remove(partitionGroupId);
        this._partitionToOffsetMap.remove(partitionGroupId);
        this._partitionsMarkedForVerification.remove(partitionGroupId);
        this._serverMetrics.removePartitionGauge(this._metricName, partitionGroupId, (AbstractMetrics.Gauge)ServerGauge.REALTIME_INGESTION_DELAY_MS);
        this._serverMetrics.removePartitionGauge(this._metricName, partitionGroupId, (AbstractMetrics.Gauge)ServerGauge.END_TO_END_REALTIME_INGESTION_DELAY_MS);
        this._serverMetrics.removePartitionGauge(this._metricName, partitionGroupId, (AbstractMetrics.Gauge)ServerGauge.REALTIME_INGESTION_OFFSET_LAG);
    }

    private List<Integer> getPartitionsToBeVerified() {
        ArrayList<Integer> partitionsToVerify = new ArrayList<Integer>();
        for (Map.Entry<Integer, Long> entry : this._partitionsMarkedForVerification.entrySet()) {
            long timeMarked = this._clock.millis() - entry.getValue();
            if (timeMarked <= 600000L) continue;
            partitionsToVerify.add(entry.getKey());
        }
        return partitionsToVerify;
    }

    @VisibleForTesting
    void setClock(Clock clock) {
        this._clock = clock;
    }

    public void updateIngestionMetrics(long ingestionTimeMs, long firstStreamIngestionTimeMs, StreamPartitionMsgOffset msgOffset, StreamPartitionMsgOffset latestOffset, int partitionGroupId) {
        if (!this._isServerReadyToServeQueries.get().booleanValue() || this._realTimeTableDataManager.isShutDown()) {
            return;
        }
        this.updateIngestionDelay(ingestionTimeMs, firstStreamIngestionTimeMs, partitionGroupId);
        this.updateIngestionOffsets(msgOffset, latestOffset, partitionGroupId);
        this._partitionsMarkedForVerification.remove(partitionGroupId);
    }

    public void updateIngestionDelay(long ingestionTimeMs, long firstStreamIngestionTimeMs, int partitionGroupId) {
        if (ingestionTimeMs < 0L && firstStreamIngestionTimeMs < 0L) {
            return;
        }
        IngestionTimestamps previousMeasure = this._partitionToIngestionTimestampsMap.put(partitionGroupId, new IngestionTimestamps(ingestionTimeMs, firstStreamIngestionTimeMs));
        if (previousMeasure == null) {
            if (ingestionTimeMs >= 0L) {
                this._serverMetrics.setOrUpdatePartitionGauge(this._metricName, partitionGroupId, (AbstractMetrics.Gauge)ServerGauge.REALTIME_INGESTION_DELAY_MS, () -> this.getPartitionIngestionDelayMs(partitionGroupId));
            }
            if (firstStreamIngestionTimeMs >= 0L) {
                this._serverMetrics.setOrUpdatePartitionGauge(this._metricName, partitionGroupId, (AbstractMetrics.Gauge)ServerGauge.END_TO_END_REALTIME_INGESTION_DELAY_MS, () -> this.getPartitionEndToEndIngestionDelayMs(partitionGroupId));
            }
        }
    }

    public void updateIngestionOffsets(StreamPartitionMsgOffset currentOffset, StreamPartitionMsgOffset latestOffset, int partitionGroupId) {
        if (currentOffset == null) {
            return;
        }
        IngestionOffsets previousMeasure = this._partitionToOffsetMap.put(partitionGroupId, new IngestionOffsets(currentOffset, latestOffset));
        if (previousMeasure == null && currentOffset != null) {
            this._serverMetrics.setOrUpdatePartitionGauge(this._metricName, partitionGroupId, (AbstractMetrics.Gauge)ServerGauge.REALTIME_INGESTION_OFFSET_LAG, () -> this.getPartitionIngestionOffsetLag(partitionGroupId));
        }
    }

    public void stopTrackingPartitionIngestionDelay(int partitionGroupId) {
        this.removePartitionId(partitionGroupId);
    }

    public void timeoutInactivePartitions() {
        Set<Integer> partitionsHostedByThisServer;
        if (!this._isServerReadyToServeQueries.get().booleanValue()) {
            return;
        }
        List<Integer> partitionsToVerify = this.getPartitionsToBeVerified();
        if (partitionsToVerify.size() == 0) {
            return;
        }
        try {
            partitionsHostedByThisServer = this._realTimeTableDataManager.getHostedPartitionsGroupIds();
        }
        catch (Exception e) {
            _logger.error("Failed to get partitions hosted by this server, table={}, exception={}:{}", new Object[]{this._tableNameWithType, e.getClass(), e.getMessage()});
            return;
        }
        for (int partitionGroupId : partitionsToVerify) {
            if (partitionsHostedByThisServer.contains(partitionGroupId)) continue;
            this.removePartitionId(partitionGroupId);
        }
    }

    public void markPartitionForVerification(int partitionGroupId) {
        if (!this._isServerReadyToServeQueries.get().booleanValue()) {
            return;
        }
        this._partitionsMarkedForVerification.put(partitionGroupId, this._clock.millis());
    }

    public long getPartitionIngestionTimeMs(int partitionGroupId) {
        IngestionTimestamps currentMeasure = this._partitionToIngestionTimestampsMap.get(partitionGroupId);
        if (currentMeasure == null) {
            return Long.MIN_VALUE;
        }
        return currentMeasure._ingestionTimeMs;
    }

    public long getPartitionIngestionDelayMs(int partitionGroupId) {
        IngestionTimestamps currentMeasure = this._partitionToIngestionTimestampsMap.get(partitionGroupId);
        if (currentMeasure == null) {
            return 0L;
        }
        return this.getIngestionDelayMs(currentMeasure._ingestionTimeMs);
    }

    public long getPartitionIngestionOffsetLag(int partitionGroupId) {
        IngestionOffsets currentMeasure = this._partitionToOffsetMap.get(partitionGroupId);
        if (currentMeasure == null) {
            return 0L;
        }
        return this.getPartitionOffsetLag(currentMeasure);
    }

    public long getPartitionEndToEndIngestionDelayMs(int partitionGroupId) {
        IngestionTimestamps currentMeasure = this._partitionToIngestionTimestampsMap.get(partitionGroupId);
        if (currentMeasure == null) {
            return 0L;
        }
        return this.getIngestionDelayMs(currentMeasure._firstStreamIngestionTimeMs);
    }

    public void shutdown() {
        this._scheduledExecutor.shutdown();
        if (!this._isServerReadyToServeQueries.get().booleanValue()) {
            return;
        }
        for (Map.Entry<Integer, IngestionTimestamps> entry : this._partitionToIngestionTimestampsMap.entrySet()) {
            this.removePartitionId(entry.getKey());
        }
    }

    private static class IngestionOffsets {
        private final StreamPartitionMsgOffset _latestOffset;
        private final StreamPartitionMsgOffset _offset;

        IngestionOffsets(StreamPartitionMsgOffset offset, StreamPartitionMsgOffset latestOffset) {
            this._offset = offset;
            this._latestOffset = latestOffset;
        }
    }

    private static class IngestionTimestamps {
        private final long _firstStreamIngestionTimeMs;
        private final long _ingestionTimeMs;

        IngestionTimestamps(long ingestionTimesMs, long firstStreamIngestionTimeMs) {
            this._ingestionTimeMs = ingestionTimesMs;
            this._firstStreamIngestionTimeMs = firstStreamIngestionTimeMs;
        }
    }
}

