/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.deltalake.delete;

import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import io.delta.kernel.internal.deletionvectors.Base85Codec;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoInput;
import io.trino.filesystem.TrinoInputFile;
import io.trino.plugin.deltalake.DeltaLakeErrorCode;
import io.trino.plugin.deltalake.delete.RoaringBitmapArray;
import io.trino.plugin.deltalake.transactionlog.DeletionVectorEntry;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.OptionalInt;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;
import java.util.zip.CRC32;
import org.roaringbitmap.RoaringBitmap;

public final class DeletionVectors {
    private static final int PORTABLE_ROARING_BITMAP_MAGIC_NUMBER = 1681511377;
    private static final int MAGIC_NUMBER_BYTE_SIZE = 4;
    private static final int BIT_MAP_COUNT_BYTE_SIZE = 8;
    private static final int BIT_MAP_KEY_BYTE_SIZE = 4;
    private static final int FORMAT_VERSION_V1 = 1;
    private static final String UUID_MARKER = "u";
    private static final String PATH_MARKER = "p";
    private static final String INLINE_MARKER = "i";
    private static final CharMatcher ALPHANUMERIC = CharMatcher.inRange((char)'A', (char)'Z').or(CharMatcher.inRange((char)'a', (char)'z')).or(CharMatcher.inRange((char)'0', (char)'9')).precomputed();

    private DeletionVectors() {
    }

    public static RoaringBitmapArray readDeletionVectors(TrinoFileSystem fileSystem, Location location, DeletionVectorEntry deletionVector) throws IOException {
        if (deletionVector.storageType().equals(UUID_MARKER)) {
            TrinoInputFile inputFile = fileSystem.newInputFile(location.appendPath(DeletionVectors.toFileName(deletionVector.pathOrInlineDv())));
            ByteBuffer buffer = DeletionVectors.readDeletionVector(inputFile, deletionVector.offset().orElseThrow(), deletionVector.sizeInBytes());
            return DeletionVectors.deserializeDeletionVectors(buffer);
        }
        if (deletionVector.storageType().equals(INLINE_MARKER) || deletionVector.storageType().equals(PATH_MARKER)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Unsupported storage type for deletion vector: " + deletionVector.storageType());
        }
        throw new IllegalArgumentException("Unexpected storage type: " + deletionVector.storageType());
    }

    public static DeletionVectorEntry writeDeletionVectors(TrinoFileSystem fileSystem, Location location, RoaringBitmapArray deletedRows, int randomPrefixLength) throws IOException {
        OptionalInt offset;
        UUID uuid = UUID.randomUUID();
        Object pathOrInlineDv = Base85Codec.encodeUUID((UUID)uuid);
        String deletionVectorFilename = "deletion_vector_" + String.valueOf(uuid) + ".bin";
        if (randomPrefixLength > 0) {
            String randomPrefix = DeletionVectors.randomPrefix(randomPrefixLength);
            pathOrInlineDv = randomPrefix + (String)pathOrInlineDv;
            location = location.appendPath(randomPrefix);
        }
        int sizeInBytes = 16 + deletedRows.serializedSizeInBytes();
        long cardinality = deletedRows.cardinality();
        Preconditions.checkArgument((sizeInBytes > 0 ? 1 : 0) != 0, (String)"sizeInBytes must be positive: %s", (int)sizeInBytes);
        Preconditions.checkArgument((cardinality > 0L ? 1 : 0) != 0, (String)"cardinality must be positive: %s", (long)cardinality);
        byte[] data = DeletionVectors.serializeAsByteArray(deletedRows, sizeInBytes);
        try (DataOutputStream output = new DataOutputStream(fileSystem.newOutputFile(location.appendPath(deletionVectorFilename)).create());){
            output.writeByte(1);
            offset = OptionalInt.of(output.size());
            output.writeInt(sizeInBytes);
            output.write(data);
            output.writeInt(DeletionVectors.calculateChecksum(data));
        }
        return new DeletionVectorEntry(UUID_MARKER, (String)pathOrInlineDv, offset, sizeInBytes, cardinality);
    }

    private static String randomPrefix(int length) {
        String alphanumeric = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";
        StringBuilder prefix = new StringBuilder(length);
        for (int i = 0; i < length; ++i) {
            prefix.append(alphanumeric.charAt(ThreadLocalRandom.current().nextInt(alphanumeric.length())));
        }
        return prefix.toString();
    }

    private static byte[] serializeAsByteArray(RoaringBitmapArray bitmaps, int sizeInBytes) {
        ByteBuffer buffer = ByteBuffer.allocate(sizeInBytes).order(ByteOrder.LITTLE_ENDIAN);
        buffer.putInt(1681511377);
        buffer.putLong(bitmaps.length());
        int i = 0;
        while ((long)i < bitmaps.length()) {
            buffer.putInt(i);
            RoaringBitmap bitmap = bitmaps.get(i);
            bitmap.runOptimize();
            bitmap.serialize(buffer);
            ++i;
        }
        return buffer.array();
    }

    public static String toFileName(String pathOrInlineDv) {
        int randomPrefixLength = pathOrInlineDv.length() - 20;
        String randomPrefix = pathOrInlineDv.substring(0, randomPrefixLength);
        Preconditions.checkArgument((boolean)ALPHANUMERIC.matchesAllOf((CharSequence)randomPrefix), (String)"Random prefix must be alphanumeric: %s", (Object)randomPrefix);
        Object prefix = randomPrefix.isEmpty() ? "" : randomPrefix + "/";
        String encodedUuid = pathOrInlineDv.substring(randomPrefixLength);
        UUID uuid = Base85Codec.decodeUUID((String)encodedUuid);
        return "%sdeletion_vector_%s.bin".formatted(prefix, uuid);
    }

    public static ByteBuffer readDeletionVector(TrinoInputFile inputFile, int offset, int expectedSize) throws IOException {
        try (TrinoInput input = inputFile.newInput();){
            ByteBuffer buffer = input.readFully((long)offset, 4 + expectedSize + 4).toByteBuffer();
            int actualSize = buffer.getInt(0);
            if (actualSize != expectedSize) {
                throw new TrinoException((ErrorCodeSupplier)DeltaLakeErrorCode.DELTA_LAKE_INVALID_SCHEMA, "The size of deletion vector %s expects %s but got %s".formatted(inputFile.location(), expectedSize, actualSize));
            }
            int checksum = buffer.getInt(4 + expectedSize);
            if (DeletionVectors.calculateChecksum(buffer.array(), buffer.arrayOffset() + 4, expectedSize) != checksum) {
                throw new TrinoException((ErrorCodeSupplier)DeltaLakeErrorCode.DELTA_LAKE_INVALID_SCHEMA, "Checksum mismatch for deletion vector: " + String.valueOf(inputFile.location()));
            }
            ByteBuffer byteBuffer = buffer.slice(4, expectedSize).order(ByteOrder.LITTLE_ENDIAN);
            return byteBuffer;
        }
    }

    private static int calculateChecksum(byte[] data) {
        return DeletionVectors.calculateChecksum(data, 0, data.length);
    }

    private static int calculateChecksum(byte[] data, int offset, int length) {
        CRC32 crc = new CRC32();
        crc.update(data, offset, length);
        return (int)crc.getValue();
    }

    private static RoaringBitmapArray deserializeDeletionVectors(ByteBuffer buffer) throws IOException {
        Preconditions.checkArgument((buffer.order() == ByteOrder.LITTLE_ENDIAN ? 1 : 0) != 0, (String)"Byte order must be little endian: %s", (Object)buffer.order());
        int magicNumber = buffer.getInt();
        if (magicNumber == 1681511377) {
            int size = Math.toIntExact(buffer.getLong());
            RoaringBitmapArray bitmaps = new RoaringBitmapArray();
            for (int i = 0; i < size; ++i) {
                int key = buffer.getInt();
                Preconditions.checkArgument((key >= 0 ? 1 : 0) != 0, (String)"key must not be negative: %s", (int)key);
                RoaringBitmap bitmap = new RoaringBitmap();
                bitmap.deserialize(buffer);
                bitmap.stream().forEach(bitmaps::add);
                int consumedBytes = bitmap.serializedSizeInBytes();
                buffer.position(buffer.position() + consumedBytes);
            }
            return bitmaps;
        }
        throw new IllegalArgumentException("Unsupported magic number: " + magicNumber);
    }
}

