/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.nessie;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.catalog.Namespace;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.exceptions.AlreadyExistsException;
import org.apache.iceberg.exceptions.CommitFailedException;
import org.apache.iceberg.exceptions.CommitStateUnknownException;
import org.apache.iceberg.exceptions.NamespaceNotEmptyException;
import org.apache.iceberg.exceptions.NoSuchNamespaceException;
import org.apache.iceberg.exceptions.NoSuchTableException;
import org.apache.iceberg.exceptions.NoSuchViewException;
import org.apache.iceberg.nessie.NessieIcebergClient;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.Maps;
import org.apache.iceberg.view.ViewMetadata;
import org.projectnessie.client.api.GetContentBuilder;
import org.projectnessie.client.http.HttpClientException;
import org.projectnessie.error.NessieConflictException;
import org.projectnessie.error.NessieNotFoundException;
import org.projectnessie.error.NessieReferenceConflictException;
import org.projectnessie.error.ReferenceConflicts;
import org.projectnessie.model.CommitMeta;
import org.projectnessie.model.Conflict;
import org.projectnessie.model.Content;
import org.projectnessie.model.ContentKey;
import org.projectnessie.model.IcebergTable;
import org.projectnessie.model.ImmutableCommitMeta;
import org.projectnessie.model.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class NessieUtil {
    private static final Logger LOG = LoggerFactory.getLogger(NessieUtil.class);
    public static final String NESSIE_CONFIG_PREFIX = "nessie.";
    static final String APPLICATION_TYPE = "application-type";
    public static final String CLIENT_API_VERSION = "nessie.client-api-version";

    private NessieUtil() {
    }

    static TableIdentifier removeCatalogName(TableIdentifier to, String name) {
        String[] levels = to.namespace().levels();
        if (levels.length >= 2 && name.equalsIgnoreCase(to.namespace().level(0))) {
            Namespace trimmedNamespace = Namespace.of((String[])Arrays.copyOfRange(levels, 1, levels.length));
            return TableIdentifier.of((Namespace)trimmedNamespace, (String)to.name());
        }
        return to;
    }

    static ContentKey toKey(TableIdentifier tableIdentifier) {
        ArrayList identifiers = Lists.newArrayList();
        if (tableIdentifier.hasNamespace()) {
            identifiers.addAll(Arrays.asList(tableIdentifier.namespace().levels()));
        }
        identifiers.add(tableIdentifier.name());
        return ContentKey.of((List)identifiers);
    }

    static CommitMeta buildCommitMetadata(String commitMsg, Map<String, String> catalogOptions) {
        return NessieUtil.catalogOptions(CommitMeta.builder().message(commitMsg), catalogOptions).build();
    }

    static ImmutableCommitMeta.Builder catalogOptions(ImmutableCommitMeta.Builder commitMetaBuilder, Map<String, String> catalogOptions) {
        Preconditions.checkArgument((null != catalogOptions ? 1 : 0) != 0, (Object)"catalogOptions must not be null");
        commitMetaBuilder.author(NessieUtil.commitAuthor(catalogOptions));
        commitMetaBuilder.putProperties(APPLICATION_TYPE, "iceberg");
        if (catalogOptions.containsKey("app-id")) {
            commitMetaBuilder.putProperties("app-id", catalogOptions.get("app-id"));
        }
        return commitMetaBuilder;
    }

    @Nullable
    private static String commitAuthor(Map<String, String> catalogOptions) {
        return Optional.ofNullable(catalogOptions.get("user")).orElseGet(() -> System.getProperty("user.name"));
    }

    private static void checkAndUpdateGCProperties(TableMetadata tableMetadata, Map<String, String> updatedProperties, String identifier) {
        if (tableMetadata.propertyAsBoolean("nessie.gc.no-warning", false)) {
            return;
        }
        if (tableMetadata.propertyAsBoolean("gc.enabled", true) || tableMetadata.propertyAsBoolean("write.metadata.delete-after-commit.enabled", false)) {
            updatedProperties.put("nessie.gc.no-warning", "true");
            LOG.warn("The Iceberg property '{}' and/or '{}' is enabled on table '{}' in NessieCatalog. This will likely make data in other Nessie branches and tags and in earlier, historical Nessie commits inaccessible. The recommended setting for those properties is 'false'. Use the 'nessie-gc' tool for Nessie reference-aware garbage collection.", new Object[]{"gc.enabled", "write.metadata.delete-after-commit.enabled", identifier});
        }
    }

    public static TableMetadata updateTableMetadataWithNessieSpecificProperties(TableMetadata tableMetadata, String metadataLocation, IcebergTable table, String identifier, Reference reference) {
        HashMap newProperties = Maps.newHashMap((Map)tableMetadata.properties());
        newProperties.put("nessie.commit.id", reference.getHash());
        NessieUtil.checkAndUpdateGCProperties(tableMetadata, newProperties, identifier);
        TableMetadata.Builder builder = TableMetadata.buildFrom((TableMetadata)tableMetadata).setPreviousFileLocation(null).setCurrentSchema(table.getSchemaId()).setDefaultSortOrder(table.getSortOrderId()).setDefaultPartitionSpec(table.getSpecId()).withMetadataLocation(metadataLocation).setProperties((Map)newProperties);
        if (table.getSnapshotId() != -1L) {
            builder.setBranchSnapshot(table.getSnapshotId(), "main");
        }
        LOG.info("loadTableMetadata for '{}' from location '{}' at '{}'", new Object[]{identifier, metadataLocation, reference});
        return builder.discardChanges().build();
    }

    public static Optional<Conflict> extractSingleConflict(NessieReferenceConflictException ex, Collection<Conflict.ConflictType> handledConflictTypes) {
        ReferenceConflicts referenceConflicts = ex.getErrorDetails();
        if (referenceConflicts == null) {
            return Optional.empty();
        }
        List conflicts = referenceConflicts.conflicts().stream().filter(c -> handledConflictTypes.contains(c.conflictType())).collect(Collectors.toList());
        if (conflicts.size() != 1) {
            return Optional.empty();
        }
        Conflict conflict = (Conflict)conflicts.get(0);
        return Optional.of(conflict);
    }

    public static ViewMetadata loadViewMetadata(ViewMetadata metadata, String metadataLocation, Reference reference) {
        HashMap newProperties = Maps.newHashMap((Map)metadata.properties());
        newProperties.put("nessie.commit.id", reference.getHash());
        return ViewMetadata.buildFrom((ViewMetadata)ViewMetadata.buildFrom((ViewMetadata)metadata).setProperties((Map)newProperties).build()).setMetadataLocation(metadataLocation).build();
    }

    static Optional<RuntimeException> handleExceptionsForCommits(Exception exception, String refName, Content.Type type) {
        if (exception instanceof NessieConflictException) {
            Optional<RuntimeException> specializedException;
            if (exception instanceof NessieReferenceConflictException && (specializedException = NessieUtil.maybeUseSpecializedException((NessieReferenceConflictException)((Object)exception), type)).isPresent()) {
                return specializedException;
            }
            return Optional.of(new CommitFailedException((Throwable)exception, "Cannot commit: Reference hash is out of date. Update the reference '%s' and try again", new Object[]{refName}));
        }
        if (exception instanceof NessieNotFoundException) {
            return Optional.of(new RuntimeException(String.format("Cannot commit: Reference '%s' no longer exists", refName), exception));
        }
        if (exception instanceof HttpClientException) {
            return Optional.of(new CommitStateUnknownException((Throwable)exception));
        }
        return Optional.empty();
    }

    static Optional<RuntimeException> handleBadRequestForCommit(NessieIcebergClient client, ContentKey key, Content.Type type) {
        Content.Type anotherType = type == Content.Type.ICEBERG_TABLE ? Content.Type.ICEBERG_VIEW : Content.Type.ICEBERG_TABLE;
        try {
            Content content = (Content)((GetContentBuilder)client.getApi().getContent().key(key).reference(client.getReference())).get().get(key);
            if (content != null) {
                if (content.getType().equals(anotherType)) {
                    return Optional.of(new AlreadyExistsException("%s with same name already exists: %s in %s", new Object[]{NessieUtil.contentTypeString(anotherType), key, client.getReference()}));
                }
                if (!content.getType().equals(type)) {
                    return Optional.of(new AlreadyExistsException("Another content with same name already exists: %s in %s", new Object[]{key, client.getReference()}));
                }
            }
        }
        catch (NessieNotFoundException e) {
            return Optional.of(new RuntimeException(e));
        }
        return Optional.empty();
    }

    private static Optional<RuntimeException> maybeUseSpecializedException(NessieReferenceConflictException ex, Content.Type type) {
        String contentType = NessieUtil.contentTypeString(type);
        Optional<Conflict> singleConflict = NessieUtil.extractSingleConflict(ex, EnumSet.of(Conflict.ConflictType.NAMESPACE_ABSENT, Conflict.ConflictType.NAMESPACE_NOT_EMPTY, Conflict.ConflictType.KEY_DOES_NOT_EXIST, Conflict.ConflictType.KEY_EXISTS));
        if (!singleConflict.isPresent()) {
            return Optional.empty();
        }
        Conflict conflict = singleConflict.get();
        switch (conflict.conflictType()) {
            case NAMESPACE_ABSENT: {
                return Optional.of(new NoSuchNamespaceException((Throwable)ex, "Namespace does not exist: %s", new Object[]{conflict.key()}));
            }
            case NAMESPACE_NOT_EMPTY: {
                return Optional.of(new NamespaceNotEmptyException((Throwable)ex, "Namespace not empty: %s", new Object[]{conflict.key()}));
            }
            case KEY_DOES_NOT_EXIST: {
                if (type == Content.Type.ICEBERG_VIEW) {
                    return Optional.of(new NoSuchViewException((Throwable)ex, "%s does not exist: %s", new Object[]{contentType, conflict.key()}));
                }
                return Optional.of(new NoSuchTableException((Throwable)ex, "%s does not exist: %s", new Object[]{contentType, conflict.key()}));
            }
            case KEY_EXISTS: {
                return Optional.of(new AlreadyExistsException((Throwable)ex, "%s already exists: %s", new Object[]{contentType, conflict.key()}));
            }
        }
        return Optional.empty();
    }

    static String contentTypeString(Content.Type type) {
        if (type.equals(Content.Type.ICEBERG_VIEW)) {
            return "View";
        }
        if (type.equals(Content.Type.ICEBERG_TABLE)) {
            return "Table";
        }
        if (type.equals(Content.Type.NAMESPACE)) {
            return "Namespace";
        }
        throw new IllegalArgumentException("Unsupported Nessie content type " + type.name());
    }
}

