/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.bigquery;

import com.google.cloud.bigquery.Field;
import com.google.cloud.bigquery.FieldList;
import com.google.cloud.bigquery.LegacySQLTypeName;
import com.google.cloud.bigquery.StandardSQLTypeName;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.inject.Inject;
import io.airlift.slice.Slice;
import io.trino.plugin.bigquery.BigQueryColumnHandle;
import io.trino.plugin.bigquery.ColumnMapping;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.type.ArrayType;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.Int128;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.LongTimestampWithTimeZone;
import io.trino.spi.type.RowType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimeWithTimeZoneType;
import io.trino.spi.type.TimeZoneKey;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.TypeManager;
import io.trino.spi.type.TypeSignature;
import io.trino.spi.type.TypeSignatureParameter;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import jakarta.annotation.Nullable;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Base64;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

public final class BigQueryTypeManager {
    private static final int[] NANO_FACTOR = new int[]{-1, 100000000, 10000000, 1000000, 100000, 10000, 1000, 100, 10, 1};
    private static final DateTimeFormatter TIME_FORMATTER = DateTimeFormatter.ofPattern("''HH:mm:ss.SSSSSS''");
    private static final DateTimeFormatter DATETIME_FORMATTER = DateTimeFormatter.ofPattern("uuuu-MM-dd HH:mm:ss.SSSSSS").withZone(ZoneOffset.UTC);
    private final Type jsonType;

    @Inject
    public BigQueryTypeManager(TypeManager typeManager) {
        this.jsonType = Objects.requireNonNull(typeManager, "typeManager is null").getType(new TypeSignature("json", new TypeSignatureParameter[0]));
    }

    private RowType.Field toRawTypeField(String name, Field field) {
        Type trinoType = this.convertToTrinoType(field).orElseThrow(() -> new IllegalArgumentException("Unsupported column " + String.valueOf(field))).type();
        return RowType.field((String)name, (Type)(field.getMode() == Field.Mode.REPEATED ? new ArrayType(trinoType) : trinoType));
    }

    @VisibleForTesting
    public static LocalDateTime toLocalDateTime(String datetime) {
        int dotPosition = datetime.indexOf(46);
        if (dotPosition == -1) {
            return LocalDateTime.from(DateTimeFormatter.ISO_LOCAL_DATE_TIME.parse(datetime));
        }
        LocalDateTime result = LocalDateTime.from(DateTimeFormatter.ISO_LOCAL_DATE_TIME.parse(datetime.substring(0, dotPosition)));
        String nanosStr = datetime.substring(dotPosition + 1);
        int nanoOfSecond = Integer.parseInt(nanosStr) * NANO_FACTOR[nanosStr.length()];
        return result.withNano(nanoOfSecond);
    }

    public static long toTrinoTimestamp(String datetime) {
        Instant instant = BigQueryTypeManager.toLocalDateTime(datetime).toInstant(ZoneOffset.UTC);
        return instant.getEpochSecond() * 1000000L + (long)(instant.getNano() / 1000);
    }

    private static String floatToStringConverter(Object value) {
        return String.format("CAST('%s' AS float64)", value);
    }

    private static String simpleToStringConverter(Object value) {
        return String.valueOf(value);
    }

    @VisibleForTesting
    public static String dateToStringConverter(Object value) {
        LocalDate date = LocalDate.ofEpochDay((Long)value);
        return "'" + String.valueOf(date) + "'";
    }

    public static String datetimeToStringConverter(Object value) {
        long epochMicros = (Long)value;
        long epochSeconds = Math.floorDiv(epochMicros, 1000000);
        int nanoAdjustment = Math.floorMod(epochMicros, 1000000) * 1000;
        return BigQueryTypeManager.formatTimestamp(epochSeconds, nanoAdjustment, ZoneOffset.UTC);
    }

    @VisibleForTesting
    public static String timeToStringConverter(Object value) {
        long time = (Long)value;
        Verify.verify((0L <= time ? 1 : 0) != 0, (String)"Invalid time value: %s", (long)time);
        long epochSeconds = time / 1000000000000L;
        long nanoAdjustment = time % 1000000000000L / 1000L;
        return TIME_FORMATTER.format(BigQueryTypeManager.toZonedDateTime(epochSeconds, nanoAdjustment, ZoneOffset.UTC));
    }

    public static String timestampToStringConverter(Object value) {
        LongTimestampWithTimeZone timestamp = (LongTimestampWithTimeZone)value;
        long epochMillis = timestamp.getEpochMillis();
        long epochSeconds = Math.floorDiv(epochMillis, 1000);
        int nanoAdjustment = Math.floorMod(epochMillis, 1000) * 1000000 + timestamp.getPicosOfMilli() / 1000;
        ZoneId zoneId = TimeZoneKey.getTimeZoneKey((short)timestamp.getTimeZoneKey()).getZoneId();
        return BigQueryTypeManager.formatTimestamp(epochSeconds, nanoAdjustment, zoneId);
    }

    private static String formatTimestamp(long epochSeconds, long nanoAdjustment, ZoneId zoneId) {
        return DATETIME_FORMATTER.format(BigQueryTypeManager.toZonedDateTime(epochSeconds, nanoAdjustment, zoneId));
    }

    public static ZonedDateTime toZonedDateTime(long epochSeconds, long nanoAdjustment, ZoneId zoneId) {
        Instant instant = Instant.ofEpochSecond(epochSeconds, nanoAdjustment);
        return ZonedDateTime.ofInstant(instant, zoneId);
    }

    static String stringToStringConverter(Object value) {
        Slice slice = (Slice)value;
        return "'%s'".formatted(slice.toStringUtf8().replace("\\", "\\\\").replace("\n", "\\n").replace("'", "\\'"));
    }

    static String numericToStringConverter(Object value) {
        return Decimals.toString((Int128)((Int128)value), (int)9);
    }

    static String bytesToStringConverter(Object value) {
        Slice slice = (Slice)value;
        return String.format("FROM_BASE64('%s')", Base64.getEncoder().encodeToString(slice.getBytes()));
    }

    public Field toField(String name, Type type, @Nullable String comment) {
        if (type instanceof ArrayType) {
            Type elementType = ((ArrayType)type).getElementType();
            return this.toInnerField(name, elementType, true, comment);
        }
        return this.toInnerField(name, type, false, comment);
    }

    private Field toInnerField(String name, Type type, boolean repeated, @Nullable String comment) {
        Field.Builder builder = type instanceof RowType ? Field.newBuilder((String)name, (StandardSQLTypeName)StandardSQLTypeName.STRUCT, (FieldList)this.toFieldList((RowType)type)).setDescription(comment) : Field.newBuilder((String)name, (StandardSQLTypeName)this.toStandardSqlTypeName(type), (Field[])new Field[0]).setDescription(comment);
        if (repeated) {
            builder = builder.setMode(Field.Mode.REPEATED);
        }
        return builder.build();
    }

    private FieldList toFieldList(RowType rowType) {
        ImmutableList.Builder fields = ImmutableList.builder();
        for (RowType.Field field : rowType.getFields()) {
            String fieldName = (String)field.getName().orElseThrow(() -> new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "ROW type does not have field names declared: " + String.valueOf(rowType)));
            fields.add((Object)this.toField(fieldName, field.getType(), null));
        }
        return FieldList.of((Iterable)fields.build());
    }

    StandardSQLTypeName toStandardSqlTypeName(Type type) {
        if (type == BooleanType.BOOLEAN) {
            return StandardSQLTypeName.BOOL;
        }
        if (type == TinyintType.TINYINT || type == SmallintType.SMALLINT || type == IntegerType.INTEGER || type == BigintType.BIGINT) {
            return StandardSQLTypeName.INT64;
        }
        if (type == DoubleType.DOUBLE) {
            return StandardSQLTypeName.FLOAT64;
        }
        if (type instanceof DecimalType) {
            return StandardSQLTypeName.NUMERIC;
        }
        if (type == DateType.DATE) {
            return StandardSQLTypeName.DATE;
        }
        if (type == TimeWithTimeZoneType.createTimeWithTimeZoneType((int)3)) {
            return StandardSQLTypeName.TIME;
        }
        if (type == TimestampType.TIMESTAMP_MICROS) {
            return StandardSQLTypeName.DATETIME;
        }
        if (type == TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS) {
            return StandardSQLTypeName.TIMESTAMP;
        }
        if (type instanceof VarcharType) {
            return StandardSQLTypeName.STRING;
        }
        if (type == VarbinaryType.VARBINARY) {
            return StandardSQLTypeName.BYTES;
        }
        if (type instanceof ArrayType) {
            return StandardSQLTypeName.ARRAY;
        }
        if (type instanceof RowType) {
            return StandardSQLTypeName.STRUCT;
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Unsupported column type: " + type.getDisplayName());
    }

    public static String convertToString(Type type, StandardSQLTypeName bigqueryType, Object value) {
        switch (bigqueryType) {
            case BOOL: {
                return BigQueryTypeManager.simpleToStringConverter(value);
            }
            case BYTES: {
                return BigQueryTypeManager.bytesToStringConverter(value);
            }
            case DATE: {
                return BigQueryTypeManager.dateToStringConverter(value);
            }
            case DATETIME: {
                return "'%s'".formatted(BigQueryTypeManager.datetimeToStringConverter(value));
            }
            case FLOAT64: {
                return BigQueryTypeManager.floatToStringConverter(value);
            }
            case INT64: {
                return BigQueryTypeManager.simpleToStringConverter(value);
            }
            case NUMERIC: 
            case BIGNUMERIC: {
                String bigqueryTypeName = bigqueryType.name();
                DecimalType decimalType = (DecimalType)type;
                if (decimalType.isShort()) {
                    return String.format("%s '%s'", bigqueryTypeName, Decimals.toString((long)((Long)value), (int)((DecimalType)type).getScale()));
                }
                return String.format("%s '%s'", bigqueryTypeName, Decimals.toString((Int128)((Int128)value), (int)((DecimalType)type).getScale()));
            }
            case STRING: {
                return BigQueryTypeManager.stringToStringConverter(value);
            }
            case TIME: {
                return BigQueryTypeManager.timeToStringConverter(value);
            }
            case TIMESTAMP: {
                return "'%s'".formatted(BigQueryTypeManager.timestampToStringConverter(value));
            }
        }
        throw new IllegalArgumentException("Unsupported type: " + String.valueOf(bigqueryType));
    }

    public Optional<ColumnMapping> toTrinoType(Field field) {
        return this.convertToTrinoType(field).map(columnMapping -> field.getMode() == Field.Mode.REPEATED ? new ColumnMapping((Type)new ArrayType(columnMapping.type()), false) : columnMapping);
    }

    private Optional<ColumnMapping> convertToTrinoType(Field field) {
        switch (field.getType().getStandardType()) {
            case BOOL: {
                return Optional.of(new ColumnMapping((Type)BooleanType.BOOLEAN, true));
            }
            case INT64: {
                return Optional.of(new ColumnMapping((Type)BigintType.BIGINT, true));
            }
            case FLOAT64: {
                return Optional.of(new ColumnMapping((Type)DoubleType.DOUBLE, true));
            }
            case NUMERIC: 
            case BIGNUMERIC: {
                Long precision = field.getPrecision();
                Long scale = field.getScale();
                if (precision != null && scale != null) {
                    return Optional.of(new ColumnMapping((Type)DecimalType.createDecimalType((int)Math.toIntExact(precision), (int)Math.toIntExact(scale)), true));
                }
                if (precision != null) {
                    return Optional.of(new ColumnMapping((Type)DecimalType.createDecimalType((int)Math.toIntExact(precision)), true));
                }
                return Optional.of(new ColumnMapping((Type)DecimalType.createDecimalType((int)38, (int)9), true));
            }
            case STRING: {
                return Optional.of(new ColumnMapping((Type)VarcharType.createUnboundedVarcharType(), true));
            }
            case BYTES: {
                return Optional.of(new ColumnMapping((Type)VarbinaryType.VARBINARY, true));
            }
            case DATE: {
                return Optional.of(new ColumnMapping((Type)DateType.DATE, true));
            }
            case DATETIME: {
                return Optional.of(new ColumnMapping((Type)TimestampType.TIMESTAMP_MICROS, true));
            }
            case TIME: {
                return Optional.of(new ColumnMapping((Type)TimeType.TIME_MICROS, true));
            }
            case TIMESTAMP: {
                return Optional.of(new ColumnMapping((Type)TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS, true));
            }
            case GEOGRAPHY: {
                return Optional.of(new ColumnMapping((Type)VarcharType.VARCHAR, false));
            }
            case JSON: {
                return Optional.of(new ColumnMapping(this.jsonType, false));
            }
            case STRUCT: {
                FieldList subTypes = field.getSubFields();
                Preconditions.checkArgument((!subTypes.isEmpty() ? 1 : 0) != 0, (Object)"a record or struct must have sub-fields");
                List fields = subTypes.stream().map(subField -> this.toRawTypeField(subField.getName(), (Field)subField)).collect(Collectors.toList());
                RowType rowType = RowType.from(fields);
                return Optional.of(new ColumnMapping((Type)rowType, false));
            }
        }
        return Optional.empty();
    }

    public BigQueryColumnHandle toColumnHandle(Field field, boolean useStorageApi) {
        FieldList subFields = field.getSubFields();
        List<BigQueryColumnHandle> subColumns = subFields == null ? Collections.emptyList() : subFields.stream().filter(column -> this.isSupportedType((Field)column, useStorageApi)).map(column -> this.toColumnHandle((Field)column, useStorageApi)).collect(Collectors.toList());
        ColumnMapping columnMapping = this.toTrinoType(field).orElseThrow(() -> new IllegalArgumentException("Unsupported type: " + String.valueOf(field)));
        return new BigQueryColumnHandle(field.getName(), (List<String>)ImmutableList.of(), columnMapping.type(), field.getType().getStandardType(), columnMapping.isPushdownSupported(), BigQueryTypeManager.getMode(field), subColumns, field.getDescription(), false);
    }

    public boolean isSupportedType(Field field, boolean useStorageApi) {
        LegacySQLTypeName type = field.getType();
        if (type == LegacySQLTypeName.BIGNUMERIC) {
            if (field.getPrecision() == null && field.getScale() == null) {
                return false;
            }
            if (field.getPrecision() != null && field.getPrecision() > 38L) {
                return false;
            }
        }
        if (!useStorageApi && type == LegacySQLTypeName.TIMESTAMP) {
            return false;
        }
        return this.toTrinoType(field).isPresent();
    }

    public boolean isJsonType(Type type) {
        return type.equals((Object)this.jsonType);
    }

    private static Field.Mode getMode(Field field) {
        return (Field.Mode)MoreObjects.firstNonNull((Object)field.getMode(), (Object)Field.Mode.NULLABLE);
    }
}

