/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.log;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.errorprone.annotations.ThreadSafe;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import io.airlift.log.LogFileName;
import io.airlift.units.DataSize;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Objects;
import java.util.Optional;
import java.util.PriorityQueue;
import java.util.Set;
import java.util.stream.Stream;

@ThreadSafe
final class LogHistoryManager {
    private final Path masterLogFile;
    private final long maxTotalSize;
    @GuardedBy(value="this")
    private long totalSize;
    @GuardedBy(value="this")
    private final PriorityQueue<LogFile> files = new PriorityQueue();

    public LogHistoryManager(Path masterLogFile, DataSize maxTotalSize) {
        Objects.requireNonNull(masterLogFile, "masterLogFile is null");
        Objects.requireNonNull(maxTotalSize, "maxTotalSize is null");
        this.masterLogFile = masterLogFile;
        this.maxTotalSize = maxTotalSize.toBytes();
        try (Stream<Path> paths = Files.list(masterLogFile.getParent());){
            paths.map(this::createLogFile).filter(Optional::isPresent).map(Optional::get).forEach(this.files::add);
            this.totalSize = this.files.stream().mapToLong(LogFile::getSize).sum();
        }
        catch (IOException e) {
            throw new UncheckedIOException("Unable to list existing history log files for " + String.valueOf(masterLogFile), e);
        }
        this.pruneLogFilesIfNecessary(0L);
    }

    public synchronized long getTotalSize() {
        return this.totalSize;
    }

    public synchronized Set<LogFileName> getFiles() {
        return (Set)this.files.stream().map(LogFile::getLogFileName).collect(ImmutableSet.toImmutableSet());
    }

    public synchronized void pruneLogFilesIfNecessary(long otherDataSize) {
        LogFile logFile;
        while (this.totalSize + otherDataSize > this.maxTotalSize && (logFile = this.files.poll()) != null) {
            this.totalSize -= logFile.getSize();
            try {
                Files.deleteIfExists(logFile.getPath());
            }
            catch (IOException iOException) {}
        }
    }

    private Optional<LogFile> createLogFile(Path path) {
        BasicFileAttributes fileAttributes;
        Optional<LogFileName> logFileName = LogFileName.parseHistoryLogFileName(this.masterLogFile.getFileName().toString(), path.getFileName().toString());
        if (!logFileName.isPresent()) {
            return Optional.empty();
        }
        try {
            fileAttributes = Files.readAttributes(path, BasicFileAttributes.class, new LinkOption[0]);
        }
        catch (IOException e) {
            return Optional.empty();
        }
        return Optional.of(new LogFile(path, logFileName.get(), fileAttributes.size()));
    }

    public synchronized void addFile(Path file, LogFileName fileName, long size) {
        this.files.add(new LogFile(file, fileName, size));
        this.totalSize += size;
    }

    public synchronized boolean removeFile(Path path) {
        return this.files.stream().filter(file -> file.getPath().equals(path)).findFirst().map(this::removeFile).orElse(false);
    }

    private synchronized boolean removeFile(LogFile file) {
        if (!this.files.remove(file)) {
            return false;
        }
        this.totalSize -= file.getSize();
        return true;
    }

    private static class LogFile
    implements Comparable<LogFile> {
        private final Path path;
        private final LogFileName logFileName;
        private final long size;

        public LogFile(Path path, LogFileName logFileName, long size) {
            this.path = Objects.requireNonNull(path, "path is null");
            this.logFileName = Objects.requireNonNull(logFileName, "logFileName is null");
            Preconditions.checkArgument((size >= 0L ? 1 : 0) != 0, (Object)"size is negative");
            this.size = size;
        }

        public Path getPath() {
            return this.path;
        }

        public LogFileName getLogFileName() {
            return this.logFileName;
        }

        public long getSize() {
            return this.size;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            LogFile logFile = (LogFile)o;
            return this.logFileName.equals(logFile.logFileName);
        }

        public int hashCode() {
            return Objects.hash(this.logFileName);
        }

        @Override
        public int compareTo(LogFile o) {
            return this.logFileName.compareTo(o.logFileName);
        }

        public String toString() {
            return this.logFileName.toString();
        }
    }
}

