/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.common.utils.tls;

import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import io.netty.handler.ssl.ClientAuth;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import io.netty.handler.ssl.SslProvider;
import io.netty.handler.ssl.util.InsecureTrustManagerFactory;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.security.KeyStore;
import java.security.SecureRandom;
import java.util.concurrent.atomic.AtomicReference;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;
import nl.altindag.ssl.SSLFactory;
import nl.altindag.ssl.exception.GenericSSLContextException;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.ssl.SSLContexts;
import org.apache.pinot.common.config.TlsConfig;
import org.apache.pinot.common.utils.tls.PinotInsecureMode;
import org.apache.pinot.common.utils.tls.RenewableTlsUtils;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class TlsUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(TlsUtils.class);
    private static final String CLIENT_AUTH_ENABLED = "client.auth.enabled";
    private static final String KEYSTORE_TYPE = "keystore.type";
    private static final String KEYSTORE_PATH = "keystore.path";
    private static final String KEYSTORE_PASSWORD = "keystore.password";
    private static final String TRUSTSTORE_TYPE = "truststore.type";
    private static final String TRUSTSTORE_PATH = "truststore.path";
    private static final String TRUSTSTORE_PASSWORD = "truststore.password";
    private static final String SSL_PROVIDER = "ssl.provider";
    private static final String FILE_SCHEME = "file";
    private static final String FILE_SCHEME_PREFIX = "file://";
    private static final String FILE_SCHEME_PREFIX_WITHOUT_SLASH = "file:";
    private static final String INSECURE = "insecure";
    private static final AtomicReference<SSLContext> SSL_CONTEXT_REF = new AtomicReference();

    private TlsUtils() {
    }

    public static TlsConfig extractTlsConfig(PinotConfiguration pinotConfig, String namespace) {
        return TlsUtils.extractTlsConfig(pinotConfig, namespace, new TlsConfig());
    }

    public static TlsConfig extractTlsConfig(PinotConfiguration pinotConfig, String namespace, TlsConfig defaultConfig) {
        TlsConfig tlsConfig = new TlsConfig(defaultConfig);
        tlsConfig.setClientAuthEnabled(pinotConfig.getProperty(TlsUtils.key(namespace, CLIENT_AUTH_ENABLED), defaultConfig.isClientAuthEnabled()));
        tlsConfig.setKeyStoreType(pinotConfig.getProperty(TlsUtils.key(namespace, KEYSTORE_TYPE), defaultConfig.getKeyStoreType()));
        tlsConfig.setKeyStorePath(pinotConfig.getProperty(TlsUtils.key(namespace, KEYSTORE_PATH), defaultConfig.getKeyStorePath()));
        tlsConfig.setKeyStorePassword(pinotConfig.getProperty(TlsUtils.key(namespace, KEYSTORE_PASSWORD), defaultConfig.getKeyStorePassword()));
        tlsConfig.setTrustStoreType(pinotConfig.getProperty(TlsUtils.key(namespace, TRUSTSTORE_TYPE), defaultConfig.getTrustStoreType()));
        tlsConfig.setTrustStorePath(pinotConfig.getProperty(TlsUtils.key(namespace, TRUSTSTORE_PATH), defaultConfig.getTrustStorePath()));
        tlsConfig.setTrustStorePassword(pinotConfig.getProperty(TlsUtils.key(namespace, TRUSTSTORE_PASSWORD), defaultConfig.getTrustStorePassword()));
        tlsConfig.setSslProvider(pinotConfig.getProperty(TlsUtils.key(namespace, SSL_PROVIDER), defaultConfig.getSslProvider()));
        tlsConfig.setInsecure(pinotConfig.getProperty(TlsUtils.key(namespace, INSECURE), defaultConfig.isInsecure()));
        return tlsConfig;
    }

    public static KeyManagerFactory createKeyManagerFactory(TlsConfig tlsConfig) {
        return TlsUtils.createKeyManagerFactory(tlsConfig.getKeyStorePath(), tlsConfig.getKeyStorePassword(), tlsConfig.getKeyStoreType());
    }

    public static KeyManagerFactory createKeyManagerFactory(String keyStorePath, String keyStorePassword, String keyStoreType) {
        Preconditions.checkNotNull((Object)keyStorePath, (Object)"key store path must not be null");
        Preconditions.checkNotNull((Object)keyStorePassword, (Object)"key store password must not be null");
        try {
            KeyStore keyStore = KeyStore.getInstance(keyStoreType);
            try (InputStream is = TlsUtils.makeKeyOrTrustStoreUrl(keyStorePath).openStream();){
                keyStore.load(is, keyStorePassword.toCharArray());
            }
            KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            keyManagerFactory.init(keyStore, keyStorePassword.toCharArray());
            return keyManagerFactory;
        }
        catch (Exception e) {
            throw new RuntimeException(String.format("Could not create key manager factory '%s'", keyStorePath), e);
        }
    }

    public static TrustManagerFactory createTrustManagerFactory(TlsConfig tlsConfig) {
        if (tlsConfig.isInsecure()) {
            return InsecureTrustManagerFactory.INSTANCE;
        }
        return TlsUtils.createTrustManagerFactory(tlsConfig.getTrustStorePath(), tlsConfig.getTrustStorePassword(), tlsConfig.getTrustStoreType());
    }

    public static TrustManagerFactory createTrustManagerFactory(String trustStorePath, String trustStorePassword, String trustStoreType) {
        Preconditions.checkNotNull((Object)trustStorePath, (Object)"trust store path must not be null");
        Preconditions.checkNotNull((Object)trustStorePassword, (Object)"trust store password must not be null");
        try {
            KeyStore keyStore = KeyStore.getInstance(trustStoreType);
            try (InputStream is = TlsUtils.makeKeyOrTrustStoreUrl(trustStorePath).openStream();){
                keyStore.load(is, trustStorePassword.toCharArray());
            }
            TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            trustManagerFactory.init(keyStore);
            return trustManagerFactory;
        }
        catch (Exception e) {
            throw new RuntimeException(String.format("Could not create trust manager factory '%s'", trustStorePath), e);
        }
    }

    public static void installDefaultSSLSocketFactory(TlsConfig tlsConfig) {
        TlsUtils.installDefaultSSLSocketFactory(tlsConfig.getKeyStoreType(), tlsConfig.getKeyStorePath(), tlsConfig.getKeyStorePassword(), tlsConfig.getTrustStoreType(), tlsConfig.getTrustStorePath(), tlsConfig.getTrustStorePassword());
    }

    public static void installDefaultSSLSocketFactory(String keyStoreType, String keyStorePath, String keyStorePassword, String trustStoreType, String trustStorePath, String trustStorePassword) {
        try {
            SecureRandom secureRandom = new SecureRandom();
            SSLFactory sslFactory = RenewableTlsUtils.createSSLFactory(keyStoreType, keyStorePath, keyStorePassword, trustStoreType, trustStorePath, trustStorePassword, "SSL", secureRandom, true, false);
            if (TlsUtils.isKeyOrTrustStorePathNullOrHasFileScheme(keyStorePath) && TlsUtils.isKeyOrTrustStorePathNullOrHasFileScheme(trustStorePath)) {
                RenewableTlsUtils.enableAutoRenewalFromFileStoreForSSLFactory(sslFactory, keyStoreType, keyStorePath, keyStorePassword, trustStoreType, trustStorePath, trustStorePassword, "SSL", secureRandom, (Supplier<Boolean>)((Supplier)PinotInsecureMode::isPinotInInsecureMode));
            }
            HttpsURLConnection.setDefaultSSLSocketFactory(sslFactory.getSslSocketFactory());
            TlsUtils.setSslContext(sslFactory.getSslContext());
        }
        catch (GenericSSLContextException e) {
            throw new IllegalStateException("Could not initialize SSL support", e);
        }
    }

    private static String key(String namespace, String suffix) {
        return namespace + "." + suffix;
    }

    public static URL makeKeyOrTrustStoreUrl(String storePath) throws URISyntaxException, MalformedURLException {
        URI inputUri = new URI(storePath);
        if (StringUtils.isBlank((CharSequence)inputUri.getScheme())) {
            if (storePath.startsWith("/")) {
                return new URL(FILE_SCHEME_PREFIX + storePath);
            }
            return new URL("file://./" + storePath);
        }
        return inputUri.toURL();
    }

    public static SSLContext getSslContext() {
        return SSLContextHolder.SSL_CONTEXT;
    }

    public static void setSslContext(SSLContext sslContext) {
        if (!SSL_CONTEXT_REF.compareAndSet(null, sslContext)) {
            LOGGER.warn("SSL Context has already been set.");
        }
    }

    public static SslContext buildClientContext(TlsConfig tlsConfig) {
        SSLFactory sslFactory = RenewableTlsUtils.createSSLFactoryAndEnableAutoRenewalWhenUsingFileStores(tlsConfig, (Supplier<Boolean>)((Supplier)PinotInsecureMode::isPinotInInsecureMode));
        SslContextBuilder sslContextBuilder = SslContextBuilder.forClient().sslProvider(SslProvider.valueOf((String)tlsConfig.getSslProvider()));
        sslFactory.getKeyManagerFactory().ifPresent(arg_0 -> ((SslContextBuilder)sslContextBuilder).keyManager(arg_0));
        sslFactory.getTrustManagerFactory().ifPresent(arg_0 -> ((SslContextBuilder)sslContextBuilder).trustManager(arg_0));
        try {
            return sslContextBuilder.build();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static SslContext buildServerContext(TlsConfig tlsConfig) {
        if (tlsConfig.getKeyStorePath() == null) {
            throw new IllegalArgumentException("Must provide key store path for secured server");
        }
        SSLFactory sslFactory = RenewableTlsUtils.createSSLFactoryAndEnableAutoRenewalWhenUsingFileStores(tlsConfig, (Supplier<Boolean>)((Supplier)PinotInsecureMode::isPinotInInsecureMode));
        SslContextBuilder sslContextBuilder = SslContextBuilder.forServer((KeyManagerFactory)((KeyManagerFactory)sslFactory.getKeyManagerFactory().get())).sslProvider(SslProvider.valueOf((String)tlsConfig.getSslProvider()));
        sslFactory.getTrustManagerFactory().ifPresent(arg_0 -> ((SslContextBuilder)sslContextBuilder).trustManager(arg_0));
        if (tlsConfig.isClientAuthEnabled()) {
            sslContextBuilder.clientAuth(ClientAuth.REQUIRE);
        }
        try {
            return sslContextBuilder.build();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    static boolean isKeyOrTrustStorePathNullOrHasFileScheme(String keyOrTrustStorePath) {
        try {
            return keyOrTrustStorePath == null || TlsUtils.makeKeyOrTrustStoreUrl(keyOrTrustStorePath).toURI().getScheme().startsWith(FILE_SCHEME);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static final class SSLContextHolder {
        static final SSLContext SSL_CONTEXT = SSL_CONTEXT_REF.get() == null ? SSLContexts.createDefault() : SSL_CONTEXT_REF.get();

        private SSLContextHolder() {
        }
    }
}

