/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.kudu;

import com.google.common.base.Strings;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.plugin.kudu.KuduBucketFunction;
import io.trino.plugin.kudu.KuduClientWrapper;
import io.trino.plugin.kudu.KuduColumnHandle;
import io.trino.plugin.kudu.KuduSplit;
import io.trino.plugin.kudu.KuduTableHandle;
import io.trino.plugin.kudu.RangePartitionChange;
import io.trino.plugin.kudu.TypeHelper;
import io.trino.plugin.kudu.properties.ColumnDesign;
import io.trino.plugin.kudu.properties.HashPartitionDefinition;
import io.trino.plugin.kudu.properties.KuduColumnProperties;
import io.trino.plugin.kudu.properties.KuduTableProperties;
import io.trino.plugin.kudu.properties.PartitionDesign;
import io.trino.plugin.kudu.properties.RangePartition;
import io.trino.plugin.kudu.properties.RangePartitionDefinition;
import io.trino.plugin.kudu.schema.SchemaEmulation;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.HostAddress;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ColumnHandle;
import io.trino.spi.connector.ColumnMetadata;
import io.trino.spi.connector.ConnectorTableMetadata;
import io.trino.spi.connector.DynamicFilter;
import io.trino.spi.connector.SchemaNotFoundException;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import io.trino.spi.predicate.DiscreteValues;
import io.trino.spi.predicate.EquatableValueSet;
import io.trino.spi.predicate.Range;
import io.trino.spi.predicate.Ranges;
import io.trino.spi.predicate.SortedRangeSet;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.predicate.ValueSet;
import io.trino.spi.type.DecimalType;
import jakarta.annotation.PreDestroy;
import java.io.IOException;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.kudu.ColumnSchema;
import org.apache.kudu.ColumnTypeAttributes;
import org.apache.kudu.Schema;
import org.apache.kudu.Type;
import org.apache.kudu.client.AlterTableOptions;
import org.apache.kudu.client.CreateTableOptions;
import org.apache.kudu.client.KuduException;
import org.apache.kudu.client.KuduPredicate;
import org.apache.kudu.client.KuduScanToken;
import org.apache.kudu.client.KuduScanner;
import org.apache.kudu.client.KuduSession;
import org.apache.kudu.client.KuduTable;
import org.apache.kudu.client.PartialRow;

public class KuduClientSession {
    private static final Logger log = Logger.get(KuduClientSession.class);
    public static final String DEFAULT_SCHEMA = "default";
    private final KuduClientWrapper client;
    private final SchemaEmulation schemaEmulation;
    private final boolean allowLocalScheduling;

    public KuduClientSession(KuduClientWrapper client, SchemaEmulation schemaEmulation, boolean allowLocalScheduling) {
        this.client = client;
        this.schemaEmulation = schemaEmulation;
        this.allowLocalScheduling = allowLocalScheduling;
    }

    public List<String> listSchemaNames() {
        return this.schemaEmulation.listSchemaNames(this.client);
    }

    private List<String> internalListTables(String prefix) {
        try {
            if (prefix.isEmpty()) {
                return this.client.getTablesList().getTablesList();
            }
            return this.client.getTablesList(prefix).getTablesList();
        }
        catch (KuduException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, (Throwable)e);
        }
    }

    public List<SchemaTableName> listTables(Optional<String> optSchemaName) {
        if (optSchemaName.isPresent()) {
            return this.listTablesSingleSchema(optSchemaName.get());
        }
        ArrayList<SchemaTableName> all = new ArrayList<SchemaTableName>();
        for (String schemaName : this.listSchemaNames()) {
            List<SchemaTableName> single = this.listTablesSingleSchema(schemaName);
            all.addAll(single);
        }
        return all;
    }

    private List<SchemaTableName> listTablesSingleSchema(String schemaName) {
        String prefix = this.schemaEmulation.getPrefixForTablesOfSchema(schemaName);
        List<String> tables = this.internalListTables(prefix);
        if (schemaName.equals(DEFAULT_SCHEMA)) {
            tables = this.schemaEmulation.filterTablesForDefaultSchema(tables);
        }
        return (List)tables.stream().map(this.schemaEmulation::fromRawName).filter(Objects::nonNull).collect(ImmutableList.toImmutableList());
    }

    public Schema getTableSchema(KuduTableHandle tableHandle) {
        KuduTable table = tableHandle.getTable(this);
        return table.getSchema();
    }

    public Map<String, Object> getTableProperties(KuduTableHandle tableHandle) {
        KuduTable table = tableHandle.getTable(this);
        return KuduTableProperties.toMap(table);
    }

    public List<KuduSplit> buildKuduSplits(KuduTableHandle tableHandle, DynamicFilter dynamicFilter) {
        List columnIndexes;
        KuduTable table = tableHandle.getTable(this);
        int primaryKeyColumnCount = table.getSchema().getPrimaryKeyColumnCount();
        KuduScanToken.KuduScanTokenBuilder builder = this.client.newScanTokenBuilder(table);
        builder.includeTabletMetadata(false);
        TupleDomain constraint = tableHandle.getConstraint().intersect(dynamicFilter.getCurrentPredicate().simplify(100));
        if (constraint.isNone()) {
            return ImmutableList.of();
        }
        this.addConstraintPredicates(table, builder, (TupleDomain<ColumnHandle>)constraint);
        Optional<List<ColumnHandle>> desiredColumns = tableHandle.getDesiredColumns();
        if (tableHandle.isRequiresRowId()) {
            if (desiredColumns.isPresent()) {
                columnIndexes = IntStream.range(0, primaryKeyColumnCount).boxed().collect(Collectors.toList());
                for (ColumnHandle column : desiredColumns.get()) {
                    KuduColumnHandle k = (KuduColumnHandle)column;
                    int index = k.ordinalPosition();
                    if (index < primaryKeyColumnCount) continue;
                    columnIndexes.add(index);
                }
                columnIndexes = ImmutableList.copyOf((Collection)columnIndexes);
            } else {
                columnIndexes = (List)IntStream.range(0, table.getSchema().getColumnCount()).boxed().collect(ImmutableList.toImmutableList());
            }
        } else if (desiredColumns.isPresent()) {
            columnIndexes = (List)desiredColumns.get().stream().map(handle -> ((KuduColumnHandle)handle).ordinalPosition()).collect(ImmutableList.toImmutableList());
        } else {
            ImmutableList.Builder columnIndexesBuilder = ImmutableList.builder();
            Schema schema = table.getSchema();
            for (int ordinal = 0; ordinal < schema.getColumnCount(); ++ordinal) {
                ColumnSchema column = schema.getColumnByIndex(ordinal);
                if (column.isKey() && column.getName().equals("row_uuid")) continue;
                columnIndexesBuilder.add((Object)ordinal);
            }
            columnIndexes = columnIndexesBuilder.build();
        }
        builder.setProjectedColumnIndexes(columnIndexes);
        tableHandle.getLimit().ifPresent(arg_0 -> ((KuduScanToken.KuduScanTokenBuilder)builder).limit(arg_0));
        List tokens = builder.build();
        ImmutableList.Builder tokenBuilder = ImmutableList.builder();
        List hashBucketSchemas = table.getPartitionSchema().getHashBucketSchemas();
        for (KuduScanToken token : tokens) {
            List hashBuckets = token.getTablet().getPartition().getHashBuckets();
            int bucket = KuduBucketFunction.getBucket(hashBuckets, hashBucketSchemas);
            tokenBuilder.add((Object)this.toKuduSplit(tableHandle, token, primaryKeyColumnCount, bucket));
        }
        return tokenBuilder.build();
    }

    public KuduScanner createScanner(KuduSplit kuduSplit) {
        try {
            return this.client.deserializeIntoScanner(kuduSplit.getSerializedScanToken());
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public KuduTable openTable(SchemaTableName schemaTableName) {
        String rawName = this.schemaEmulation.toRawName(schemaTableName);
        try {
            return this.client.openTable(rawName);
        }
        catch (KuduException e) {
            log.debug((Throwable)e, "Error on doOpenTable");
            if (!this.listSchemaNames().contains(schemaTableName.getSchemaName())) {
                throw new SchemaNotFoundException(schemaTableName.getSchemaName(), (Throwable)e);
            }
            throw new TableNotFoundException(schemaTableName, (Throwable)e);
        }
    }

    public KuduSession newSession() {
        return this.client.newSession();
    }

    public void createSchema(String schemaName) {
        this.schemaEmulation.createSchema(this.client, schemaName);
    }

    public void dropSchema(String schemaName, boolean cascade) {
        this.schemaEmulation.dropSchema(this.client, schemaName, cascade);
    }

    public void dropTable(SchemaTableName schemaTableName) {
        try {
            String rawName = this.schemaEmulation.toRawName(schemaTableName);
            this.client.deleteTable(rawName);
        }
        catch (KuduException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, (Throwable)e);
        }
    }

    public void renameTable(SchemaTableName schemaTableName, SchemaTableName newSchemaTableName) {
        try {
            String rawName = this.schemaEmulation.toRawName(schemaTableName);
            String newRawName = this.schemaEmulation.toRawName(newSchemaTableName);
            AlterTableOptions alterOptions = new AlterTableOptions();
            alterOptions.renameTable(newRawName);
            this.client.alterTable(rawName, alterOptions);
        }
        catch (KuduException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, (Throwable)e);
        }
    }

    public KuduTable createTable(ConnectorTableMetadata tableMetadata, boolean ignoreExisting) {
        try {
            String rawName = this.schemaEmulation.toRawName(tableMetadata.getTable());
            if (ignoreExisting && this.client.tableExists(rawName)) {
                return null;
            }
            if (!this.schemaEmulation.existsSchema(this.client, tableMetadata.getTable().getSchemaName())) {
                throw new SchemaNotFoundException(tableMetadata.getTable().getSchemaName());
            }
            List columns = tableMetadata.getColumns();
            Map properties = tableMetadata.getProperties();
            Schema schema = this.buildSchema(columns);
            CreateTableOptions options = this.buildCreateTableOptions(schema, properties);
            tableMetadata.getComment().ifPresent(arg_0 -> ((CreateTableOptions)options).setComment(arg_0));
            return this.client.createTable(rawName, schema, options);
        }
        catch (KuduException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, (Throwable)e);
        }
    }

    public void addColumn(SchemaTableName schemaTableName, ColumnMetadata column) {
        try {
            String rawName = this.schemaEmulation.toRawName(schemaTableName);
            AlterTableOptions alterOptions = new AlterTableOptions();
            Type type = TypeHelper.toKuduClientType(column.getType());
            ColumnSchema.ColumnSchemaBuilder builder = new ColumnSchema.ColumnSchemaBuilder(column.getName(), type).nullable(true).defaultValue(null).comment(Strings.nullToEmpty((String)column.getComment()));
            this.setTypeAttributes(column, builder);
            alterOptions.addColumn(builder.build());
            this.client.alterTable(rawName, alterOptions);
        }
        catch (KuduException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, (Throwable)e);
        }
    }

    public void dropColumn(SchemaTableName schemaTableName, String name) {
        try {
            String rawName = this.schemaEmulation.toRawName(schemaTableName);
            AlterTableOptions alterOptions = new AlterTableOptions();
            alterOptions.dropColumn(name);
            this.client.alterTable(rawName, alterOptions);
        }
        catch (KuduException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, (Throwable)e);
        }
    }

    public void renameColumn(SchemaTableName schemaTableName, String oldName, String newName) {
        try {
            String rawName = this.schemaEmulation.toRawName(schemaTableName);
            AlterTableOptions alterOptions = new AlterTableOptions();
            alterOptions.renameColumn(oldName, newName);
            this.client.alterTable(rawName, alterOptions);
        }
        catch (KuduException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, (Throwable)e);
        }
    }

    public void addRangePartition(SchemaTableName schemaTableName, RangePartition rangePartition) {
        this.changeRangePartition(schemaTableName, rangePartition, RangePartitionChange.ADD);
    }

    public void dropRangePartition(SchemaTableName schemaTableName, RangePartition rangePartition) {
        this.changeRangePartition(schemaTableName, rangePartition, RangePartitionChange.DROP);
    }

    private void changeRangePartition(SchemaTableName schemaTableName, RangePartition rangePartition, RangePartitionChange change) {
        try {
            String rawName = this.schemaEmulation.toRawName(schemaTableName);
            KuduTable table = this.client.openTable(rawName);
            Schema schema = table.getSchema();
            PartitionDesign design = KuduTableProperties.getPartitionDesign(table);
            RangePartitionDefinition definition = design.getRange();
            if (definition == null) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.QUERY_REJECTED, "Table " + String.valueOf(schemaTableName) + " has no range partition");
            }
            PartialRow lowerBound = KuduTableProperties.toRangeBoundToPartialRow(schema, definition, rangePartition.lower());
            PartialRow upperBound = KuduTableProperties.toRangeBoundToPartialRow(schema, definition, rangePartition.upper());
            AlterTableOptions alterOptions = new AlterTableOptions();
            switch (change) {
                case ADD: {
                    alterOptions.addRangePartition(lowerBound, upperBound);
                    break;
                }
                case DROP: {
                    alterOptions.dropRangePartition(lowerBound, upperBound);
                }
            }
            this.client.alterTable(rawName, alterOptions);
        }
        catch (KuduException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, (Throwable)e);
        }
    }

    private Schema buildSchema(List<ColumnMetadata> columns) {
        List kuduColumns = (List)columns.stream().map(this::toColumnSchema).collect(ImmutableList.toImmutableList());
        return new Schema(kuduColumns);
    }

    private ColumnSchema toColumnSchema(ColumnMetadata columnMetadata) {
        String name = columnMetadata.getName();
        ColumnDesign design = KuduColumnProperties.getColumnDesign(columnMetadata.getProperties());
        Type ktype = TypeHelper.toKuduClientType(columnMetadata.getType());
        ColumnSchema.ColumnSchemaBuilder builder = new ColumnSchema.ColumnSchemaBuilder(name, ktype);
        builder.key(design.isPrimaryKey()).nullable(design.isNullable());
        this.setEncoding(name, builder, design);
        this.setCompression(name, builder, design);
        this.setTypeAttributes(columnMetadata, builder);
        return builder.build();
    }

    private void setTypeAttributes(ColumnMetadata columnMetadata, ColumnSchema.ColumnSchemaBuilder builder) {
        io.trino.spi.type.Type type = columnMetadata.getType();
        if (type instanceof DecimalType) {
            DecimalType type2 = (DecimalType)type;
            ColumnTypeAttributes attributes = new ColumnTypeAttributes.ColumnTypeAttributesBuilder().precision(type2.getPrecision()).scale(type2.getScale()).build();
            builder.typeAttributes(attributes);
        }
    }

    private void setCompression(String name, ColumnSchema.ColumnSchemaBuilder builder, ColumnDesign design) {
        if (design.getCompression() != null) {
            try {
                ColumnSchema.CompressionAlgorithm algorithm = KuduColumnProperties.lookupCompression(design.getCompression());
                builder.compressionAlgorithm(algorithm);
            }
            catch (IllegalArgumentException e) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Unknown compression algorithm " + design.getCompression() + " for column " + name);
            }
        }
    }

    private void setEncoding(String name, ColumnSchema.ColumnSchemaBuilder builder, ColumnDesign design) {
        if (design.getEncoding() != null) {
            try {
                ColumnSchema.Encoding encoding = KuduColumnProperties.lookupEncoding(design.getEncoding());
                builder.encoding(encoding);
            }
            catch (IllegalArgumentException e) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Unknown encoding " + design.getEncoding() + " for column " + name);
            }
        }
    }

    private CreateTableOptions buildCreateTableOptions(Schema schema, Map<String, Object> properties) {
        CreateTableOptions options = new CreateTableOptions();
        RangePartitionDefinition rangePartitionDefinition = null;
        PartitionDesign partitionDesign = KuduTableProperties.getPartitionDesign(properties);
        if (partitionDesign.getHash() != null) {
            for (HashPartitionDefinition hashPartitionDefinition : partitionDesign.getHash()) {
                options.addHashPartitions(hashPartitionDefinition.columns(), hashPartitionDefinition.buckets());
            }
        }
        if (partitionDesign.getRange() != null) {
            rangePartitionDefinition = partitionDesign.getRange();
            options.setRangePartitionColumns(rangePartitionDefinition.columns());
        }
        List<RangePartition> rangePartitions = KuduTableProperties.getRangePartitions(properties);
        if (rangePartitionDefinition != null && !rangePartitions.isEmpty()) {
            for (RangePartition rangePartition : rangePartitions) {
                PartialRow lower = KuduTableProperties.toRangeBoundToPartialRow(schema, rangePartitionDefinition, rangePartition.lower());
                PartialRow upper = KuduTableProperties.toRangeBoundToPartialRow(schema, rangePartitionDefinition, rangePartition.upper());
                options.addRangePartition(lower, upper);
            }
        }
        Optional<Integer> optional = KuduTableProperties.getNumReplicas(properties);
        optional.ifPresent(arg_0 -> ((CreateTableOptions)options).setNumReplicas(arg_0));
        return options;
    }

    private void addConstraintPredicates(KuduTable table, KuduScanToken.KuduScanTokenBuilder builder, TupleDomain<ColumnHandle> constraintSummary) {
        Verify.verify((!constraintSummary.isNone() ? 1 : 0) != 0, (String)"constraintSummary is none", (Object[])new Object[0]);
        if (constraintSummary.isAll()) {
            return;
        }
        Schema schema = table.getSchema();
        ((Map)constraintSummary.getDomains().orElseThrow()).forEach((columnHandle, domain) -> {
            int position = ((KuduColumnHandle)columnHandle).ordinalPosition();
            ColumnSchema columnSchema = schema.getColumnByIndex(position);
            Verify.verify((!domain.isNone() ? 1 : 0) != 0, (String)"Domain is none", (Object[])new Object[0]);
            if (!domain.isAll()) {
                if (domain.isOnlyNull()) {
                    builder.addPredicate(KuduPredicate.newIsNullPredicate((ColumnSchema)columnSchema));
                } else if (domain.getValues().isNone() || !domain.isNullAllowed()) {
                    if (domain.getValues().isAll() && !domain.isNullAllowed()) {
                        builder.addPredicate(KuduPredicate.newIsNotNullPredicate((ColumnSchema)columnSchema));
                    } else if (domain.isSingleValue()) {
                        KuduPredicate predicate = this.createEqualsPredicate(columnSchema, domain.getSingleValue());
                        builder.addPredicate(predicate);
                    } else {
                        ValueSet valueSet = domain.getValues();
                        if (valueSet instanceof EquatableValueSet) {
                            DiscreteValues discreteValues = valueSet.getDiscreteValues();
                            KuduPredicate predicate = this.createInListPredicate(columnSchema, discreteValues);
                            builder.addPredicate(predicate);
                        } else if (valueSet instanceof SortedRangeSet) {
                            SortedRangeSet sortedRangeSet = (SortedRangeSet)valueSet;
                            Ranges ranges = sortedRangeSet.getRanges();
                            List rangeList = ranges.getOrderedRanges();
                            if (rangeList.stream().allMatch(Range::isSingleValue)) {
                                io.trino.spi.type.Type type = TypeHelper.fromKuduColumn(columnSchema);
                                List javaValues = (List)rangeList.stream().map(range -> TypeHelper.getJavaValue(type, range.getSingleValue())).collect(ImmutableList.toImmutableList());
                                KuduPredicate predicate = KuduPredicate.newInListPredicate((ColumnSchema)columnSchema, (List)javaValues);
                                builder.addPredicate(predicate);
                            } else {
                                KuduPredicate predicate;
                                KuduPredicate.ComparisonOp op;
                                Range span = ranges.getSpan();
                                if (!span.isLowUnbounded()) {
                                    op = span.isLowInclusive() ? KuduPredicate.ComparisonOp.GREATER_EQUAL : KuduPredicate.ComparisonOp.GREATER;
                                    predicate = this.createComparisonPredicate(columnSchema, op, span.getLowBoundedValue());
                                    builder.addPredicate(predicate);
                                }
                                if (!span.isHighUnbounded()) {
                                    op = span.isHighInclusive() ? KuduPredicate.ComparisonOp.LESS_EQUAL : KuduPredicate.ComparisonOp.LESS;
                                    predicate = this.createComparisonPredicate(columnSchema, op, span.getHighBoundedValue());
                                    builder.addPredicate(predicate);
                                }
                            }
                        } else {
                            throw new IllegalStateException("Unexpected domain: " + String.valueOf(domain));
                        }
                    }
                }
            }
        });
    }

    private KuduPredicate createInListPredicate(ColumnSchema columnSchema, DiscreteValues discreteValues) {
        io.trino.spi.type.Type type = TypeHelper.fromKuduColumn(columnSchema);
        List javaValues = (List)discreteValues.getValues().stream().map(value -> TypeHelper.getJavaValue(type, value)).collect(ImmutableList.toImmutableList());
        return KuduPredicate.newInListPredicate((ColumnSchema)columnSchema, (List)javaValues);
    }

    private KuduPredicate createEqualsPredicate(ColumnSchema columnSchema, Object value) {
        return this.createComparisonPredicate(columnSchema, KuduPredicate.ComparisonOp.EQUAL, value);
    }

    private KuduPredicate createComparisonPredicate(ColumnSchema columnSchema, KuduPredicate.ComparisonOp op, Object value) {
        io.trino.spi.type.Type type = TypeHelper.fromKuduColumn(columnSchema);
        Object javaValue = TypeHelper.getJavaValue(type, value);
        if (javaValue instanceof Long) {
            Long longValue = (Long)javaValue;
            return KuduPredicate.newComparisonPredicate((ColumnSchema)columnSchema, (KuduPredicate.ComparisonOp)op, (Object)longValue);
        }
        if (javaValue instanceof BigDecimal) {
            BigDecimal bigDecimal = (BigDecimal)javaValue;
            return KuduPredicate.newComparisonPredicate((ColumnSchema)columnSchema, (KuduPredicate.ComparisonOp)op, (BigDecimal)bigDecimal);
        }
        if (javaValue instanceof Integer) {
            Integer integerValue = (Integer)javaValue;
            return KuduPredicate.newComparisonPredicate((ColumnSchema)columnSchema, (KuduPredicate.ComparisonOp)op, (Object)integerValue);
        }
        if (javaValue instanceof Short) {
            Short shortValue = (Short)javaValue;
            return KuduPredicate.newComparisonPredicate((ColumnSchema)columnSchema, (KuduPredicate.ComparisonOp)op, (Object)shortValue);
        }
        if (javaValue instanceof Byte) {
            Byte byteValue = (Byte)javaValue;
            return KuduPredicate.newComparisonPredicate((ColumnSchema)columnSchema, (KuduPredicate.ComparisonOp)op, (Object)byteValue);
        }
        if (javaValue instanceof String) {
            String stringValue = (String)javaValue;
            return KuduPredicate.newComparisonPredicate((ColumnSchema)columnSchema, (KuduPredicate.ComparisonOp)op, (String)stringValue);
        }
        if (javaValue instanceof Double) {
            Double doubleValue = (Double)javaValue;
            return KuduPredicate.newComparisonPredicate((ColumnSchema)columnSchema, (KuduPredicate.ComparisonOp)op, (Object)doubleValue);
        }
        if (javaValue instanceof Float) {
            Float floatValue = (Float)javaValue;
            return KuduPredicate.newComparisonPredicate((ColumnSchema)columnSchema, (KuduPredicate.ComparisonOp)op, (Object)floatValue);
        }
        if (javaValue instanceof Boolean) {
            Boolean booleanValue = (Boolean)javaValue;
            return KuduPredicate.newComparisonPredicate((ColumnSchema)columnSchema, (KuduPredicate.ComparisonOp)op, (Object)booleanValue);
        }
        if (javaValue instanceof byte[]) {
            byte[] byteArrayValue = (byte[])javaValue;
            return KuduPredicate.newComparisonPredicate((ColumnSchema)columnSchema, (KuduPredicate.ComparisonOp)op, (byte[])byteArrayValue);
        }
        if (javaValue instanceof ByteBuffer) {
            ByteBuffer byteBuffer = (ByteBuffer)javaValue;
            Slice slice = Slices.wrappedHeapBuffer((ByteBuffer)byteBuffer);
            return KuduPredicate.newComparisonPredicate((ColumnSchema)columnSchema, (KuduPredicate.ComparisonOp)op, (byte[])slice.getBytes(0, slice.length()));
        }
        if (javaValue == null) {
            throw new IllegalStateException("Unexpected null java value for column " + columnSchema.getName());
        }
        throw new IllegalStateException("Unexpected java value for column " + columnSchema.getName() + ": " + String.valueOf(javaValue) + "(" + String.valueOf(javaValue.getClass()) + ")");
    }

    private KuduSplit toKuduSplit(KuduTableHandle tableHandle, KuduScanToken token, int primaryKeyColumnCount, int bucketNumber) {
        try {
            byte[] serializedScanToken = token.serialize();
            Object addresses = ImmutableList.of();
            if (this.allowLocalScheduling) {
                List replicas = token.getTablet().getReplicas();
                addresses = (List)replicas.stream().filter(replica -> replica.getRole().toLowerCase(Locale.ENGLISH).equals("leader")).map(replica -> HostAddress.fromParts((String)replica.getRpcHost(), (int)replica.getRpcPort())).collect(ImmutableList.toImmutableList());
            }
            return new KuduSplit(tableHandle.getSchemaTableName(), primaryKeyColumnCount, serializedScanToken, bucketNumber, (List<HostAddress>)addresses);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, (Throwable)e);
        }
    }

    @PreDestroy
    public void close() throws KuduException {
        this.client.close();
    }
}

