/*
 * Decompiled with CFR 0.152.
 */
package io.trino.connector;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.errorprone.annotations.ThreadSafe;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import com.google.inject.Inject;
import io.airlift.log.Logger;
import io.trino.Session;
import io.trino.connector.CatalogConnector;
import io.trino.connector.CatalogFactory;
import io.trino.connector.ConnectorServices;
import io.trino.connector.ConnectorServicesProvider;
import io.trino.connector.system.GlobalSystemConnector;
import io.trino.metadata.Catalog;
import io.trino.metadata.CatalogManager;
import io.trino.server.ForStartup;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.catalog.CatalogName;
import io.trino.spi.catalog.CatalogProperties;
import io.trino.spi.catalog.CatalogStore;
import io.trino.spi.connector.CatalogHandle;
import io.trino.spi.connector.ConnectorName;
import io.trino.util.Executors;
import jakarta.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

@ThreadSafe
public class CoordinatorDynamicCatalogManager
implements CatalogManager,
ConnectorServicesProvider {
    private static final Logger log = Logger.get(CoordinatorDynamicCatalogManager.class);
    private final CatalogStore catalogStore;
    private final CatalogFactory catalogFactory;
    private final Executor executor;
    private final Lock catalogsUpdateLock = new ReentrantLock();
    private final ConcurrentMap<CatalogName, Catalog> activeCatalogs = new ConcurrentHashMap<CatalogName, Catalog>();
    private final ConcurrentMap<CatalogHandle, CatalogConnector> allCatalogs = new ConcurrentHashMap<CatalogHandle, CatalogConnector>();
    @GuardedBy(value="catalogsUpdateLock")
    private State state = State.CREATED;

    @Inject
    public CoordinatorDynamicCatalogManager(CatalogStore catalogStore, CatalogFactory catalogFactory, @ForStartup Executor executor) {
        this.catalogStore = Objects.requireNonNull(catalogStore, "catalogStore is null");
        this.catalogFactory = Objects.requireNonNull(catalogFactory, "catalogFactory is null");
        this.executor = Objects.requireNonNull(executor, "executor is null");
    }

    @PreDestroy
    public void stop() {
        ImmutableList catalogs;
        this.catalogsUpdateLock.lock();
        try {
            if (this.state == State.STOPPED) {
                return;
            }
            this.state = State.STOPPED;
            catalogs = ImmutableList.copyOf(this.allCatalogs.values());
            this.allCatalogs.clear();
            this.activeCatalogs.clear();
        }
        finally {
            this.catalogsUpdateLock.unlock();
        }
        for (CatalogConnector connector : catalogs) {
            connector.shutdown();
        }
    }

    @Override
    public void loadInitialCatalogs() {
        this.catalogsUpdateLock.lock();
        try {
            if (this.state == State.INITIALIZED) {
                return;
            }
            Preconditions.checkState((this.state != State.STOPPED ? 1 : 0) != 0, (Object)"ConnectorManager is stopped");
            this.state = State.INITIALIZED;
            Executors.executeUntilFailure(this.executor, (Collection)this.catalogStore.getCatalogs().stream().map(storedCatalog -> () -> {
                CatalogProperties catalog = null;
                try {
                    catalog = storedCatalog.loadProperties();
                    Verify.verify((boolean)catalog.catalogHandle().getCatalogName().equals((Object)storedCatalog.name()), (String)"Catalog name does not match catalog handle", (Object[])new Object[0]);
                    CatalogConnector newCatalog = this.catalogFactory.createCatalog(catalog);
                    this.activeCatalogs.put(storedCatalog.name(), newCatalog.getCatalog());
                    this.allCatalogs.put(catalog.catalogHandle(), newCatalog);
                    log.debug("-- Added catalog %s using connector %s --", new Object[]{storedCatalog.name(), catalog.connectorName()});
                }
                catch (Throwable e) {
                    CatalogHandle catalogHandle = catalog != null ? catalog.catalogHandle() : CatalogHandle.createRootCatalogHandle((CatalogName)storedCatalog.name(), (CatalogHandle.CatalogVersion)new CatalogHandle.CatalogVersion("failed"));
                    ConnectorName connectorName = catalog != null ? catalog.connectorName() : new ConnectorName("unknown");
                    this.activeCatalogs.put(storedCatalog.name(), Catalog.failedCatalog(storedCatalog.name(), catalogHandle, connectorName));
                    log.error(e, "-- Failed to load catalog %s using connector %s --", new Object[]{storedCatalog.name(), connectorName});
                }
                return null;
            }).collect(ImmutableList.toImmutableList()));
        }
        finally {
            this.catalogsUpdateLock.unlock();
        }
    }

    @Override
    public Set<CatalogName> getCatalogNames() {
        return ImmutableSet.copyOf(this.activeCatalogs.keySet());
    }

    @Override
    public Optional<Catalog> getCatalog(CatalogName catalogName) {
        return Optional.ofNullable((Catalog)this.activeCatalogs.get(catalogName));
    }

    @Override
    public Set<CatalogHandle> getActiveCatalogs() {
        return (Set)this.activeCatalogs.values().stream().map(Catalog::getCatalogHandle).collect(ImmutableSet.toImmutableSet());
    }

    @Override
    public void ensureCatalogsLoaded(Session session, List<CatalogProperties> catalogs) {
        List missingCatalogs = (List)catalogs.stream().filter(catalog -> !this.allCatalogs.containsKey(catalog.catalogHandle())).collect(ImmutableList.toImmutableList());
        if (!missingCatalogs.isEmpty()) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.CATALOG_NOT_AVAILABLE, "Missing catalogs: " + String.valueOf(missingCatalogs));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void pruneCatalogs(Set<CatalogHandle> catalogsInUse) {
        ArrayList<CatalogConnector> removedCatalogs = new ArrayList<CatalogConnector>();
        this.catalogsUpdateLock.lock();
        try {
            if (this.state == State.STOPPED) {
                return;
            }
            Iterator iterator = this.allCatalogs.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry entry = iterator.next();
                Catalog activeCatalog = (Catalog)this.activeCatalogs.get(((CatalogHandle)entry.getKey()).getCatalogName());
                if (activeCatalog != null && activeCatalog.getCatalogHandle().equals(entry.getKey()) || catalogsInUse.contains(entry.getKey())) continue;
                iterator.remove();
                removedCatalogs.add((CatalogConnector)entry.getValue());
            }
        }
        finally {
            this.catalogsUpdateLock.unlock();
        }
        for (CatalogConnector removedCatalog : removedCatalogs) {
            try {
                removedCatalog.shutdown();
            }
            catch (Throwable e) {
                log.error(e, "Error shutting down catalog: %s".formatted(removedCatalog));
            }
        }
        if (!removedCatalogs.isEmpty()) {
            List<String> sortedHandles = removedCatalogs.stream().map(connector -> connector.getCatalogHandle().toString()).sorted().toList();
            log.debug("Pruned catalogs: %s", new Object[]{sortedHandles});
        }
    }

    @Override
    public Optional<CatalogProperties> getCatalogProperties(CatalogHandle catalogHandle) {
        return Optional.ofNullable((CatalogConnector)this.allCatalogs.get(catalogHandle.getRootCatalogHandle())).flatMap(CatalogConnector::getCatalogProperties);
    }

    @Override
    public ConnectorServices getConnectorServices(CatalogHandle catalogHandle) {
        CatalogConnector catalogConnector = (CatalogConnector)this.allCatalogs.get(catalogHandle.getRootCatalogHandle());
        Preconditions.checkArgument((catalogConnector != null ? 1 : 0) != 0, (String)"No catalog '%s'", (Object)catalogHandle.getCatalogName());
        return catalogConnector.getMaterializedConnector(catalogHandle.getType());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void createCatalog(CatalogName catalogName, ConnectorName connectorName, Map<String, String> properties, boolean notExists) {
        Objects.requireNonNull(catalogName, "catalogName is null");
        Objects.requireNonNull(connectorName, "connectorName is null");
        Objects.requireNonNull(properties, "properties is null");
        this.catalogsUpdateLock.lock();
        try {
            Preconditions.checkState((this.state != State.STOPPED ? 1 : 0) != 0, (Object)"ConnectorManager is stopped");
            if (this.activeCatalogs.containsKey(catalogName)) {
                if (!notExists) {
                    throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.ALREADY_EXISTS, String.format("Catalog '%s' already exists", catalogName));
                }
                return;
            }
            CatalogProperties catalogProperties = this.catalogStore.createCatalogProperties(catalogName, connectorName, properties);
            CatalogConnector catalog = this.allCatalogs.computeIfAbsent(catalogProperties.catalogHandle(), handle -> this.catalogFactory.createCatalog(catalogProperties));
            this.catalogStore.addOrReplaceCatalog(catalogProperties);
            this.activeCatalogs.put(catalogName, catalog.getCatalog());
            log.debug("Added catalog: %s", new Object[]{catalog.getCatalogHandle()});
        }
        finally {
            this.catalogsUpdateLock.unlock();
        }
    }

    public void registerGlobalSystemConnector(GlobalSystemConnector connector) {
        Objects.requireNonNull(connector, "connector is null");
        this.catalogsUpdateLock.lock();
        try {
            if (this.state == State.STOPPED) {
                return;
            }
            CatalogConnector catalog = this.catalogFactory.createCatalog(GlobalSystemConnector.CATALOG_HANDLE, new ConnectorName("system"), connector);
            if (this.activeCatalogs.putIfAbsent(new CatalogName("system"), catalog.getCatalog()) != null) {
                throw new IllegalStateException("Global system catalog already registered");
            }
            this.allCatalogs.put(GlobalSystemConnector.CATALOG_HANDLE, catalog);
        }
        finally {
            this.catalogsUpdateLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void dropCatalog(CatalogName catalogName, boolean exists) {
        boolean removed;
        Objects.requireNonNull(catalogName, "catalogName is null");
        this.catalogsUpdateLock.lock();
        try {
            Preconditions.checkState((this.state != State.STOPPED ? 1 : 0) != 0, (Object)"ConnectorManager is stopped");
            this.catalogStore.removeCatalog(catalogName);
            removed = this.activeCatalogs.remove(catalogName) != null;
        }
        finally {
            this.catalogsUpdateLock.unlock();
        }
        if (!removed && !exists) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.CATALOG_NOT_FOUND, String.format("Catalog '%s' not found", catalogName));
        }
    }

    private static enum State {
        CREATED,
        INITIALIZED,
        STOPPED;

    }
}

