/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.function.Consumer;
import org.apache.iceberg.ExpireSnapshots;
import org.apache.iceberg.FileCleanupStrategy;
import org.apache.iceberg.IncrementalFileCleanup;
import org.apache.iceberg.ReachableFileCleanup;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.SnapshotRef;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.TableOperations;
import org.apache.iceberg.exceptions.CommitFailedException;
import org.apache.iceberg.exceptions.ValidationException;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.Maps;
import org.apache.iceberg.relocated.com.google.common.collect.Sets;
import org.apache.iceberg.relocated.com.google.common.util.concurrent.MoreExecutors;
import org.apache.iceberg.util.DateTimeUtil;
import org.apache.iceberg.util.PropertyUtil;
import org.apache.iceberg.util.SnapshotUtil;
import org.apache.iceberg.util.Tasks;
import org.apache.iceberg.util.ThreadPools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RemoveSnapshots
implements ExpireSnapshots {
    private static final Logger LOG = LoggerFactory.getLogger(RemoveSnapshots.class);
    private static final ExecutorService DEFAULT_DELETE_EXECUTOR_SERVICE = MoreExecutors.newDirectExecutorService();
    private final Consumer<String> defaultDelete = new Consumer<String>(){

        @Override
        public void accept(String file) {
            RemoveSnapshots.this.ops.io().deleteFile(file);
        }
    };
    private final TableOperations ops;
    private final Set<Long> idsToRemove = Sets.newHashSet();
    private final long now;
    private final long defaultMaxRefAgeMs;
    private boolean cleanExpiredFiles = true;
    private TableMetadata base;
    private long defaultExpireOlderThan;
    private int defaultMinNumSnapshots;
    private Consumer<String> deleteFunc = this.defaultDelete;
    private ExecutorService deleteExecutorService = DEFAULT_DELETE_EXECUTOR_SERVICE;
    private ExecutorService planExecutorService = ThreadPools.getWorkerPool();
    private Boolean incrementalCleanup;
    private boolean specifiedSnapshotId = false;

    RemoveSnapshots(TableOperations ops) {
        this.ops = ops;
        this.base = ops.current();
        ValidationException.check((boolean)PropertyUtil.propertyAsBoolean(this.base.properties(), "gc.enabled", true), (String)"Cannot expire snapshots: GC is disabled (deleting files may corrupt other tables)", (Object[])new Object[0]);
        long defaultMaxSnapshotAgeMs = PropertyUtil.propertyAsLong(this.base.properties(), "history.expire.max-snapshot-age-ms", 432000000L);
        this.now = System.currentTimeMillis();
        this.defaultExpireOlderThan = this.now - defaultMaxSnapshotAgeMs;
        this.defaultMinNumSnapshots = PropertyUtil.propertyAsInt(this.base.properties(), "history.expire.min-snapshots-to-keep", 1);
        this.defaultMaxRefAgeMs = PropertyUtil.propertyAsLong(this.base.properties(), "history.expire.max-ref-age-ms", Long.MAX_VALUE);
    }

    public ExpireSnapshots cleanExpiredFiles(boolean clean) {
        this.cleanExpiredFiles = clean;
        return this;
    }

    public ExpireSnapshots expireSnapshotId(long expireSnapshotId) {
        LOG.info("Expiring snapshot with id: {}", (Object)expireSnapshotId);
        this.idsToRemove.add(expireSnapshotId);
        this.specifiedSnapshotId = true;
        return this;
    }

    public ExpireSnapshots expireOlderThan(long timestampMillis) {
        LOG.info("Expiring snapshots older than: {} ({})", (Object)DateTimeUtil.formatTimestampMillis((long)timestampMillis), (Object)timestampMillis);
        this.defaultExpireOlderThan = timestampMillis;
        return this;
    }

    public ExpireSnapshots retainLast(int numSnapshots) {
        Preconditions.checkArgument((1 <= numSnapshots ? 1 : 0) != 0, (String)"Number of snapshots to retain must be at least 1, cannot be: %s", (int)numSnapshots);
        this.defaultMinNumSnapshots = numSnapshots;
        return this;
    }

    public ExpireSnapshots deleteWith(Consumer<String> newDeleteFunc) {
        this.deleteFunc = newDeleteFunc;
        return this;
    }

    public ExpireSnapshots executeDeleteWith(ExecutorService executorService) {
        this.deleteExecutorService = executorService;
        return this;
    }

    public ExpireSnapshots planWith(ExecutorService executorService) {
        this.planExecutorService = executorService;
        return this;
    }

    public List<Snapshot> apply() {
        TableMetadata updated = this.internalApply();
        ArrayList removed = Lists.newArrayList(this.base.snapshots());
        removed.removeAll(updated.snapshots());
        return removed;
    }

    private TableMetadata internalApply() {
        this.base = this.ops.refresh();
        if (this.base.snapshots().isEmpty()) {
            return this.base;
        }
        HashSet idsToRetain = Sets.newHashSet();
        Map<String, SnapshotRef> retainedRefs = this.computeRetainedRefs(this.base.refs());
        HashMap retainedIdToRefs = Maps.newHashMap();
        for (Map.Entry<String, SnapshotRef> retainedRefEntry : retainedRefs.entrySet()) {
            long snapshotId = retainedRefEntry.getValue().snapshotId();
            retainedIdToRefs.putIfAbsent(snapshotId, Lists.newArrayList());
            ((List)retainedIdToRefs.get(snapshotId)).add(retainedRefEntry.getKey());
            idsToRetain.add(snapshotId);
        }
        Iterator<Object> iterator = this.idsToRemove.iterator();
        while (iterator.hasNext()) {
            long idToRemove = (Long)iterator.next();
            List refsForId = (List)retainedIdToRefs.get(idToRemove);
            Preconditions.checkArgument((refsForId == null ? 1 : 0) != 0, (String)"Cannot expire %s. Still referenced by refs: %s", (long)idToRemove, (Object)refsForId);
        }
        idsToRetain.addAll(this.computeAllBranchSnapshotsToRetain(retainedRefs.values()));
        idsToRetain.addAll(this.unreferencedSnapshotsToRetain(retainedRefs.values()));
        TableMetadata.Builder updatedMetaBuilder = TableMetadata.buildFrom(this.base);
        this.base.refs().keySet().stream().filter(ref -> !retainedRefs.containsKey(ref)).forEach(updatedMetaBuilder::removeRef);
        this.base.snapshots().stream().map(Snapshot::snapshotId).filter(snapshot -> !idsToRetain.contains(snapshot)).forEach(this.idsToRemove::add);
        updatedMetaBuilder.removeSnapshots(this.idsToRemove);
        return updatedMetaBuilder.build();
    }

    private Map<String, SnapshotRef> computeRetainedRefs(Map<String, SnapshotRef> refs) {
        HashMap retainedRefs = Maps.newHashMap();
        for (Map.Entry<String, SnapshotRef> refEntry : refs.entrySet()) {
            long maxRefAgeMs;
            String name = refEntry.getKey();
            SnapshotRef ref = refEntry.getValue();
            if (name.equals("main")) {
                retainedRefs.put(name, ref);
                continue;
            }
            Snapshot snapshot = this.base.snapshot(ref.snapshotId());
            long l = maxRefAgeMs = ref.maxRefAgeMs() != null ? ref.maxRefAgeMs() : this.defaultMaxRefAgeMs;
            if (snapshot != null) {
                long refAgeMs = this.now - snapshot.timestampMillis();
                if (refAgeMs > maxRefAgeMs) continue;
                retainedRefs.put(name, ref);
                continue;
            }
            LOG.warn("Removing invalid ref {}: snapshot {} does not exist", (Object)name, (Object)ref.snapshotId());
        }
        return retainedRefs;
    }

    private Set<Long> computeAllBranchSnapshotsToRetain(Collection<SnapshotRef> refs) {
        HashSet branchSnapshotsToRetain = Sets.newHashSet();
        for (SnapshotRef ref : refs) {
            if (!ref.isBranch()) continue;
            long expireSnapshotsOlderThan = ref.maxSnapshotAgeMs() != null ? this.now - ref.maxSnapshotAgeMs() : this.defaultExpireOlderThan;
            int minSnapshotsToKeep = ref.minSnapshotsToKeep() != null ? ref.minSnapshotsToKeep() : this.defaultMinNumSnapshots;
            branchSnapshotsToRetain.addAll(this.computeBranchSnapshotsToRetain(ref.snapshotId(), expireSnapshotsOlderThan, minSnapshotsToKeep));
        }
        return branchSnapshotsToRetain;
    }

    private Set<Long> computeBranchSnapshotsToRetain(long snapshot, long expireSnapshotsOlderThan, int minSnapshotsToKeep) {
        HashSet idsToRetain = Sets.newHashSet();
        for (Snapshot ancestor : SnapshotUtil.ancestorsOf(snapshot, this.base::snapshot)) {
            if (idsToRetain.size() < minSnapshotsToKeep || ancestor.timestampMillis() >= expireSnapshotsOlderThan) {
                idsToRetain.add(ancestor.snapshotId());
                continue;
            }
            return idsToRetain;
        }
        return idsToRetain;
    }

    private Set<Long> unreferencedSnapshotsToRetain(Collection<SnapshotRef> refs) {
        HashSet referencedSnapshots = Sets.newHashSet();
        for (SnapshotRef ref : refs) {
            if (ref.isBranch()) {
                for (Snapshot snapshot : SnapshotUtil.ancestorsOf(ref.snapshotId(), this.base::snapshot)) {
                    referencedSnapshots.add(snapshot.snapshotId());
                }
                continue;
            }
            referencedSnapshots.add(ref.snapshotId());
        }
        HashSet snapshotsToRetain = Sets.newHashSet();
        for (Snapshot snapshot : this.base.snapshots()) {
            if (referencedSnapshots.contains(snapshot.snapshotId()) || snapshot.timestampMillis() < this.defaultExpireOlderThan) continue;
            snapshotsToRetain.add(snapshot.snapshotId());
        }
        return snapshotsToRetain;
    }

    public void commit() {
        Tasks.foreach(this.ops).retry(this.base.propertyAsInt("commit.retry.num-retries", 4)).exponentialBackoff(this.base.propertyAsInt("commit.retry.min-wait-ms", 100), this.base.propertyAsInt("commit.retry.max-wait-ms", 60000), this.base.propertyAsInt("commit.retry.total-timeout-ms", 1800000), 2.0).onlyRetryOn((Class<Exception>)CommitFailedException.class).run(item -> {
            TableMetadata updated = this.internalApply();
            this.ops.commit(this.base, updated);
        });
        LOG.info("Committed snapshot changes");
        if (this.cleanExpiredFiles) {
            this.cleanExpiredSnapshots();
        }
    }

    ExpireSnapshots withIncrementalCleanup(boolean useIncrementalCleanup) {
        this.incrementalCleanup = useIncrementalCleanup;
        return this;
    }

    private void cleanExpiredSnapshots() {
        TableMetadata current = this.ops.refresh();
        if (this.specifiedSnapshotId) {
            if (this.incrementalCleanup != null && this.incrementalCleanup.booleanValue()) {
                throw new UnsupportedOperationException("Cannot clean files incrementally when snapshot IDs are specified");
            }
            this.incrementalCleanup = false;
        }
        if (this.incrementalCleanup == null) {
            this.incrementalCleanup = current.refs().size() == 1;
        }
        LOG.info("Cleaning up expired files (local, {})", (Object)(this.incrementalCleanup != false ? "incremental" : "reachable"));
        FileCleanupStrategy cleanupStrategy = this.incrementalCleanup != false ? new IncrementalFileCleanup(this.ops.io(), this.deleteExecutorService, this.planExecutorService, this.deleteFunc) : new ReachableFileCleanup(this.ops.io(), this.deleteExecutorService, this.planExecutorService, this.deleteFunc);
        cleanupStrategy.cleanFiles(this.base, current);
    }
}

