/*
 * Decompiled with CFR 0.152.
 */
package io.trino.geospatial.serde;

import com.esri.core.geometry.Envelope;
import com.esri.core.geometry.Geometry;
import com.esri.core.geometry.GeometryEngine;
import com.esri.core.geometry.MultiPath;
import com.esri.core.geometry.MultiPoint;
import com.esri.core.geometry.OperatorImportFromESRIShape;
import com.esri.core.geometry.Point;
import com.esri.core.geometry.Polygon;
import com.esri.core.geometry.Polyline;
import com.esri.core.geometry.ogc.OGCConcreteGeometryCollection;
import com.esri.core.geometry.ogc.OGCGeometry;
import com.esri.core.geometry.ogc.OGCGeometryCollection;
import com.esri.core.geometry.ogc.OGCLineString;
import com.esri.core.geometry.ogc.OGCMultiLineString;
import com.esri.core.geometry.ogc.OGCMultiPoint;
import com.esri.core.geometry.ogc.OGCMultiPolygon;
import com.esri.core.geometry.ogc.OGCPoint;
import com.esri.core.geometry.ogc.OGCPolygon;
import com.google.common.base.Verify;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceInput;
import io.trino.geospatial.GeometryType;
import io.trino.geospatial.GeometryUtils;
import io.trino.geospatial.serde.GeometrySerializationType;
import jakarta.annotation.Nullable;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Objects;

public final class GeometrySerde {
    private GeometrySerde() {
    }

    public static Slice serialize(OGCGeometry input) {
        Objects.requireNonNull(input, "input is null");
        DynamicSliceOutput output = new DynamicSliceOutput(100);
        GeometrySerde.writeGeometry(output, input);
        return output.slice();
    }

    public static Slice serialize(Envelope envelope) {
        Objects.requireNonNull(envelope, "envelope is null");
        Verify.verify((!envelope.isEmpty() ? 1 : 0) != 0);
        DynamicSliceOutput output = new DynamicSliceOutput(100);
        output.appendByte(GeometrySerializationType.ENVELOPE.code());
        GeometrySerde.writeEnvelopeCoordinates(output, envelope);
        return output.slice();
    }

    public static GeometryType getGeometryType(Slice shape) {
        return GeometrySerde.deserializeType(shape).geometryType();
    }

    private static void writeGeometry(DynamicSliceOutput output, OGCGeometry geometry) {
        GeometryType type = GeometryType.getForEsriGeometryType(geometry.geometryType());
        switch (type) {
            case POINT: {
                GeometrySerde.writePoint(output, geometry);
                break;
            }
            case MULTI_POINT: {
                GeometrySerde.writeSimpleGeometry(output, GeometrySerializationType.MULTI_POINT, geometry);
                break;
            }
            case LINE_STRING: {
                GeometrySerde.writeSimpleGeometry(output, GeometrySerializationType.LINE_STRING, geometry);
                break;
            }
            case MULTI_LINE_STRING: {
                GeometrySerde.writeSimpleGeometry(output, GeometrySerializationType.MULTI_LINE_STRING, geometry);
                break;
            }
            case POLYGON: {
                GeometrySerde.writeSimpleGeometry(output, GeometrySerializationType.POLYGON, geometry);
                break;
            }
            case MULTI_POLYGON: {
                GeometrySerde.writeSimpleGeometry(output, GeometrySerializationType.MULTI_POLYGON, geometry);
                break;
            }
            case GEOMETRY_COLLECTION: {
                Verify.verify((boolean)(geometry instanceof OGCConcreteGeometryCollection));
                GeometrySerde.writeGeometryCollection(output, (OGCGeometryCollection)((OGCConcreteGeometryCollection)geometry));
                break;
            }
            default: {
                throw new IllegalArgumentException("Unexpected type: " + String.valueOf((Object)type));
            }
        }
    }

    private static void writeGeometryCollection(DynamicSliceOutput output, OGCGeometryCollection collection) {
        output.appendByte(GeometrySerializationType.GEOMETRY_COLLECTION.code());
        for (int geometryIndex = 0; geometryIndex < collection.numGeometries(); ++geometryIndex) {
            OGCGeometry geometry = collection.geometryN(geometryIndex);
            int startPosition = output.size();
            output.appendInt(0);
            GeometrySerde.writeGeometry(output, geometry);
            int endPosition = output.size();
            int length = endPosition - startPosition - 4;
            output.getUnderlyingSlice().setInt(startPosition, length);
        }
    }

    private static void writeSimpleGeometry(DynamicSliceOutput output, GeometrySerializationType type, OGCGeometry geometry) {
        output.appendByte(type.code());
        Geometry esriGeometry = Objects.requireNonNull(geometry.getEsriGeometry(), "esriGeometry is null");
        byte[] shape = GeometryEngine.geometryToEsriShape((Geometry)esriGeometry);
        output.appendBytes(shape);
    }

    private static void writePoint(DynamicSliceOutput output, OGCGeometry geometry) {
        Geometry esriGeometry = geometry.getEsriGeometry();
        Verify.verify((boolean)(esriGeometry instanceof Point), (String)"geometry is expected to be an instance of Point", (Object[])new Object[0]);
        Point point = (Point)esriGeometry;
        Verify.verify((!point.hasAttribute(1) && !point.hasAttribute(2) && !point.hasAttribute(3) ? 1 : 0) != 0, (String)"Only 2D points with no ID nor M attribute are supported", (Object[])new Object[0]);
        output.appendByte(GeometrySerializationType.POINT.code());
        if (!point.isEmpty()) {
            output.appendDouble(point.getX());
            output.appendDouble(point.getY());
        } else {
            output.appendDouble(Double.NaN);
            output.appendDouble(Double.NaN);
        }
    }

    public static GeometrySerializationType deserializeType(Slice shape) {
        Objects.requireNonNull(shape, "shape is null");
        BasicSliceInput input = shape.getInput();
        Verify.verify((input.available() > 0 ? 1 : 0) != 0);
        return GeometrySerializationType.getForCode(input.readByte());
    }

    public static OGCGeometry deserialize(Slice shape) {
        Objects.requireNonNull(shape, "shape is null");
        BasicSliceInput input = shape.getInput();
        Verify.verify((input.available() > 0 ? 1 : 0) != 0);
        int length = input.available() - 1;
        GeometrySerializationType type = GeometrySerializationType.getForCode(input.readByte());
        return GeometrySerde.readGeometry(input, shape, type, length);
    }

    private static OGCGeometry readGeometry(BasicSliceInput input, Slice inputSlice, GeometrySerializationType type, int length) {
        return switch (type) {
            default -> throw new MatchException(null, null);
            case GeometrySerializationType.POINT -> GeometrySerde.readPoint(input);
            case GeometrySerializationType.MULTI_POINT, GeometrySerializationType.LINE_STRING, GeometrySerializationType.MULTI_LINE_STRING, GeometrySerializationType.POLYGON, GeometrySerializationType.MULTI_POLYGON -> GeometrySerde.readSimpleGeometry(input, inputSlice, type, length);
            case GeometrySerializationType.GEOMETRY_COLLECTION -> GeometrySerde.readGeometryCollection(input, inputSlice);
            case GeometrySerializationType.ENVELOPE -> GeometrySerde.createFromEsriGeometry((Geometry)GeometrySerde.readEnvelope((SliceInput)input), false);
        };
    }

    private static OGCConcreteGeometryCollection readGeometryCollection(BasicSliceInput input, Slice inputSlice) {
        ArrayList<OGCGeometry> geometries = new ArrayList<OGCGeometry>();
        while (input.available() > 0) {
            int length = input.readInt() - 1;
            GeometrySerializationType type = GeometrySerializationType.getForCode(input.readByte());
            geometries.add(GeometrySerde.readGeometry(input, inputSlice, type, length));
        }
        return new OGCConcreteGeometryCollection(geometries, null);
    }

    private static OGCGeometry readSimpleGeometry(BasicSliceInput input, Slice inputSlice, GeometrySerializationType type, int length) {
        int currentPosition = Math.toIntExact(input.position());
        ByteBuffer geometryBuffer = inputSlice.toByteBuffer(currentPosition, length).slice();
        input.setPosition((long)(currentPosition + length));
        Geometry esriGeometry = OperatorImportFromESRIShape.local().execute(0, Geometry.Type.Unknown, geometryBuffer);
        return GeometrySerde.createFromEsriGeometry(esriGeometry, type.geometryType().isMultitype());
    }

    private static OGCGeometry createFromEsriGeometry(Geometry geometry, boolean multiType) {
        Geometry.Type type = geometry.getType();
        switch (type) {
            case Polygon: {
                if (!multiType && ((Polygon)geometry).getExteriorRingCount() <= 1) {
                    return new OGCPolygon((Polygon)geometry, null);
                }
                return new OGCMultiPolygon((Polygon)geometry, null);
            }
            case Polyline: {
                if (!multiType && ((Polyline)geometry).getPathCount() <= 1) {
                    return new OGCLineString((MultiPath)((Polyline)geometry), 0, null);
                }
                return new OGCMultiLineString((Polyline)geometry, null);
            }
            case MultiPoint: {
                if (!multiType && ((MultiPoint)geometry).getPointCount() <= 1) {
                    if (geometry.isEmpty()) {
                        return new OGCPoint(new Point(), null);
                    }
                    return new OGCPoint(((MultiPoint)geometry).getPoint(0), null);
                }
                return new OGCMultiPoint((MultiPoint)geometry, null);
            }
            case Point: {
                if (!multiType) {
                    return new OGCPoint((Point)geometry, null);
                }
                return new OGCMultiPoint((Point)geometry, null);
            }
            case Envelope: {
                Polygon polygon = new Polygon();
                polygon.addEnvelope((Envelope)geometry, false);
                return new OGCPolygon(polygon, null);
            }
            case Line: {
                break;
            }
        }
        throw new IllegalArgumentException("Unexpected geometry type: " + String.valueOf(type));
    }

    private static OGCPoint readPoint(BasicSliceInput input) {
        double x = input.readDouble();
        double y = input.readDouble();
        Point point = Double.isNaN(x) || Double.isNaN(y) ? new Point() : new Point(x, y);
        return new OGCPoint(point, null);
    }

    @Nullable
    public static Envelope deserializeEnvelope(Slice shape) {
        Objects.requireNonNull(shape, "shape is null");
        BasicSliceInput input = shape.getInput();
        Verify.verify((input.available() > 0 ? 1 : 0) != 0);
        int length = input.available() - 1;
        GeometrySerializationType type = GeometrySerializationType.getForCode(input.readByte());
        return GeometrySerde.getEnvelope(input, type, length);
    }

    private static Envelope getEnvelope(BasicSliceInput input, GeometrySerializationType type, int length) {
        return switch (type) {
            default -> throw new MatchException(null, null);
            case GeometrySerializationType.POINT -> GeometrySerde.getPointEnvelope(input);
            case GeometrySerializationType.MULTI_POINT, GeometrySerializationType.LINE_STRING, GeometrySerializationType.MULTI_LINE_STRING, GeometrySerializationType.POLYGON, GeometrySerializationType.MULTI_POLYGON -> GeometrySerde.getSimpleGeometryEnvelope(input, length);
            case GeometrySerializationType.GEOMETRY_COLLECTION -> GeometrySerde.getGeometryCollectionOverallEnvelope(input);
            case GeometrySerializationType.ENVELOPE -> GeometrySerde.readEnvelope((SliceInput)input);
        };
    }

    private static Envelope getGeometryCollectionOverallEnvelope(BasicSliceInput input) {
        Envelope overallEnvelope = new Envelope();
        while (input.available() > 0) {
            int length = input.readInt() - 1;
            GeometrySerializationType type = GeometrySerializationType.getForCode(input.readByte());
            Envelope envelope = GeometrySerde.getEnvelope(input, type, length);
            overallEnvelope = GeometrySerde.merge(overallEnvelope, envelope);
        }
        return overallEnvelope;
    }

    private static Envelope getSimpleGeometryEnvelope(BasicSliceInput input, int length) {
        input.readInt();
        Envelope envelope = GeometrySerde.readEnvelope((SliceInput)input);
        int skipLength = length - 32 - 4;
        Verify.verify((input.skip((long)skipLength) == (long)skipLength ? 1 : 0) != 0);
        return envelope;
    }

    private static Envelope getPointEnvelope(BasicSliceInput input) {
        double x = input.readDouble();
        double y = input.readDouble();
        if (Double.isNaN(x) || Double.isNaN(y)) {
            return new Envelope();
        }
        return new Envelope(x, y, x, y);
    }

    private static Envelope readEnvelope(SliceInput input) {
        Verify.verify((input.available() > 0 ? 1 : 0) != 0);
        double xMin = input.readDouble();
        double yMin = input.readDouble();
        double xMax = input.readDouble();
        double yMax = input.readDouble();
        if (GeometryUtils.isEsriNaN(xMin) || GeometryUtils.isEsriNaN(yMin) || GeometryUtils.isEsriNaN(xMax) || GeometryUtils.isEsriNaN(yMax)) {
            return new Envelope();
        }
        return new Envelope(xMin, yMin, xMax, yMax);
    }

    private static void writeEnvelopeCoordinates(DynamicSliceOutput output, Envelope envelope) {
        if (envelope.isEmpty()) {
            output.appendDouble(Double.NaN);
            output.appendDouble(Double.NaN);
            output.appendDouble(Double.NaN);
            output.appendDouble(Double.NaN);
        } else {
            output.appendDouble(envelope.getXMin());
            output.appendDouble(envelope.getYMin());
            output.appendDouble(envelope.getXMax());
            output.appendDouble(envelope.getYMax());
        }
    }

    @Nullable
    private static Envelope merge(@Nullable Envelope left, @Nullable Envelope right) {
        if (left == null) {
            return right;
        }
        if (right == null) {
            return left;
        }
        right.merge(left);
        return right;
    }
}

