/*
 * Decompiled with CFR 0.152.
 */
package io.trino.hive.formats.line.csv;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.Slices;
import io.trino.hive.formats.line.Column;
import io.trino.hive.formats.line.LineBuffer;
import io.trino.hive.formats.line.LineDeserializer;
import io.trino.spi.PageBuilder;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarcharType;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

public class CsvDeserializer
implements LineDeserializer {
    private final List<Column> columns;
    private final char separatorChar;
    private final char quoteChar;
    private final char escapeChar;
    private final StringBuilder buffer = new StringBuilder(1024);
    private final String[] rowValues;

    public CsvDeserializer(List<Column> columns, char separatorChar, char quoteChar, char escapeChar) {
        Preconditions.checkArgument(((long)columns.size() == columns.stream().mapToInt(Column::ordinal).distinct().count() ? 1 : 0) != 0, (String)"Columns cannot have duplicate ordinals: %s", columns);
        columns.forEach(column -> Preconditions.checkArgument((column.type() == VarcharType.VARCHAR ? 1 : 0) != 0, (String)"CSV only supports VARCHAR columns: %s", (Object)column));
        this.columns = ImmutableList.copyOf(columns);
        int columnCount = this.columns.stream().mapToInt(Column::ordinal).max().orElse(-1) + 1;
        this.rowValues = new String[columnCount];
        Preconditions.checkArgument((separatorChar != '\u0000' ? 1 : 0) != 0, (Object)"Separator cannot be the null character (ASCII 0)");
        Preconditions.checkArgument((separatorChar != quoteChar ? 1 : 0) != 0, (Object)"Separator and quote character cannot be the same");
        Preconditions.checkArgument((separatorChar != escapeChar ? 1 : 0) != 0, (Object)"Separator and escape character cannot be the same");
        Preconditions.checkArgument((quoteChar != escapeChar ? 1 : 0) != 0, (Object)"Quote and escape character cannot be the same");
        this.separatorChar = separatorChar;
        this.quoteChar = quoteChar;
        this.escapeChar = escapeChar;
    }

    @Override
    public List<? extends Type> getTypes() {
        return Collections.nCopies(this.columns.size(), VarcharType.VARCHAR);
    }

    @Override
    public void deserialize(LineBuffer lineBuffer, PageBuilder builder) {
        this.parseLine(lineBuffer);
        builder.declarePosition();
        for (int i = 0; i < this.columns.size(); ++i) {
            BlockBuilder blockBuilder = builder.getBlockBuilder(i);
            String fieldValue = this.rowValues[this.columns.get(i).ordinal()];
            if (fieldValue == null) {
                blockBuilder.appendNull();
                continue;
            }
            VarcharType.VARCHAR.writeSlice(blockBuilder, Slices.utf8Slice((String)fieldValue));
        }
    }

    private void parseLine(LineBuffer lineBuffer) {
        Objects.requireNonNull(lineBuffer, "lineBuffer is null");
        Arrays.fill(this.rowValues, null);
        this.buffer.setLength(0);
        if (lineBuffer.isEmpty() || this.columns.isEmpty()) {
            return;
        }
        String line = new String(lineBuffer.getBuffer(), 0, lineBuffer.getLength(), StandardCharsets.UTF_8);
        int columnIndex = 0;
        boolean inQuotes = false;
        boolean inField = false;
        for (int position = 0; position < line.length() && columnIndex < this.rowValues.length; ++position) {
            char c = line.charAt(position);
            if (c == this.escapeChar) {
                int nextCharacter;
                if (!inQuotes && !inField || (nextCharacter = CsvDeserializer.peekNextCharacter(line, position)) != this.quoteChar && nextCharacter != this.escapeChar) continue;
                this.buffer.append(line.charAt(position + 1));
                ++position;
                continue;
            }
            if (c == this.quoteChar) {
                if ((inQuotes || inField) && CsvDeserializer.peekNextCharacter(line, position) == this.quoteChar) {
                    this.buffer.append(line.charAt(position + 1));
                    ++position;
                } else {
                    if (position > 2 && line.charAt(position - 1) != this.separatorChar && line.length() > position + 1 && line.charAt(position + 1) != this.separatorChar) {
                        if (!this.buffer.isEmpty() && CsvDeserializer.isAllWhitespace(this.buffer)) {
                            this.buffer.setLength(0);
                        } else {
                            this.buffer.append(c);
                        }
                    }
                    inQuotes = !inQuotes;
                }
                inField = !inField;
                continue;
            }
            if (c == this.separatorChar && !inQuotes) {
                this.rowValues[columnIndex] = this.buffer.toString();
                ++columnIndex;
                this.buffer.setLength(0);
                inField = false;
                continue;
            }
            this.buffer.append(c);
            inField = true;
        }
        if (columnIndex < this.rowValues.length && !inQuotes) {
            this.rowValues[columnIndex] = this.buffer.toString();
        }
        this.buffer.setLength(0);
    }

    private static int peekNextCharacter(String line, int position) {
        return line.length() > position + 1 ? (int)line.charAt(position + 1) : -1;
    }

    private static boolean isAllWhitespace(CharSequence sequence) {
        for (int i = 0; i < sequence.length(); ++i) {
            if (Character.isWhitespace(sequence.charAt(i))) continue;
            return false;
        }
        return true;
    }
}

