/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.google.sheets;

import com.google.api.client.auth.oauth2.Credential;
import com.google.api.client.googleapis.auth.oauth2.GoogleCredential;
import com.google.api.client.googleapis.javanet.GoogleNetHttpTransport;
import com.google.api.client.http.HttpRequestInitializer;
import com.google.api.client.http.HttpTransport;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.jackson2.JacksonFactory;
import com.google.api.services.sheets.v4.Sheets;
import com.google.api.services.sheets.v4.model.ValueRange;
import com.google.common.base.Throwables;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.google.inject.Inject;
import io.airlift.log.Logger;
import io.trino.cache.EvictableCacheBuilder;
import io.trino.cache.NonEvictableLoadingCache;
import io.trino.cache.SafeCaches;
import io.trino.plugin.google.sheets.SheetsColumnHandle;
import io.trino.plugin.google.sheets.SheetsConfig;
import io.trino.plugin.google.sheets.SheetsConnectorTableHandle;
import io.trino.plugin.google.sheets.SheetsErrorCode;
import io.trino.plugin.google.sheets.SheetsNamedTableHandle;
import io.trino.plugin.google.sheets.SheetsSheetIdAndRange;
import io.trino.plugin.google.sheets.SheetsSheetTableHandle;
import io.trino.plugin.google.sheets.SheetsTable;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.TrinoException;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarcharType;
import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.util.Base64;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;

public class SheetsClient {
    public static final String DEFAULT_RANGE = "$1:$10000";
    public static final String RANGE_SEPARATOR = "#";
    private static final Logger log = Logger.get(SheetsClient.class);
    private static final String APPLICATION_NAME = "trino google sheets integration";
    private static final JsonFactory JSON_FACTORY = JacksonFactory.getDefaultInstance();
    private static final List<String> SCOPES = ImmutableList.of((Object)"https://www.googleapis.com/auth/spreadsheets");
    private static final String INSERT_VALUE_OPTION = "RAW";
    private static final String INSERT_DATA_OPTION = "INSERT_ROWS";
    private final NonEvictableLoadingCache<String, Optional<String>> tableSheetMappingCache;
    private final LoadingCache<String, List<List<Object>>> sheetDataCache;
    private final Optional<String> metadataSheetId;
    private final Sheets sheetsService;

    @Inject
    public SheetsClient(SheetsConfig config) {
        this.metadataSheetId = config.getMetadataSheetId();
        try {
            this.sheetsService = new Sheets.Builder((HttpTransport)GoogleNetHttpTransport.newTrustedTransport(), JSON_FACTORY, this.setTimeout((HttpRequestInitializer)SheetsClient.getCredentials(config), config)).setApplicationName(APPLICATION_NAME).build();
        }
        catch (IOException | GeneralSecurityException e) {
            throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_BAD_CREDENTIALS_ERROR, (Throwable)e);
        }
        long expiresAfterWriteMillis = config.getSheetsDataExpireAfterWrite().toMillis();
        long maxCacheSize = config.getSheetsDataMaxCacheSize();
        this.tableSheetMappingCache = SafeCaches.buildNonEvictableCache((CacheBuilder)CacheBuilder.newBuilder().expireAfterWrite(expiresAfterWriteMillis, TimeUnit.MILLISECONDS).maximumSize(maxCacheSize), (CacheLoader)new CacheLoader<String, Optional<String>>(){

            public Optional<String> load(String tableName) {
                return SheetsClient.this.getSheetExpressionForTable(tableName);
            }

            public Map<String, Optional<String>> loadAll(Iterable<? extends String> tableList) {
                return SheetsClient.this.getAllTableSheetExpressionMapping();
            }
        });
        this.sheetDataCache = EvictableCacheBuilder.newBuilder().expireAfterWrite(expiresAfterWriteMillis, TimeUnit.MILLISECONDS).maximumSize(maxCacheSize).build(CacheLoader.from(this::readAllValuesFromSheetExpression));
    }

    public Optional<SheetsTable> getTable(SheetsConnectorTableHandle tableHandle) {
        if (tableHandle instanceof SheetsNamedTableHandle) {
            SheetsNamedTableHandle namedTableHandle = (SheetsNamedTableHandle)tableHandle;
            return this.getTable(namedTableHandle.tableName());
        }
        if (tableHandle instanceof SheetsSheetTableHandle) {
            SheetsSheetTableHandle sheetTableHandle = (SheetsSheetTableHandle)tableHandle;
            return this.getTableFromValues(this.readAllValuesFromSheet(sheetTableHandle.getSheetExpression()));
        }
        throw new IllegalStateException("Found unexpected table handle type " + String.valueOf(tableHandle));
    }

    public Optional<SheetsTable> getTable(String tableName) {
        List<List<Object>> values = this.readAllValues(tableName);
        return this.getTableFromValues(values);
    }

    public Optional<SheetsTable> getTableFromValues(List<List<Object>> values) {
        List<List<String>> stringValues = SheetsClient.convertToStringValues(values);
        if (stringValues.size() > 0) {
            ImmutableList.Builder columns = ImmutableList.builder();
            HashSet<Object> columnNames = new HashSet<Object>();
            List<String> header = stringValues.get(0);
            int count = 0;
            for (int i = 0; i < header.size(); ++i) {
                Object columnValue = header.get(i).toLowerCase(Locale.ENGLISH);
                if (((String)columnValue).isEmpty() || columnNames.contains(columnValue)) {
                    columnValue = "column_" + ++count;
                }
                columnNames.add(columnValue);
                columns.add((Object)new SheetsColumnHandle((String)columnValue, (Type)VarcharType.VARCHAR, i));
            }
            List<List<String>> dataValues = stringValues.subList(1, values.size());
            return Optional.of(new SheetsTable((List<SheetsColumnHandle>)columns.build(), dataValues));
        }
        return Optional.empty();
    }

    public Set<String> getTableNames() {
        if (this.metadataSheetId.isEmpty()) {
            return ImmutableSet.of();
        }
        ImmutableSet.Builder tables = ImmutableSet.builder();
        try {
            List tableMetadata = (List)this.sheetDataCache.getUnchecked((Object)this.metadataSheetId.get());
            for (int i = 1; i < tableMetadata.size(); ++i) {
                if (((List)tableMetadata.get(i)).size() <= 0) continue;
                tables.add((Object)String.valueOf(((List)tableMetadata.get(i)).get(0)));
            }
            return tables.build();
        }
        catch (UncheckedExecutionException e) {
            Throwables.throwIfInstanceOf((Throwable)e.getCause(), TrinoException.class);
            throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_METASTORE_ERROR, (Throwable)e);
        }
    }

    public List<List<Object>> readAllValues(String tableName) {
        try {
            String sheetExpression = this.getCachedSheetExpressionForTable(tableName);
            return this.readAllValuesFromSheet(sheetExpression);
        }
        catch (UncheckedExecutionException e) {
            Throwables.throwIfInstanceOf((Throwable)e.getCause(), TrinoException.class);
            throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_TABLE_LOAD_ERROR, "Error loading data for table: " + tableName, (Throwable)e);
        }
    }

    public List<List<Object>> readAllValuesFromSheet(String sheetExpression) {
        try {
            return (List)this.sheetDataCache.getUnchecked((Object)sheetExpression);
        }
        catch (UncheckedExecutionException e) {
            Throwables.throwIfInstanceOf((Throwable)e.getCause(), TrinoException.class);
            throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_TABLE_LOAD_ERROR, "Error loading data for sheet: " + sheetExpression, (Throwable)e);
        }
    }

    public void insertIntoSheet(String sheetExpression, List<List<Object>> rows) {
        ValueRange body = new ValueRange().setValues(rows);
        SheetsSheetIdAndRange sheetIdAndRange = new SheetsSheetIdAndRange(sheetExpression);
        try {
            this.sheetsService.spreadsheets().values().append(sheetIdAndRange.getSheetId(), sheetIdAndRange.getRange(), body).setValueInputOption(INSERT_VALUE_OPTION).setInsertDataOption(INSERT_DATA_OPTION).execute();
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_INSERT_ERROR, "Error inserting data to sheet: ", (Throwable)e);
        }
        this.sheetDataCache.invalidate((Object)sheetExpression);
    }

    public static List<List<String>> convertToStringValues(List<List<Object>> values) {
        return (List)values.stream().map(columns -> (ImmutableList)columns.stream().map(String::valueOf).collect(ImmutableList.toImmutableList())).collect(ImmutableList.toImmutableList());
    }

    private Optional<String> getSheetExpressionForTable(String tableName) {
        Map<String, Optional<String>> tableSheetMap = this.getAllTableSheetExpressionMapping();
        if (!tableSheetMap.containsKey(tableName)) {
            return Optional.empty();
        }
        return tableSheetMap.get(tableName);
    }

    public String getCachedSheetExpressionForTable(String tableName) {
        return (String)((Optional)this.tableSheetMappingCache.getUnchecked((Object)tableName)).orElseThrow(() -> new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_UNKNOWN_TABLE_ERROR, "Sheet expression not found for table " + tableName));
    }

    private Map<String, Optional<String>> getAllTableSheetExpressionMapping() {
        if (this.metadataSheetId.isEmpty()) {
            return ImmutableMap.of();
        }
        ImmutableMap.Builder tableSheetMap = ImmutableMap.builder();
        List<List<Object>> data = this.readAllValuesFromSheetExpression(this.metadataSheetId.get());
        for (int i = 1; i < data.size(); ++i) {
            if (data.get(i).size() < 2) continue;
            String tableId = String.valueOf(data.get(i).get(0));
            String sheetId = String.valueOf(data.get(i).get(1));
            tableSheetMap.put((Object)tableId.toLowerCase(Locale.ENGLISH), Optional.of(sheetId));
        }
        return tableSheetMap.buildOrThrow();
    }

    private static Credential getCredentials(SheetsConfig sheetsConfig) {
        if (sheetsConfig.getCredentialsFilePath().isPresent()) {
            Credential credential;
            FileInputStream in = new FileInputStream(sheetsConfig.getCredentialsFilePath().get());
            try {
                credential = SheetsClient.credentialFromStream(in);
            }
            catch (Throwable throwable) {
                try {
                    try {
                        ((InputStream)in).close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_BAD_CREDENTIALS_ERROR, (Throwable)e);
                }
            }
            ((InputStream)in).close();
            return credential;
        }
        if (sheetsConfig.getCredentialsKey().isPresent()) {
            try {
                return SheetsClient.credentialFromStream(new ByteArrayInputStream(Base64.getDecoder().decode(sheetsConfig.getCredentialsKey().get())));
            }
            catch (IOException e) {
                throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_BAD_CREDENTIALS_ERROR, (Throwable)e);
            }
        }
        throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_BAD_CREDENTIALS_ERROR, "No sheets credentials were provided");
    }

    private static Credential credentialFromStream(InputStream inputStream) throws IOException {
        return GoogleCredential.fromStream((InputStream)inputStream).createScoped(SCOPES);
    }

    private List<List<Object>> readAllValuesFromSheetExpression(String sheetExpression) {
        try {
            String defaultRange = DEFAULT_RANGE;
            String[] tableOptions = sheetExpression.split(RANGE_SEPARATOR);
            String sheetId = tableOptions[0];
            if (tableOptions.length > 1) {
                defaultRange = tableOptions[1];
            }
            log.debug("Accessing sheet id [%s] with range [%s]", new Object[]{sheetId, defaultRange});
            List values = ((ValueRange)this.sheetsService.spreadsheets().values().get(sheetId, defaultRange).execute()).getValues();
            if (values == null) {
                throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_TABLE_LOAD_ERROR, "No non-empty cells found in sheet: " + sheetExpression);
            }
            return values;
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_UNKNOWN_TABLE_ERROR, "Failed reading data from sheet: " + sheetExpression, (Throwable)e);
        }
    }

    private HttpRequestInitializer setTimeout(HttpRequestInitializer requestInitializer, SheetsConfig config) {
        Objects.requireNonNull(config.getConnectionTimeout(), "connectionTimeout is null");
        Objects.requireNonNull(config.getReadTimeout(), "readTimeout is null");
        Objects.requireNonNull(config.getWriteTimeout(), "writeTimeout is null");
        return httpRequest -> {
            requestInitializer.initialize(httpRequest);
            httpRequest.setConnectTimeout(Math.toIntExact(config.getConnectionTimeout().toMillis()));
            httpRequest.setReadTimeout(Math.toIntExact(config.getReadTimeout().toMillis()));
            httpRequest.setWriteTimeout(Math.toIntExact(config.getWriteTimeout().toMillis()));
        };
    }
}

