/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.jdbc;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.base.VerifyException;
import com.google.common.collect.ImmutableList;
import io.airlift.concurrent.MoreFutures;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import io.trino.plugin.jdbc.BaseJdbcConnectorTableHandle;
import io.trino.plugin.jdbc.BooleanReadFunction;
import io.trino.plugin.jdbc.ColumnMapping;
import io.trino.plugin.jdbc.DoubleReadFunction;
import io.trino.plugin.jdbc.JdbcClient;
import io.trino.plugin.jdbc.JdbcColumnHandle;
import io.trino.plugin.jdbc.JdbcErrorCode;
import io.trino.plugin.jdbc.JdbcProcedureHandle;
import io.trino.plugin.jdbc.JdbcSplit;
import io.trino.plugin.jdbc.JdbcTableHandle;
import io.trino.plugin.jdbc.LongReadFunction;
import io.trino.plugin.jdbc.ObjectReadFunction;
import io.trino.plugin.jdbc.ReadFunction;
import io.trino.plugin.jdbc.SliceReadFunction;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.PageBuilder;
import io.trino.spi.TrinoException;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.connector.ConnectorPageSource;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.SourcePage;
import io.trino.spi.type.Type;
import jakarta.annotation.Nullable;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Objects;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicLong;

public final class JdbcPageSource
implements ConnectorPageSource {
    private static final Logger log = Logger.get(JdbcPageSource.class);
    private static final CompletableFuture<ResultSet> UNINITIALIZED_RESULT_SET_FUTURE = CompletableFuture.completedFuture(null);
    private final List<JdbcColumnHandle> columnHandles;
    private final ReadFunction[] readFunctions;
    private final BooleanReadFunction[] booleanReadFunctions;
    private final DoubleReadFunction[] doubleReadFunctions;
    private final LongReadFunction[] longReadFunctions;
    private final SliceReadFunction[] sliceReadFunctions;
    private final ObjectReadFunction[] objectReadFunctions;
    private final JdbcClient jdbcClient;
    private final ExecutorService executor;
    private final Connection connection;
    private final PreparedStatement statement;
    private final AtomicLong readTimeNanos = new AtomicLong(0L);
    private final PageBuilder pageBuilder;
    private CompletableFuture<ResultSet> resultSetFuture;
    @Nullable
    private ResultSet resultSet;
    private boolean finished;
    private boolean closed;
    private long completedPositions;

    public JdbcPageSource(JdbcClient jdbcClient, ExecutorService executor, ConnectorSession session, JdbcSplit split, BaseJdbcConnectorTableHandle table, List<JdbcColumnHandle> columnHandles) {
        this.jdbcClient = Objects.requireNonNull(jdbcClient, "jdbcClient is null");
        this.executor = Objects.requireNonNull(executor, "executor is null");
        this.columnHandles = ImmutableList.copyOf(columnHandles);
        this.readFunctions = new ReadFunction[columnHandles.size()];
        this.booleanReadFunctions = new BooleanReadFunction[columnHandles.size()];
        this.doubleReadFunctions = new DoubleReadFunction[columnHandles.size()];
        this.longReadFunctions = new LongReadFunction[columnHandles.size()];
        this.sliceReadFunctions = new SliceReadFunction[columnHandles.size()];
        this.objectReadFunctions = new ObjectReadFunction[columnHandles.size()];
        try {
            if (table instanceof JdbcProcedureHandle) {
                JdbcProcedureHandle procedureHandle = (JdbcProcedureHandle)table;
                this.connection = jdbcClient.getConnection(session, split, procedureHandle);
            } else {
                this.connection = jdbcClient.getConnection(session, split, (JdbcTableHandle)table);
            }
            for (int i = 0; i < this.columnHandles.size(); ++i) {
                ReadFunction readFunction;
                JdbcColumnHandle columnHandle = columnHandles.get(i);
                ColumnMapping columnMapping = jdbcClient.toColumnMapping(session, this.connection, columnHandle.getJdbcTypeHandle()).orElseThrow(() -> new VerifyException("Column %s has unsupported type %s".formatted(columnHandle.getColumnName(), columnHandle.getJdbcTypeHandle())));
                Verify.verify((boolean)columnHandle.getColumnType().equals((Object)columnMapping.getType()), (String)"Type mismatch: column handle has type %s but %s is mapped to %s", (Object)columnHandle.getColumnType(), (Object)columnHandle.getJdbcTypeHandle(), (Object)columnMapping.getType());
                Class javaType = columnMapping.getType().getJavaType();
                this.readFunctions[i] = readFunction = columnMapping.getReadFunction();
                if (javaType == Boolean.TYPE) {
                    this.booleanReadFunctions[i] = (BooleanReadFunction)readFunction;
                    continue;
                }
                if (javaType == Double.TYPE) {
                    this.doubleReadFunctions[i] = (DoubleReadFunction)readFunction;
                    continue;
                }
                if (javaType == Long.TYPE) {
                    this.longReadFunctions[i] = (LongReadFunction)readFunction;
                    continue;
                }
                if (javaType == Slice.class) {
                    this.sliceReadFunctions[i] = (SliceReadFunction)readFunction;
                    continue;
                }
                this.objectReadFunctions[i] = (ObjectReadFunction)readFunction;
            }
            if (table instanceof JdbcProcedureHandle) {
                JdbcProcedureHandle procedureHandle = (JdbcProcedureHandle)table;
                this.statement = jdbcClient.buildProcedure(session, this.connection, split, procedureHandle);
            } else {
                this.statement = jdbcClient.buildSql(session, this.connection, split, (JdbcTableHandle)table, columnHandles);
            }
            this.pageBuilder = new PageBuilder((List)columnHandles.stream().map(JdbcColumnHandle::getColumnType).collect(ImmutableList.toImmutableList()));
            this.resultSetFuture = UNINITIALIZED_RESULT_SET_FUTURE;
        }
        catch (RuntimeException | SQLException e) {
            throw this.handleSqlException(e);
        }
    }

    public long getReadTimeNanos() {
        return this.readTimeNanos.get();
    }

    public boolean isFinished() {
        return this.finished;
    }

    public SourcePage getNextSourcePage() {
        Verify.verify((boolean)this.pageBuilder.isEmpty(), (String)"Expected pageBuilder to be empty", (Object[])new Object[0]);
        if (this.finished) {
            return null;
        }
        try {
            if (this.resultSetFuture == UNINITIALIZED_RESULT_SET_FUTURE && this.resultSet == null) {
                Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"page source is closed");
                this.resultSetFuture = CompletableFuture.supplyAsync(() -> {
                    long start = System.nanoTime();
                    try {
                        log.debug("Executing: %s", new Object[]{this.statement});
                        ResultSet resultSet = this.statement.executeQuery();
                        return resultSet;
                    }
                    catch (SQLException e) {
                        throw this.handleSqlException(e);
                    }
                    finally {
                        this.readTimeNanos.addAndGet(System.nanoTime() - start);
                    }
                }, this.executor);
            }
            if (this.resultSet == null) {
                if (!this.resultSetFuture.isDone()) {
                    return null;
                }
                this.resultSet = Objects.requireNonNull((ResultSet)MoreFutures.getFutureValue(this.resultSetFuture), "resultSet is null");
            }
            Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"page source is closed");
            while (!this.pageBuilder.isFull() && this.resultSet.next()) {
                this.pageBuilder.declarePosition();
                ++this.completedPositions;
                for (int i = 0; i < this.columnHandles.size(); ++i) {
                    BlockBuilder output = this.pageBuilder.getBlockBuilder(i);
                    Type type = this.columnHandles.get(i).getColumnType();
                    if (this.readFunctions[i].isNull(this.resultSet, i + 1)) {
                        output.appendNull();
                        continue;
                    }
                    if (this.booleanReadFunctions[i] != null) {
                        type.writeBoolean(output, this.booleanReadFunctions[i].readBoolean(this.resultSet, i + 1));
                        continue;
                    }
                    if (this.doubleReadFunctions[i] != null) {
                        type.writeDouble(output, this.doubleReadFunctions[i].readDouble(this.resultSet, i + 1));
                        continue;
                    }
                    if (this.longReadFunctions[i] != null) {
                        type.writeLong(output, this.longReadFunctions[i].readLong(this.resultSet, i + 1));
                        continue;
                    }
                    if (this.sliceReadFunctions[i] != null) {
                        type.writeSlice(output, this.sliceReadFunctions[i].readSlice(this.resultSet, i + 1));
                        continue;
                    }
                    type.writeObject(output, this.objectReadFunctions[i].readObject(this.resultSet, i + 1));
                }
            }
            if (!this.pageBuilder.isFull()) {
                this.finished = true;
            }
        }
        catch (RuntimeException | SQLException e) {
            throw this.handleSqlException(e);
        }
        Page page = this.pageBuilder.build();
        this.pageBuilder.reset();
        return SourcePage.create((Page)page);
    }

    public long getMemoryUsage() {
        return this.pageBuilder.getRetainedSizeInBytes();
    }

    public long getCompletedBytes() {
        return 0L;
    }

    public OptionalLong getCompletedPositions() {
        return OptionalLong.of(this.completedPositions);
    }

    public CompletableFuture<?> isBlocked() {
        return this.resultSetFuture;
    }

    public void close() {
        if (this.closed) {
            return;
        }
        this.closed = true;
        try (Connection connection = this.connection;
             PreparedStatement statement = this.statement;
             ResultSet resultSet = this.resultSet;){
            if (statement != null) {
                try {
                    statement.cancel();
                }
                catch (SQLException sQLException) {
                    // empty catch block
                }
            }
            if (connection != null && resultSet != null) {
                this.jdbcClient.abortReadConnection(connection, resultSet);
                this.resultSetFuture.cancel(true);
            }
        }
        catch (RuntimeException | SQLException exception) {
            // empty catch block
        }
        this.resultSet = null;
    }

    private RuntimeException handleSqlException(Exception e) {
        block2: {
            try {
                this.close();
            }
            catch (Exception closeException) {
                if (e == closeException) break block2;
                e.addSuppressed(closeException);
            }
        }
        return new TrinoException((ErrorCodeSupplier)JdbcErrorCode.JDBC_ERROR, (Throwable)e);
    }
}

