/*
 * Decompiled with CFR 0.152.
 */
package io.trino.spi.type;

import com.google.common.base.Preconditions;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.Int128;
import io.trino.spi.type.Int128Math;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;

public class TestInt128Math {
    private static final Int128 MAX_DECIMAL = Int128.valueOf((BigInteger)Decimals.MAX_UNSCALED_DECIMAL.toBigInteger());
    private static final Int128 MIN_DECIMAL = Int128.valueOf((BigInteger)Decimals.MIN_UNSCALED_DECIMAL.toBigInteger());
    private static final BigInteger TWO = BigInteger.valueOf(2L);

    @Test
    public void testUnscaledBigIntegerToDecimal() {
        TestInt128Math.assertConvertsUnscaledBigIntegerToDecimal(Decimals.MAX_UNSCALED_DECIMAL.toBigInteger());
        TestInt128Math.assertConvertsUnscaledBigIntegerToDecimal(Decimals.MIN_UNSCALED_DECIMAL.toBigInteger());
        TestInt128Math.assertConvertsUnscaledBigIntegerToDecimal(BigInteger.ZERO);
        TestInt128Math.assertConvertsUnscaledBigIntegerToDecimal(BigInteger.ONE);
        TestInt128Math.assertConvertsUnscaledBigIntegerToDecimal(BigInteger.ONE.negate());
    }

    @Test
    public void testUnscaledBigIntegerToInt128Overflow() {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> Int128.valueOf((BigInteger)Int128.MAX_VALUE.toBigInteger().add(BigInteger.ONE))).isInstanceOf(ArithmeticException.class)).hasMessage("BigInteger out of Int128 range");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> Int128.valueOf((BigInteger)Int128.MIN_VALUE.toBigInteger().subtract(BigInteger.ONE))).isInstanceOf(ArithmeticException.class)).hasMessage("BigInteger out of Int128 range");
    }

    @Test
    public void testUnscaledLongToDecimal() {
        TestInt128Math.assertConvertsUnscaledLongToDecimal(0L);
        TestInt128Math.assertConvertsUnscaledLongToDecimal(1L);
        TestInt128Math.assertConvertsUnscaledLongToDecimal(-1L);
        TestInt128Math.assertConvertsUnscaledLongToDecimal(Long.MAX_VALUE);
        TestInt128Math.assertConvertsUnscaledLongToDecimal(Long.MIN_VALUE);
    }

    @Test
    public void testInt128ToUnscaledLongOverflow() {
        TestInt128Math.assertInt128ToLongOverflows(BigInteger.valueOf(Long.MAX_VALUE).add(BigInteger.ONE));
        TestInt128Math.assertInt128ToLongOverflows(BigInteger.valueOf(Long.MIN_VALUE).subtract(BigInteger.ONE));
        TestInt128Math.assertInt128ToLongOverflows(Decimals.MAX_UNSCALED_DECIMAL.toBigInteger());
        TestInt128Math.assertInt128ToLongOverflows(Decimals.MIN_UNSCALED_DECIMAL.toBigInteger());
    }

    @Test
    public void testRescaleTruncate() {
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)10L), 0, Int128.valueOf((long)10L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)-10L), 0, Int128.valueOf((long)-10L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)10L), -20, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)14L), -1, Int128.valueOf((long)1L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)14L), -2, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)14L), -3, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)15L), -1, Int128.valueOf((long)1L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)15L), -2, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)15L), -3, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)1050L), -3, Int128.valueOf((long)1L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)15L), 1, Int128.valueOf((long)150L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)-14L), -1, Int128.valueOf((long)-1L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)-14L), -2, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)-14L), -20, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)-15L), -1, Int128.valueOf((long)-1L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)-15L), -2, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)-15L), -20, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)-14L), 1, Int128.valueOf((long)-140L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)0L), 1, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)0L), -1, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)0L), -20, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)4L), -1, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)5L), -1, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)5L), -2, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)10L), 10, Int128.valueOf((long)100000000000L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((String)"150000000000000000000"), -20, Int128.valueOf((long)1L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((String)"-140000000000000000000"), -20, Int128.valueOf((long)-1L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((String)"50000000000000000000"), -20, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((String)"150500000000000000000"), -18, Int128.valueOf((long)150L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((String)"-140000000000000000000"), -18, Int128.valueOf((long)-140L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((BigInteger)BigInteger.ONE.shiftLeft(63)), -18, Int128.valueOf((long)9L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((BigInteger)BigInteger.ONE.shiftLeft(62)), -18, Int128.valueOf((long)4L));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((BigInteger)BigInteger.ONE.shiftLeft(62)), -19, Int128.valueOf((long)0L));
        TestInt128Math.assertRescaleTruncate(MAX_DECIMAL, -1, Int128.valueOf((String)"9999999999999999999999999999999999999"));
        TestInt128Math.assertRescaleTruncate(MIN_DECIMAL, -10, Int128.valueOf((String)"-9999999999999999999999999999"));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)1L), 37, Int128.valueOf((String)"10000000000000000000000000000000000000"));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((long)-1L), 37, Int128.valueOf((String)"-10000000000000000000000000000000000000"));
        TestInt128Math.assertRescaleTruncate(Int128.valueOf((String)"10000000000000000000000000000000000000"), -37, Int128.valueOf((long)1L));
    }

    @Test
    public void testRescale() {
        TestInt128Math.assertRescale(Int128.valueOf((long)10L), 0, Int128.valueOf((long)10L));
        TestInt128Math.assertRescale(Int128.valueOf((long)-10L), 0, Int128.valueOf((long)-10L));
        TestInt128Math.assertRescale(Int128.valueOf((long)10L), -20, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)14L), -1, Int128.valueOf((long)1L));
        TestInt128Math.assertRescale(Int128.valueOf((long)14L), -2, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)14L), -3, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)15L), -1, Int128.valueOf((long)2L));
        TestInt128Math.assertRescale(Int128.valueOf((long)15L), -2, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)15L), -3, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)1050L), -3, Int128.valueOf((long)1L));
        TestInt128Math.assertRescale(Int128.valueOf((long)15L), 1, Int128.valueOf((long)150L));
        TestInt128Math.assertRescale(Int128.valueOf((long)-14L), -1, Int128.valueOf((long)-1L));
        TestInt128Math.assertRescale(Int128.valueOf((long)-14L), -2, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)-14L), -20, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)-15L), -1, Int128.valueOf((long)-2L));
        TestInt128Math.assertRescale(Int128.valueOf((long)-15L), -2, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)-15L), -20, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)-14L), 1, Int128.valueOf((long)-140L));
        TestInt128Math.assertRescale(Int128.valueOf((long)0L), 1, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)0L), -1, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)0L), -20, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)4L), -1, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)5L), -1, Int128.valueOf((long)1L));
        TestInt128Math.assertRescale(Int128.valueOf((long)5L), -2, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(Int128.valueOf((long)10L), 10, Int128.valueOf((long)100000000000L));
        TestInt128Math.assertRescale(Int128.valueOf((String)"150000000000000000000"), -20, Int128.valueOf((long)2L));
        TestInt128Math.assertRescale(Int128.valueOf((String)"-140000000000000000000"), -20, Int128.valueOf((long)-1L));
        TestInt128Math.assertRescale(Int128.valueOf((String)"50000000000000000000"), -20, Int128.valueOf((long)1L));
        TestInt128Math.assertRescale(Int128.valueOf((String)"150500000000000000000"), -18, Int128.valueOf((long)151L));
        TestInt128Math.assertRescale(Int128.valueOf((String)"-140000000000000000000"), -18, Int128.valueOf((long)-140L));
        TestInt128Math.assertRescale(Int128.valueOf((BigInteger)BigInteger.ONE.shiftLeft(63)), -18, Int128.valueOf((long)9L));
        TestInt128Math.assertRescale(Int128.valueOf((BigInteger)BigInteger.ONE.shiftLeft(62)), -18, Int128.valueOf((long)5L));
        TestInt128Math.assertRescale(Int128.valueOf((BigInteger)BigInteger.ONE.shiftLeft(62)), -19, Int128.valueOf((long)0L));
        TestInt128Math.assertRescale(MAX_DECIMAL, -1, Int128.valueOf((BigInteger)Decimals.MAX_UNSCALED_DECIMAL.toBigInteger().divide(BigInteger.TEN).add(BigInteger.ONE)));
        TestInt128Math.assertRescale(MIN_DECIMAL, -10, Int128.valueOf((BigInteger)Decimals.MIN_UNSCALED_DECIMAL.toBigInteger().divide(BigInteger.valueOf(10000000000L)).subtract(BigInteger.ONE)));
        TestInt128Math.assertRescale(Int128.valueOf((long)1L), 37, Int128.valueOf((String)"10000000000000000000000000000000000000"));
        TestInt128Math.assertRescale(Int128.valueOf((long)-1L), 37, Int128.valueOf((String)"-10000000000000000000000000000000000000"));
        TestInt128Math.assertRescale(Int128.valueOf((String)"10000000000000000000000000000000000000"), -37, Int128.valueOf((long)1L));
    }

    @Test
    public void testRescaleOverflows() {
        TestInt128Math.assertRescaleOverflows(Int128.valueOf((long)1L), 39);
    }

    @Test
    public void testAdd() {
        this.assertAdd(Int128.valueOf((long)0L), Int128.valueOf((long)0L), Int128.valueOf((long)0L));
        this.assertAdd(Int128.valueOf((long)1L), Int128.valueOf((long)0L), Int128.valueOf((long)1L));
        this.assertAdd(Int128.valueOf((long)1L), Int128.valueOf((long)1L), Int128.valueOf((long)2L));
        this.assertAdd(Int128.valueOf((long)-1L), Int128.valueOf((long)0L), Int128.valueOf((long)-1L));
        this.assertAdd(Int128.valueOf((long)-1L), Int128.valueOf((long)-1L), Int128.valueOf((long)-2L));
        this.assertAdd(Int128.valueOf((long)-1L), Int128.valueOf((long)1L), Int128.valueOf((long)0L));
        this.assertAdd(Int128.valueOf((long)1L), Int128.valueOf((long)-1L), Int128.valueOf((long)0L));
        this.assertAdd(Int128.valueOf((String)"10000000000000000000000000000000000000"), Int128.valueOf((long)0L), Int128.valueOf((String)"10000000000000000000000000000000000000"));
        this.assertAdd(Int128.valueOf((String)"10000000000000000000000000000000000000"), Int128.valueOf((String)"10000000000000000000000000000000000000"), Int128.valueOf((String)"20000000000000000000000000000000000000"));
        this.assertAdd(Int128.valueOf((String)"-10000000000000000000000000000000000000"), Int128.valueOf((long)0L), Int128.valueOf((String)"-10000000000000000000000000000000000000"));
        this.assertAdd(Int128.valueOf((String)"-10000000000000000000000000000000000000"), Int128.valueOf((String)"-10000000000000000000000000000000000000"), Int128.valueOf((String)"-20000000000000000000000000000000000000"));
        this.assertAdd(Int128.valueOf((String)"-10000000000000000000000000000000000000"), Int128.valueOf((String)"10000000000000000000000000000000000000"), Int128.valueOf((long)0L));
        this.assertAdd(Int128.valueOf((String)"10000000000000000000000000000000000000"), Int128.valueOf((String)"-10000000000000000000000000000000000000"), Int128.valueOf((long)0L));
        this.assertAdd(Int128.valueOf((long)0x100000000L), Int128.valueOf((long)0L), Int128.valueOf((long)0x100000000L));
        this.assertAdd(Int128.valueOf((long)0x80000000L), Int128.valueOf((long)0x80000000L), Int128.valueOf((long)0x100000000L));
        this.assertAdd(Int128.valueOf((long)0x100000000L), Int128.valueOf((long)0x200000000L), Int128.valueOf((long)0x300000000L));
    }

    @Test
    public void testMultiply() {
        TestInt128Math.assertMultiply(0L, 0L, 0L);
        TestInt128Math.assertMultiply(1L, 0L, 0L);
        TestInt128Math.assertMultiply(0L, 1L, 0L);
        TestInt128Math.assertMultiply(-1L, 0L, 0L);
        TestInt128Math.assertMultiply(0L, -1L, 0L);
        TestInt128Math.assertMultiply(1L, 1L, 1L);
        TestInt128Math.assertMultiply(1L, -1L, -1L);
        TestInt128Math.assertMultiply(-1L, -1L, 1L);
        TestInt128Math.assertMultiply(Decimals.MAX_UNSCALED_DECIMAL.toBigInteger(), 0L, 0L);
        TestInt128Math.assertMultiply(Decimals.MAX_UNSCALED_DECIMAL.toBigInteger(), 1L, Decimals.MAX_UNSCALED_DECIMAL.toBigInteger());
        TestInt128Math.assertMultiply(Decimals.MIN_UNSCALED_DECIMAL.toBigInteger(), 0L, 0L);
        TestInt128Math.assertMultiply(Decimals.MIN_UNSCALED_DECIMAL.toBigInteger(), 1L, Decimals.MIN_UNSCALED_DECIMAL.toBigInteger());
        TestInt128Math.assertMultiply(Decimals.MAX_UNSCALED_DECIMAL.toBigInteger(), -1L, Decimals.MIN_UNSCALED_DECIMAL.toBigInteger());
        TestInt128Math.assertMultiply(Decimals.MIN_UNSCALED_DECIMAL.toBigInteger(), -1L, Decimals.MAX_UNSCALED_DECIMAL.toBigInteger());
        TestInt128Math.assertMultiply(new BigInteger("FFFFFFFFFFFFFFFF", 16), new BigInteger("FFFFFFFFFFFFFF", 16), new BigInteger("fffffffffffffeff00000000000001", 16));
        TestInt128Math.assertMultiply(new BigInteger("FFFFFF0096BFB800", 16), new BigInteger("39003539D9A51600", 16), new BigInteger("39003500FB00AB761CDBB17E11D00000", 16));
        TestInt128Math.assertMultiply(Integer.MAX_VALUE, Integer.MIN_VALUE, -4611686016279904256L);
        TestInt128Math.assertMultiply(new BigInteger("99999999999999"), new BigInteger("-1000000000000000000000000"), new BigInteger("-99999999999999000000000000000000000000"));
        TestInt128Math.assertMultiply(new BigInteger("12380837221737387489365741632769922889"), 3L, new BigInteger("37142511665212162468097224898309768667"));
    }

    @Test
    public void testMultiply256() {
        TestInt128Math.assertMultiply256(MAX_DECIMAL, MAX_DECIMAL, new int[]{1, -320095360, -1131968757, -520156000, -396023432, 124040363, 295245237, 370920615});
        TestInt128Math.assertMultiply256(Int128.valueOf((long)0xFFFFFFFFFFFFFFFL, (long)-1L), Int128.valueOf((long)0xFFFFFFFFFFFFFFFL, (long)-1L), new int[]{1, 0, 0, -536870912, -1, -1, -1, 0xFFFFFF});
        TestInt128Math.assertMultiply256(Int128.valueOf((long)1070935361496367905L, (long)1311768467294899695L), Int128.valueOf((long)81985529205931230L, (long)-81986143110479071L), new int[]{-447342385, -1035320273, 228892122, -1435392499, 333416085, 2081832282, -105049452, 1108206});
    }

    private static void assertMultiply256(Int128 left, Int128 right, int[] expected) {
        int[] leftArg = new int[8];
        leftArg[0] = (int)left.getLow();
        leftArg[1] = (int)(left.getLow() >> 32);
        leftArg[2] = (int)left.getHigh();
        leftArg[3] = (int)(left.getHigh() >> 32);
        Int128Math.multiply256Destructive((int[])leftArg, (Int128)right);
        Assertions.assertThat((int[])leftArg).isEqualTo((Object)expected);
    }

    @Test
    public void testMultiplyOverflow() {
        TestInt128Math.assertMultiplyOverflows(Int128.valueOf((String)"99999999999999"), Int128.valueOf((String)"-10000000000000000000000000"));
        TestInt128Math.assertMultiplyOverflows(MAX_DECIMAL, Int128.valueOf((String)"10"));
        TestInt128Math.assertMultiplyOverflows(Int128.valueOf((String)"18446744073709551616"), Int128.valueOf((String)"18446744073709551616"));
        TestInt128Math.assertMultiplyOverflows(Int128.valueOf((String)"18446744073709551615"), Int128.valueOf((String)"18446744073709551615"));
        TestInt128Math.assertMultiplyOverflows(Int128.valueOf((String)"85070591730234615865843651857942052864"), Int128.valueOf((String)"2"));
        TestInt128Math.assertMultiplyOverflows(Int128.valueOf((String)"2"), Int128.valueOf((String)"85070591730234615865843651857942052864"));
        TestInt128Math.assertMultiplyOverflows(Int128.valueOf((String)"-18446744073709551616"), Int128.valueOf((String)"18446744073709551616"));
        TestInt128Math.assertMultiplyOverflows(Int128.valueOf((String)"-18446744073709551615"), Int128.valueOf((String)"18446744073709551615"));
        TestInt128Math.assertMultiplyOverflows(Int128.valueOf((String)"85070591730234615865843651857942052864"), Int128.valueOf((String)"-2"));
        TestInt128Math.assertMultiplyOverflows(Int128.valueOf((String)"-2"), Int128.valueOf((String)"85070591730234615865843651857942052864"));
    }

    @Test
    public void testShiftRight() {
        TestInt128Math.assertShiftRight(Int128.valueOf((long)0L), 0, true, Int128.valueOf((long)0L));
        TestInt128Math.assertShiftRight(Int128.valueOf((long)0L), 33, true, Int128.valueOf((long)0L));
        TestInt128Math.assertShiftRight(Int128.valueOf((long)1L), 1, true, Int128.valueOf((long)1L));
        TestInt128Math.assertShiftRight(Int128.valueOf((long)1L), 1, false, Int128.valueOf((long)0L));
        TestInt128Math.assertShiftRight(Int128.valueOf((long)1L), 2, true, Int128.valueOf((long)0L));
        TestInt128Math.assertShiftRight(Int128.valueOf((long)1L), 2, false, Int128.valueOf((long)0L));
        TestInt128Math.assertShiftRight(Int128.valueOf((long)0x100000000L), 32, true, Int128.valueOf((long)1L));
        TestInt128Math.assertShiftRight(Int128.valueOf((long)0x80000000L), 32, true, Int128.valueOf((long)1L));
        TestInt128Math.assertShiftRight(Int128.valueOf((long)0x80000000L), 32, false, Int128.valueOf((long)0L));
        TestInt128Math.assertShiftRight(Int128.valueOf((long)0x600000000L), 34, true, Int128.valueOf((long)2L));
        TestInt128Math.assertShiftRight(Int128.valueOf((long)0x600000000L), 34, false, Int128.valueOf((long)1L));
        TestInt128Math.assertShiftRight(Int128.valueOf((BigInteger)BigInteger.valueOf(Long.MAX_VALUE).setBit(63).setBit(64)), 1, true, Int128.valueOf((BigInteger)BigInteger.ONE.shiftLeft(64)));
        TestInt128Math.assertShiftRight(MAX_DECIMAL, 1, true, Int128.valueOf((BigInteger)MAX_DECIMAL.toBigInteger().shiftRight(1).add(BigInteger.ONE)));
        TestInt128Math.assertShiftRight(MAX_DECIMAL, 66, true, Int128.valueOf((BigInteger)MAX_DECIMAL.toBigInteger().shiftRight(66).add(BigInteger.ONE)));
    }

    @Test
    public void testDivide() {
        TestInt128Math.assertDivideAllSigns("0", "10");
        TestInt128Math.assertDivideAllSigns("5", "10");
        TestInt128Math.assertDivideAllSigns("50", "100");
        TestInt128Math.assertDivideAllSigns("99", "10");
        TestInt128Math.assertDivideAllSigns("95", "10");
        TestInt128Math.assertDivideAllSigns("91", "10");
        TestInt128Math.assertDivideAllSigns("1000000000000000000000000", "10");
        TestInt128Math.assertDivideAllSigns("1000000000000000000000000", "3");
        TestInt128Math.assertDivideAllSigns("1000000000000000000000000", "9");
        TestInt128Math.assertDivideAllSigns("1000000000000000000000000", "100000000000000000000000");
        TestInt128Math.assertDivideAllSigns("1000000000000000000000000", "333333333333333333333333");
        TestInt128Math.assertDivideAllSigns("1000000000000000000000000", "111111111111111111111111");
        TestInt128Math.assertDivideAllSigns(new int[]{4, 3, 2, 0}, new int[]{4, 3, 2, 1});
        TestInt128Math.assertDivideAllSigns(new int[]{4, 3, 0, 0}, new int[]{4, 3, 2, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{4, 0, 0, 0}, new int[]{4, 3, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, 0, 0, 0}, new int[]{4, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{1423957378, 1765820914, -1, 0}, new int[]{4, 65535, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{1423957378, 1765820914, -1, 0}, new int[]{2042457708, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{1423957378, -925263858, 0, 0}, new int[]{2042457708, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{-1, 0, 0, 0}, new int[]{2042457708, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{1423957378, -1444436990, -925263858, 1106345725}, new int[]{2042457708, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, -150994944, 0, 0x39000000}, new int[]{-1765820914, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{267387408, -150990288, -83842048, 956314880}, new int[]{-1765820914, 2042457708, -1, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{267387408, -150990288, -83842048, 956314880}, new int[]{-1765820914, -256, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{267387408, -150990288, -83842048, 956314880}, new int[]{-1765820914, -16777216, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{267387408, -150990288, -83842048, 956314880}, new int[]{-1765820914, 2042457708, -1, Integer.MAX_VALUE});
        TestInt128Math.assertDivideAllSigns(new int[]{267387408, -150990288, -83842048, 956314880}, new int[]{-1765820914, 2042457708, 0x4FFFFFFF, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{267387408, -150990288, -83842048, 956314880}, new int[]{-1765820914, 2042457708, 65535, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{1, 1, 1, Integer.MAX_VALUE}, new int[]{-1, 1, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, -1879048193, -1879048193, 0}, new int[]{-1, -1879048193, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{1, 1, -1, 0}, new int[]{-1, Integer.MAX_VALUE, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{1, 1, -1, 0}, new int[]{-1, Integer.MAX_VALUE, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{3, 0, Integer.MIN_VALUE, 0}, new int[]{1, 0, 0x20000000, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{3, 0, 32768, 0}, new int[]{1, 0, 8192, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, 0, 32768, Short.MAX_VALUE}, new int[]{1, 0, 32768, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{3, 0, 0, 0}, new int[]{2, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{3, 0, 0, 0}, new int[]{3, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{3, 0, 0, 0}, new int[]{4, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{3, 0, 0, 0}, new int[]{-1, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{-1, 0, 0, 0}, new int[]{1, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{-1, 0, 0, 0}, new int[]{-1, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{-1, 0, 0, 0}, new int[]{3, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{-1, -1, 0, 0}, new int[]{1, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{-1, -1, 0, 0}, new int[]{-1, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{-1, -2, 0, 0}, new int[]{-1, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{22136, 4660, 0, 0}, new int[]{39612, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, 0, 0, 0}, new int[]{0, 1, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, 7, 0, 0}, new int[]{0, 3, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{5, 7, 0, 0}, new int[]{0, 3, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, 6, 0, 0}, new int[]{0, 2, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, Integer.MIN_VALUE, 0, 0}, new int[]{0x40000001, 0, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, Integer.MIN_VALUE, 0, 0}, new int[]{1, 0x40000000, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{30874, 48350, 0, 0}, new int[]{30874, 48350, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{30875, 48350, 0, 0}, new int[]{30874, 48350, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{30873, 48350, 0, 0}, new int[]{30874, 48350, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{65535, 65535, 0, 0}, new int[]{65535, 65535, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{65535, 65535, 0, 0}, new int[]{0, 65535, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{35243, 17767, 291, 0}, new int[]{0, 1, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{35243, 17767, 291, 0}, new int[]{0, 1, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, 65534, 32768, 0}, new int[]{65535, 32768, 0, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{3, 0, Integer.MIN_VALUE, 0}, new int[]{1, 0, 0x20000000, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{3, 0, 32768, 0}, new int[]{1, 0, 8192, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, 0, 32768, Short.MAX_VALUE}, new int[]{1, 0, 32768, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, 65534, 0, 32768}, new int[]{65535, 0, 32768, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, -2, 0, Integer.MIN_VALUE}, new int[]{65535, 0, Integer.MIN_VALUE, 0});
        TestInt128Math.assertDivideAllSigns(new int[]{0, -2, 0, Integer.MIN_VALUE}, new int[]{-1, 0, Integer.MIN_VALUE, 0});
        TestInt128Math.assertDivideAllSigns("100000000000000000000000", 10, "111111111111111111111111", 10);
        TestInt128Math.assertDivideAllSigns("100000000000000000000000", 10, "111111111111", 22);
        TestInt128Math.assertDivideAllSigns("99999999999999999999999999999999999999", 37, "99999999999999999999999999999999999999", 37);
        TestInt128Math.assertDivideAllSigns("99999999999999999999999999999999999999", 2, "99999999999999999999999999999999999999", 1);
        TestInt128Math.assertDivideAllSigns("99999999999999999999999999999999999999", 37, "9", 37);
        TestInt128Math.assertDivideAllSigns("99999999999999999999999999999999999999", 37, "1", 37);
        TestInt128Math.assertDivideAllSigns("11111111111111111111111111111111111111", 37, "2", 37);
        TestInt128Math.assertDivideAllSigns("11111111111111111111111111111111111111", 37, "2", 1);
        TestInt128Math.assertDivideAllSigns("97764425639372288753711864842425458618", 36, "32039006229599111733094986468789901155", 0);
        TestInt128Math.assertDivideAllSigns("34354576602352622842481633786816220283", 0, "31137583115118564930544829855652258045", 0);
        TestInt128Math.assertDivideAllSigns("96690614752287690630596513604374991473", 0, "10039352042372909488692220528497751229", 0);
        TestInt128Math.assertDivideAllSigns("87568357716090115374029040878755891076", 0, "46106713604991337798209343815577148589", 0);
    }

    @Test
    public void testCompare() {
        TestInt128Math.assertCompare(Int128.valueOf((long)0L), Int128.valueOf((long)0L), 0);
        TestInt128Math.assertCompare(Int128.valueOf((long)0L), Int128.valueOf((long)10L), -1);
        TestInt128Math.assertCompare(Int128.valueOf((long)10L), Int128.valueOf((long)0L), 1);
        TestInt128Math.assertCompare(Int128.valueOf((long)-10L), Int128.valueOf((long)-11L), 1);
        TestInt128Math.assertCompare(Int128.valueOf((long)-11L), Int128.valueOf((long)-11L), 0);
        TestInt128Math.assertCompare(Int128.valueOf((long)-12L), Int128.valueOf((long)-11L), -1);
        TestInt128Math.assertCompare(Int128.valueOf((long)10L), Int128.valueOf((long)11L), -1);
        TestInt128Math.assertCompare(Int128.valueOf((long)11L), Int128.valueOf((long)11L), 0);
        TestInt128Math.assertCompare(Int128.valueOf((long)12L), Int128.valueOf((long)11L), 1);
    }

    @Test
    public void testNegate() {
        Assertions.assertThat((Comparable)Int128Math.negate((Int128)Int128Math.negate((Int128)MIN_DECIMAL))).isEqualTo((Object)MIN_DECIMAL);
        Assertions.assertThat((Comparable)Int128Math.negate((Int128)MIN_DECIMAL)).isEqualTo((Object)MAX_DECIMAL);
        Assertions.assertThat((Comparable)Int128Math.negate((Int128)MIN_DECIMAL)).isEqualTo((Object)MAX_DECIMAL);
        Assertions.assertThat((Comparable)Int128Math.negate((Int128)Int128.valueOf((long)1L))).isEqualTo((Object)Int128.valueOf((long)-1L));
        Assertions.assertThat((Comparable)Int128Math.negate((Int128)Int128.valueOf((long)-1L))).isEqualTo((Object)Int128.valueOf((long)1L));
    }

    @Test
    public void testIsNegative() {
        Assertions.assertThat((boolean)MIN_DECIMAL.isNegative()).isTrue();
        Assertions.assertThat((boolean)MAX_DECIMAL.isNegative()).isFalse();
        Assertions.assertThat((boolean)Int128.ZERO.isNegative()).isFalse();
    }

    @Test
    public void testToString() {
        Assertions.assertThat((String)Int128.ZERO.toString()).isEqualTo("0");
        Assertions.assertThat((String)Int128.valueOf((long)1L).toString()).isEqualTo("1");
        Assertions.assertThat((String)Int128.valueOf((long)-1L).toString()).isEqualTo("-1");
        Assertions.assertThat((String)MAX_DECIMAL.toString()).isEqualTo(Decimals.MAX_UNSCALED_DECIMAL.toBigInteger().toString());
        Assertions.assertThat((String)MIN_DECIMAL.toString()).isEqualTo(Decimals.MIN_UNSCALED_DECIMAL.toBigInteger().toString());
        Assertions.assertThat((String)Int128.valueOf((String)"1000000000000000000000000000000000000").toString()).isEqualTo("1000000000000000000000000000000000000");
        Assertions.assertThat((String)Int128.valueOf((String)"-1000000000002000000000000300000000000").toString()).isEqualTo("-1000000000002000000000000300000000000");
    }

    @Test
    public void testShiftLeftMultiPrecision() {
        Assertions.assertThat((int[])Int128Math.shiftLeftMultiPrecision((int[])new int[]{-1589272251, 1452717397, 1392048298, -16667989, 0}, (int)4, (int)0)).isEqualTo((Object)new int[]{-1589272251, 1452717397, 1392048298, -16667989, 0});
        Assertions.assertThat((int[])Int128Math.shiftLeftMultiPrecision((int[])new int[]{-1589272251, 1452717397, 1392048298, -16667989, 0}, (int)5, (int)1)).isEqualTo((Object)new int[]{1116422794, -1389532501, -1510870700, -33335978, 1});
        Assertions.assertThat((int[])Int128Math.shiftLeftMultiPrecision((int[])new int[]{-1589272251, 1452717397, 1392048298, -16667989, 0}, (int)5, (int)31)).isEqualTo((Object)new int[]{Integer.MIN_VALUE, -794636126, 726358698, -1451459499, 2139149653});
        Assertions.assertThat((int[])Int128Math.shiftLeftMultiPrecision((int[])new int[]{-1589272251, 1452717397, 1392048298, -16667989, 0}, (int)5, (int)32)).isEqualTo((Object)new int[]{0, -1589272251, 1452717397, 1392048298, -16667989});
        Assertions.assertThat((int[])Int128Math.shiftLeftMultiPrecision((int[])new int[]{-1589272251, 1452717397, 1392048298, -16667989, 0, 0}, (int)6, (int)33)).isEqualTo((Object)new int[]{0, 1116422794, -1389532501, -1510870700, -33335978, 1});
        Assertions.assertThat((int[])Int128Math.shiftLeftMultiPrecision((int[])new int[]{-1589272251, 1452717397, 1392048298, -16667989, 0, 0}, (int)6, (int)37)).isEqualTo((Object)new int[]{0, 682895520, -757683532, 1595872586, -533375638, 31});
        Assertions.assertThat((int[])Int128Math.shiftLeftMultiPrecision((int[])new int[]{-1589272251, 1452717397, 1392048298, -16667989, 0, 0}, (int)6, (int)64)).isEqualTo((Object)new int[]{0, 0, -1589272251, 1452717397, 1392048298, -16667989});
    }

    @Test
    public void testShiftRightMultiPrecision() {
        Assertions.assertThat((int[])Int128Math.shiftRightMultiPrecision((int[])new int[]{-1589272251, 1452717397, 1392048298, -16667989, 0}, (int)4, (int)0)).isEqualTo((Object)new int[]{-1589272251, 1452717397, 1392048298, -16667989, 0});
        Assertions.assertThat((int[])Int128Math.shiftRightMultiPrecision((int[])new int[]{0, -1589272251, 1452717397, 1392048298, -16667989}, (int)5, (int)1)).isEqualTo((Object)new int[]{Integer.MIN_VALUE, -794636126, 726358698, -1451459499, 2139149653});
        Assertions.assertThat((int[])Int128Math.shiftRightMultiPrecision((int[])new int[]{0, -1589272251, 1452717397, 1392048298, -16667989}, (int)5, (int)32)).isEqualTo((Object)new int[]{-1589272251, 1452717397, 1392048298, -16667989, 0});
        Assertions.assertThat((int[])Int128Math.shiftRightMultiPrecision((int[])new int[]{0, 0, -1589272251, 1452717397, 1392048298, -16667989}, (int)6, (int)33)).isEqualTo((Object)new int[]{Integer.MIN_VALUE, -794636126, 726358698, -1451459499, 2139149653, 0});
        Assertions.assertThat((int[])Int128Math.shiftRightMultiPrecision((int[])new int[]{0, 0, -1589272251, 1452717397, 1392048298, -16667989}, (int)6, (int)37)).isEqualTo((Object)new int[]{0x28000000, -1391842038, 1387574698, 1519896517, 133696853, 0});
        Assertions.assertThat((int[])Int128Math.shiftRightMultiPrecision((int[])new int[]{0, 0, -1589272251, 1452717397, 1392048298, -16667989}, (int)6, (int)64)).isEqualTo((Object)new int[]{-1589272251, 1452717397, 1392048298, -16667989, 0, 0});
    }

    @Test
    public void testShiftLeft() {
        Assertions.assertThat((Comparable)TestInt128Math.shiftLeft(Int128.valueOf((long)-1162850053679398111L, (long)1311768467294899695L), 0)).isEqualTo((Object)Int128.valueOf((long)-1162850053679398111L, (long)1311768467294899695L));
        Assertions.assertThat((Comparable)TestInt128Math.shiftLeft(Int128.valueOf((long)-1162850053679398111L, (long)1311768467294899695L), 1)).isEqualTo((Object)Int128.valueOf((long)-2325700107358796222L, (long)2623536934589799390L));
        Assertions.assertThat((Comparable)TestInt128Math.shiftLeft(Int128.valueOf((long)62129044965376801L, (long)1311768467294899695L), 8)).isEqualTo((Object)Int128.valueOf((long)-2541708562573090542L, (long)3771334300722392832L));
        Assertions.assertThat((Comparable)TestInt128Math.shiftLeft(Int128.valueOf((long)204550089032481L, (long)1311768467294899695L), 16)).isEqualTo((Object)Int128.valueOf((long)-5041349438876872140L, (long)6230889152035880960L));
        Assertions.assertThat((Comparable)TestInt128Math.shiftLeft(Int128.valueOf((long)2271560481L, (long)1311768467294899695L), 32)).isEqualTo((Object)Int128.valueOf((long)-8690466096623102344L, (long)-8022091884849528832L));
        Assertions.assertThat((Comparable)TestInt128Math.shiftLeft(Int128.valueOf((long)0L, (long)1311768467294899695L), 64)).isEqualTo((Object)Int128.valueOf((long)1311768467294899695L, (long)0L));
        Assertions.assertThat((Comparable)TestInt128Math.shiftLeft(Int128.valueOf((long)0L, (long)14731774612196847L), 72)).isEqualTo((Object)Int128.valueOf((long)3771334300722392832L, (long)0L));
        Assertions.assertThat((Comparable)TestInt128Math.shiftLeft(Int128.valueOf((long)0L, (long)52719L), 112)).isEqualTo((Object)Int128.valueOf((long)-3607664776500477952L, (long)0L));
        Assertions.assertThat((Comparable)TestInt128Math.shiftLeft(Int128.valueOf((long)0L, (long)1L), 127)).isEqualTo((Object)Int128.valueOf((long)Long.MIN_VALUE, (long)0L));
        this.assertShiftLeft(new BigInteger("446319580078125"), 19);
        this.assertShiftLeft(TWO.pow(1), 10);
        this.assertShiftLeft(TWO.pow(5).add(TWO.pow(1)), 10);
        this.assertShiftLeft(TWO.pow(1), 100);
        this.assertShiftLeft(TWO.pow(5).add(TWO.pow(1)), 100);
        this.assertShiftLeft(TWO.pow(70), 30);
        this.assertShiftLeft(TWO.pow(70).add(TWO.pow(1)), 30);
        this.assertShiftLeft(TWO.pow(106), 20);
        this.assertShiftLeft(TWO.pow(106).add(TWO.pow(1)), 20);
    }

    @Test
    public void testAbsExact() {
        Assertions.assertThat((Comparable)Int128Math.absExact((Int128)Int128.ZERO)).isEqualTo((Object)Int128.ZERO);
        Assertions.assertThat((Comparable)Int128Math.absExact((Int128)Int128.valueOf((long)1L))).isEqualTo((Object)Int128.valueOf((long)1L));
        Assertions.assertThat((Comparable)Int128Math.absExact((Int128)Int128.valueOf((long)-1L))).isEqualTo((Object)Int128.valueOf((long)1L));
        Assertions.assertThat((Comparable)Int128Math.absExact((Int128)Int128.MAX_VALUE)).isEqualTo((Object)Int128.MAX_VALUE);
        Assertions.assertThat((Comparable)Int128Math.absExact((Int128)Int128.valueOf((long)-4611686018427387904L, (long)0L))).isEqualTo((Object)Int128Math.negate((Int128)Int128.valueOf((long)-4611686018427387904L, (long)0L)));
        Assertions.assertThatThrownBy(() -> Int128Math.absExact((Int128)Int128.MIN_VALUE)).isInstanceOf(ArithmeticException.class);
    }

    private void assertAdd(Int128 left, Int128 right, Int128 result) {
        long[] resultArray = new long[2];
        Int128Math.add((long)left.getHigh(), (long)left.getLow(), (long)right.getHigh(), (long)right.getLow(), (long[])resultArray, (int)0);
        Assertions.assertThat((BigInteger)Int128.valueOf((long)resultArray[0], (long)resultArray[1]).toBigInteger()).isEqualTo((Object)result.toBigInteger());
    }

    private static void assertInt128ToLongOverflows(BigInteger value) {
        Int128 decimal = Int128.valueOf((BigInteger)value);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((Int128)decimal).toLongExact()).isInstanceOf(ArithmeticException.class)).hasMessage("Overflow");
    }

    private static void assertMultiplyOverflows(Int128 left, Int128 right) {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> Int128Math.multiply((Int128)left, (Int128)right)).isInstanceOf(ArithmeticException.class)).hasMessage("Overflow");
    }

    private static void assertRescaleOverflows(Int128 decimal, int rescaleFactor) {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> Int128Math.rescale((Int128)decimal, (int)rescaleFactor)).isInstanceOf(ArithmeticException.class)).hasMessage("Overflow");
    }

    private static void assertCompare(Int128 left, Int128 right, int expectedResult) {
        Assertions.assertThat((int)left.compareTo(right)).isEqualTo(expectedResult);
        Assertions.assertThat((int)Int128.compare((long)left.getHigh(), (long)left.getLow(), (long)right.getHigh(), (long)right.getLow())).isEqualTo(expectedResult);
    }

    private static void assertConvertsUnscaledBigIntegerToDecimal(BigInteger value) {
        Assertions.assertThat((BigInteger)Int128.valueOf((BigInteger)value).toBigInteger()).isEqualTo((Object)value);
    }

    private static void assertConvertsUnscaledLongToDecimal(long value) {
        Assertions.assertThat((long)Int128.valueOf((long)value).toLongExact()).isEqualTo(value);
        Assertions.assertThat((Comparable)Int128.valueOf((long)value)).isEqualTo((Object)Int128.valueOf((BigInteger)BigInteger.valueOf(value)));
    }

    private static void assertShiftRight(Int128 decimal, int rightShifts, boolean roundUp, Int128 expectedResult) {
        long[] result = new long[2];
        Int128Math.shiftRight((long)decimal.getHigh(), (long)decimal.getLow(), (int)rightShifts, (boolean)roundUp, (long[])result, (int)0);
        Assertions.assertThat((Comparable)Int128.valueOf((long[])result)).isEqualTo((Object)expectedResult);
    }

    private static void assertDivideAllSigns(int[] dividend, int[] divisor) {
        TestInt128Math.assertDivideAllSigns(TestInt128Math.valueOf(dividend), 0, TestInt128Math.valueOf(divisor), 0);
    }

    private void assertShiftLeft(BigInteger value, int leftShifts) {
        Int128 decimal = Int128.valueOf((BigInteger)value);
        BigInteger expectedResult = value.multiply(TWO.pow(leftShifts));
        decimal = TestInt128Math.shiftLeft(decimal, leftShifts);
        Assertions.assertThat((BigInteger)decimal.toBigInteger()).isEqualTo((Object)expectedResult);
    }

    private static void assertDivideAllSigns(String dividend, String divisor) {
        TestInt128Math.assertDivideAllSigns(dividend, 0, divisor, 0);
    }

    private static void assertDivideAllSigns(String dividend, int dividendRescaleFactor, String divisor, int divisorRescaleFactor) {
        TestInt128Math.assertDivideAllSigns(Int128.valueOf((String)dividend), dividendRescaleFactor, Int128.valueOf((String)divisor), divisorRescaleFactor);
    }

    private static void assertDivideAllSigns(Int128 dividend, int dividendRescaleFactor, Int128 divisor, int divisorRescaleFactor) {
        TestInt128Math.assertDivide(dividend, dividendRescaleFactor, divisor, divisorRescaleFactor);
        if (!divisor.isZero()) {
            TestInt128Math.assertDivide(dividend, dividendRescaleFactor, Int128Math.negate((Int128)divisor), divisorRescaleFactor);
        }
        if (!dividend.isZero()) {
            TestInt128Math.assertDivide(Int128Math.negate((Int128)dividend), dividendRescaleFactor, divisor, divisorRescaleFactor);
        }
        if (!dividend.isZero() && !divisor.isZero()) {
            TestInt128Math.assertDivide(Int128Math.negate((Int128)dividend), dividendRescaleFactor, Int128Math.negate((Int128)divisor), divisorRescaleFactor);
        }
    }

    private static void assertDivide(Int128 dividend, int dividendRescaleFactor, Int128 divisor, int divisorRescaleFactor) {
        BigInteger dividendBigInteger = dividend.toBigInteger();
        BigInteger divisorBigInteger = divisor.toBigInteger();
        BigInteger rescaledDividend = dividendBigInteger.multiply(Decimals.bigIntegerTenToNth((int)dividendRescaleFactor));
        BigInteger rescaledDivisor = divisorBigInteger.multiply(Decimals.bigIntegerTenToNth((int)divisorRescaleFactor));
        BigInteger expectedQuotient = new BigDecimal(rescaledDividend).divide(new BigDecimal(rescaledDivisor), RoundingMode.HALF_UP).toBigIntegerExact();
        BigInteger expectedRemainder = rescaledDividend.remainder(rescaledDivisor);
        boolean overflowIsExpected = expectedQuotient.abs().compareTo(Int128.MAX_VALUE.toBigInteger()) >= 0 || expectedRemainder.abs().compareTo(Int128.MAX_VALUE.toBigInteger()) >= 0;
        try {
            Int128 quotient = Int128Math.divideRoundUp((long)dividend.getHigh(), (long)dividend.getLow(), (int)dividendRescaleFactor, (long)divisor.getHigh(), (long)divisor.getLow(), (int)divisorRescaleFactor);
            Int128 remainder = Int128Math.remainder((long)dividend.getHigh(), (long)dividend.getLow(), (int)dividendRescaleFactor, (long)divisor.getHigh(), (long)divisor.getLow(), (int)divisorRescaleFactor);
            if (overflowIsExpected) {
                throw new AssertionError((Object)"overflow is expected");
            }
            BigInteger actualQuotient = quotient.toBigInteger();
            BigInteger actualRemainder = remainder.toBigInteger();
            if (expectedQuotient.equals(actualQuotient) && expectedRemainder.equals(actualRemainder)) {
                return;
            }
            throw new AssertionError((Object)String.format("%s / %s ([%s * 2^%d] / [%s * 2^%d]) Expected: %s(%s). Actual: %s(%s)", rescaledDividend, rescaledDivisor, dividendBigInteger, dividendRescaleFactor, divisorBigInteger, divisorRescaleFactor, expectedQuotient, expectedRemainder, actualQuotient, actualRemainder));
        }
        catch (ArithmeticException e) {
            if (!overflowIsExpected) {
                throw new AssertionError((Object)"overflow wasn't expected");
            }
            return;
        }
    }

    private static boolean isShort(BigInteger value) {
        return value.abs().shiftRight(63).equals(BigInteger.ZERO);
    }

    private static void assertMultiply(BigInteger a, long b, BigInteger result) {
        TestInt128Math.assertMultiply(a, BigInteger.valueOf(b), result);
    }

    private static void assertMultiply(BigInteger a, long b, long result) {
        TestInt128Math.assertMultiply(a, BigInteger.valueOf(b), BigInteger.valueOf(result));
    }

    private static void assertMultiply(long a, long b, BigInteger result) {
        TestInt128Math.assertMultiply(BigInteger.valueOf(a), b, result);
    }

    private static void assertMultiply(long a, long b, long result) {
        TestInt128Math.assertMultiply(a, b, BigInteger.valueOf(result));
    }

    private static void assertMultiply(BigInteger a, BigInteger b, BigInteger result) {
        Assertions.assertThat((Comparable)Int128.valueOf((BigInteger)result)).isEqualTo((Object)Int128Math.multiply((Int128)Int128.valueOf((BigInteger)a), (Int128)Int128.valueOf((BigInteger)b)));
        if (TestInt128Math.isShort(a) && TestInt128Math.isShort(b)) {
            Assertions.assertThat((Comparable)Int128.valueOf((BigInteger)result)).isEqualTo((Object)Int128Math.multiply((long)a.longValue(), (long)b.longValue()));
        }
        if (TestInt128Math.isShort(a) && !TestInt128Math.isShort(b)) {
            Assertions.assertThat((Comparable)Int128.valueOf((BigInteger)result)).isEqualTo((Object)Int128Math.multiply((Int128)Int128.valueOf((BigInteger)b), (long)a.longValue()));
        }
        if (!TestInt128Math.isShort(a) && TestInt128Math.isShort(b)) {
            Assertions.assertThat((Comparable)Int128.valueOf((BigInteger)result)).isEqualTo((Object)Int128Math.multiply((Int128)Int128.valueOf((BigInteger)a), (long)b.longValue()));
        }
    }

    private static void assertRescale(Int128 decimal, int rescale, Int128 expected) {
        Assertions.assertThat((Comparable)Int128Math.rescale((Int128)decimal, (int)rescale)).isEqualTo((Object)expected);
        long[] result = new long[3];
        Int128Math.rescale((long)decimal.getHigh(), (long)decimal.getLow(), (int)rescale, (long[])result, (int)1);
        Assertions.assertThat((Comparable)Int128.valueOf((long)result[1], (long)result[2])).isEqualTo((Object)expected);
    }

    private static void assertRescaleTruncate(Int128 decimal, int rescale, Int128 expected) {
        Assertions.assertThat((Comparable)Int128Math.rescaleTruncate((Int128)decimal, (int)rescale)).isEqualTo((Object)expected);
        long[] result = new long[3];
        Int128Math.rescaleTruncate((long)decimal.getHigh(), (long)decimal.getLow(), (int)rescale, (long[])result, (int)1);
        Assertions.assertThat((Comparable)Int128.valueOf((long)result[1], (long)result[2])).isEqualTo((Object)expected);
    }

    private static Int128 shiftLeft(Int128 value, int shift) {
        long[] parts = value.toLongArray();
        Int128Math.shiftLeft((long[])parts, (int)shift);
        return Int128.valueOf((long[])parts);
    }

    private static Int128 valueOf(int[] value) {
        Preconditions.checkArgument((value.length == 4 ? 1 : 0) != 0, (Object)"Expected int[4]");
        return Int128.valueOf((long)((long)value[0] << 32 | (long)value[1]), (long)((long)value[2] << 32 | (long)value[3]));
    }
}

