/*
 * Decompiled with CFR 0.152.
 */
package io.trino.spi.block;

import com.google.errorprone.annotations.ThreadSafe;
import io.airlift.slice.SizeOf;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.block.DuplicateMapKeyException;
import io.trino.spi.type.MapType;
import jakarta.annotation.Nullable;
import java.util.Arrays;
import java.util.Optional;

@ThreadSafe
public final class MapHashTables {
    public static final int INSTANCE_SIZE = SizeOf.instanceSize(MapHashTables.class);
    static final int HASH_MULTIPLIER = 2;
    private final MapType mapType;
    private final HashBuildMode mode;
    private final int hashTableCount;
    @Nullable
    private volatile int[] hashTables;

    static MapHashTables createSingleTable(MapType mapType, HashBuildMode mode, Block keyBlock) {
        int[] hashTables = new int[keyBlock.getPositionCount() * 2];
        Arrays.fill(hashTables, -1);
        MapHashTables.buildHashTable(mode, mapType, keyBlock, 0, keyBlock.getPositionCount(), hashTables);
        return new MapHashTables(mapType, mode, 1, Optional.of(hashTables));
    }

    static MapHashTables create(HashBuildMode mode, MapType mapType, int hashTableCount, Block keyBlock, int[] offsets, @Nullable boolean[] mapIsNull) {
        MapHashTables hashTables = new MapHashTables(mapType, mode, hashTableCount, Optional.empty());
        hashTables.buildAllHashTables(keyBlock, offsets, mapIsNull);
        return hashTables;
    }

    MapHashTables(MapType mapType, HashBuildMode mode, int hashTableCount, Optional<int[]> hashTables) {
        this.mapType = mapType;
        this.mode = mode;
        this.hashTableCount = hashTableCount;
        this.hashTables = hashTables.orElse(null);
    }

    public long getRetainedSizeInBytes() {
        return (long)INSTANCE_SIZE + SizeOf.sizeOf((int[])this.hashTables);
    }

    int[] get() {
        if (this.hashTables == null) {
            throw new IllegalStateException("hashTables are not built");
        }
        return this.hashTables;
    }

    Optional<int[]> tryGet() {
        return Optional.ofNullable(this.hashTables);
    }

    void buildAllHashTablesIfNecessary(Block rawKeyBlock, int[] offsets, @Nullable boolean[] mapIsNull) {
        if (this.hashTables == null) {
            this.buildAllHashTables(rawKeyBlock, offsets, mapIsNull);
        }
    }

    private synchronized void buildAllHashTables(Block rawKeyBlock, int[] offsets, @Nullable boolean[] mapIsNull) {
        if (this.hashTables != null) {
            return;
        }
        int[] hashTables = new int[rawKeyBlock.getPositionCount() * 2];
        Arrays.fill(hashTables, -1);
        for (int i = 0; i < this.hashTableCount; ++i) {
            int keyOffset = offsets[i];
            int keyCount = offsets[i + 1] - keyOffset;
            if (keyCount < 0) {
                throw new IllegalArgumentException(String.format("Offset is not monotonically ascending. offsets[%s]=%s, offsets[%s]=%s", i, offsets[i], i + 1, offsets[i + 1]));
            }
            if (mapIsNull != null && mapIsNull[i] && keyCount != 0) {
                throw new IllegalArgumentException("A null map must have zero entries");
            }
            MapHashTables.buildHashTable(this.mode, this.mapType, rawKeyBlock, keyOffset, keyCount, hashTables);
        }
        this.hashTables = hashTables;
    }

    private static void buildHashTable(HashBuildMode mode, MapType mapType, Block rawKeyBlock, int keyOffset, int keyCount, int[] hashTables) {
        switch (mode.ordinal()) {
            case 0: {
                MapHashTables.buildHashTableDuplicateNotChecked(mapType, rawKeyBlock, keyOffset, keyCount, hashTables);
                break;
            }
            case 1: {
                MapHashTables.buildHashTableStrict(mapType, rawKeyBlock, keyOffset, keyCount, hashTables);
                break;
            }
            case 2: {
                MapHashTables.buildDistinctHashTableStrict(mapType, rawKeyBlock, keyOffset, keyCount, hashTables);
            }
        }
    }

    private static void buildHashTableDuplicateNotChecked(MapType mapType, Block keyBlock, int keyOffset, int keyCount, int[] hashTables) {
        int hashTableOffset = keyOffset * 2;
        int hashTableSize = keyCount * 2;
        int i = 0;
        while (i < keyCount) {
            int hash = MapHashTables.getHashPosition(mapType, keyBlock, keyOffset + i, hashTableSize);
            while (true) {
                if (hashTables[hashTableOffset + hash] == -1) break;
                if (++hash != hashTableSize) continue;
                hash = 0;
            }
            hashTables[hashTableOffset + hash] = i++;
        }
    }

    private static void buildHashTableStrict(MapType mapType, Block keyBlock, int keyOffset, int keyCount, int[] hashTables) {
        int hashTableOffset = keyOffset * 2;
        int hashTableSize = keyCount * 2;
        int i = 0;
        while (i < keyCount) {
            int hash = MapHashTables.getHashPosition(mapType, keyBlock, keyOffset + i, hashTableSize);
            while (true) {
                Boolean isDuplicateKey;
                if (hashTables[hashTableOffset + hash] == -1) break;
                try {
                    isDuplicateKey = mapType.getKeyBlockEqual().invokeExact(keyBlock, keyOffset + i, keyBlock, keyOffset + hashTables[hashTableOffset + hash]);
                }
                catch (RuntimeException e) {
                    throw e;
                }
                catch (Throwable throwable) {
                    throw new RuntimeException(throwable);
                }
                if (isDuplicateKey == null) {
                    throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "map key cannot be null or contain nulls");
                }
                if (isDuplicateKey.booleanValue()) {
                    throw new DuplicateMapKeyException(keyBlock, keyOffset + i);
                }
                if (++hash != hashTableSize) continue;
                hash = 0;
            }
            hashTables[hashTableOffset + hash] = i++;
        }
    }

    private static void buildDistinctHashTableStrict(MapType mapType, Block keyBlock, int keyOffset, int keyCount, int[] hashTables) {
        int hashTableOffset = keyOffset * 2;
        int hashTableSize = keyCount * 2;
        int i = 0;
        while (i < keyCount) {
            int hash = MapHashTables.getHashPosition(mapType, keyBlock, keyOffset + i, hashTableSize);
            while (true) {
                boolean isDuplicateKey;
                if (hashTables[hashTableOffset + hash] == -1) break;
                try {
                    isDuplicateKey = mapType.getKeyBlockNotDistinctFrom().invokeExact(keyBlock, keyOffset + i, keyBlock, keyOffset + hashTables[hashTableOffset + hash]);
                }
                catch (RuntimeException e) {
                    throw e;
                }
                catch (Throwable throwable) {
                    throw new RuntimeException(throwable);
                }
                if (isDuplicateKey) {
                    throw new DuplicateMapKeyException(keyBlock, keyOffset + i);
                }
                if (++hash != hashTableSize) continue;
                hash = 0;
            }
            hashTables[hashTableOffset + hash] = i++;
        }
    }

    private static int getHashPosition(MapType mapType, Block keyBlock, int position, int hashTableSize) {
        long hashCode;
        if (keyBlock.isNull(position)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "map key cannot be null");
        }
        try {
            hashCode = mapType.getKeyBlockHashCode().invokeExact(keyBlock, position);
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
        return MapHashTables.computePosition(hashCode, hashTableSize);
    }

    static int computePosition(long hashcode, int hashTableSize) {
        return (int)(Integer.toUnsignedLong(Long.hashCode(hashcode)) * (long)hashTableSize >> 32);
    }

    public static enum HashBuildMode {
        DUPLICATE_NOT_CHECKED,
        STRICT_EQUALS,
        STRICT_NOT_DISTINCT_FROM;

    }
}

