/*
 * Decompiled with CFR 0.152.
 */
package io.trino.spooling.filesystem;

import com.google.common.collect.ImmutableMap;
import com.google.inject.Inject;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceOutput;
import io.airlift.slice.Slices;
import io.airlift.units.Duration;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoFileSystemFactory;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.filesystem.encryption.EncryptionKey;
import io.trino.spi.protocol.SpooledLocation;
import io.trino.spi.protocol.SpooledSegmentHandle;
import io.trino.spi.protocol.SpoolingContext;
import io.trino.spi.protocol.SpoolingManager;
import io.trino.spi.security.ConnectorIdentity;
import io.trino.spooling.filesystem.FileSystemLayout;
import io.trino.spooling.filesystem.FileSystemSpooledSegmentHandle;
import io.trino.spooling.filesystem.FileSystemSpoolingConfig;
import io.trino.spooling.filesystem.encryption.EncryptionHeadersTranslator;
import io.trino.spooling.filesystem.encryption.ExceptionMappingInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

public class FileSystemSpoolingManager
implements SpoolingManager {
    private final Location location;
    private final EncryptionHeadersTranslator encryptionHeadersTranslator;
    private final TrinoFileSystem fileSystem;
    private final FileSystemLayout fileSystemLayout;
    private final Duration ttl;
    private final Duration directAccessTtl;
    private final boolean encryptionEnabled;
    private final boolean explicitAckEnabled;
    private final Random random = ThreadLocalRandom.current();

    @Inject
    public FileSystemSpoolingManager(FileSystemSpoolingConfig config, TrinoFileSystemFactory fileSystemFactory, FileSystemLayout fileSystemLayout) {
        Objects.requireNonNull(config, "config is null");
        this.location = Location.of((String)config.getLocation());
        this.fileSystem = Objects.requireNonNull(fileSystemFactory, "fileSystemFactory is null").create(ConnectorIdentity.ofUser((String)"ignored"));
        this.fileSystemLayout = Objects.requireNonNull(fileSystemLayout, "fileSystemLayout is null");
        this.encryptionHeadersTranslator = EncryptionHeadersTranslator.encryptionHeadersTranslator(this.location);
        this.ttl = config.getTtl();
        this.directAccessTtl = config.getDirectAccessTtl();
        this.encryptionEnabled = config.isEncryptionEnabled();
        this.explicitAckEnabled = config.isExplicitAckEnabled();
    }

    public OutputStream createOutputStream(SpooledSegmentHandle handle) throws IOException {
        FileSystemSpooledSegmentHandle fileHandle = (FileSystemSpooledSegmentHandle)handle;
        Location storageLocation = this.fileSystemLayout.location(this.location, fileHandle);
        Optional<EncryptionKey> encryption = fileHandle.encryptionKey();
        TrinoOutputFile outputFile = this.encryptionEnabled ? this.fileSystem.newEncryptedOutputFile(storageLocation, encryption.orElseThrow()) : this.fileSystem.newOutputFile(storageLocation);
        return outputFile.create();
    }

    public FileSystemSpooledSegmentHandle create(SpoolingContext context) {
        Instant expireAt = Instant.now().plusMillis(this.ttl.toMillis());
        if (this.encryptionEnabled) {
            return FileSystemSpooledSegmentHandle.random(this.random, context, expireAt, Optional.of(EncryptionKey.randomAes256()));
        }
        return FileSystemSpooledSegmentHandle.random(this.random, context, expireAt);
    }

    public InputStream openInputStream(SpooledSegmentHandle handle) throws IOException {
        FileSystemSpooledSegmentHandle fileHandle = (FileSystemSpooledSegmentHandle)handle;
        this.checkExpiration(fileHandle);
        Optional<EncryptionKey> encryption = fileHandle.encryptionKey();
        Location storageLocation = this.fileSystemLayout.location(this.location, fileHandle);
        TrinoInputFile inputFile = this.encryptionEnabled ? this.fileSystem.newEncryptedInputFile(storageLocation, encryption.orElseThrow()) : this.fileSystem.newInputFile(storageLocation);
        FileSystemSpoolingManager.checkFileExists(inputFile);
        return new ExceptionMappingInputStream((InputStream)inputFile.newStream());
    }

    public void acknowledge(SpooledSegmentHandle handle) throws IOException {
        if (!this.explicitAckEnabled) {
            return;
        }
        this.fileSystem.deleteFile(this.fileSystemLayout.location(this.location, (FileSystemSpooledSegmentHandle)handle));
    }

    public Optional<SpooledLocation.DirectLocation> directLocation(SpooledSegmentHandle handle) throws IOException {
        FileSystemSpooledSegmentHandle fileHandle = (FileSystemSpooledSegmentHandle)handle;
        Location storageLocation = this.fileSystemLayout.location(this.location, fileHandle);
        Duration ttl = this.remainingTtl(fileHandle.expirationTime(), this.directAccessTtl);
        Optional<EncryptionKey> key = fileHandle.encryptionKey();
        Optional<SpooledLocation.DirectLocation> directLocation = this.encryptionEnabled ? this.fileSystem.encryptedPreSignedUri(storageLocation, ttl, key.orElseThrow()).map(uri -> new SpooledLocation.DirectLocation(FileSystemSpoolingManager.serialize(fileHandle), uri.uri(), uri.headers())) : this.fileSystem.preSignedUri(storageLocation, ttl).map(uri -> new SpooledLocation.DirectLocation(FileSystemSpoolingManager.serialize(fileHandle), uri.uri(), uri.headers()));
        if (directLocation.isEmpty()) {
            throw new IOException("Failed to generate pre-signed URI for segment %s".formatted(fileHandle.identifier()));
        }
        return directLocation;
    }

    public SpooledLocation location(SpooledSegmentHandle handle) {
        FileSystemSpooledSegmentHandle fileHandle = (FileSystemSpooledSegmentHandle)handle;
        return SpooledLocation.coordinatorLocation((Slice)FileSystemSpoolingManager.serialize(fileHandle), this.headers(fileHandle));
    }

    private static Slice serialize(FileSystemSpooledSegmentHandle fileHandle) {
        byte[] encoding = fileHandle.encoding().getBytes(StandardCharsets.UTF_8);
        Slice slice = Slices.allocate((int)(18 + encoding.length + 1));
        SliceOutput output = slice.getOutput();
        output.writeBytes(fileHandle.uuid());
        output.writeShort(fileHandle.encoding().length());
        output.writeBytes(encoding);
        output.writeBoolean(fileHandle.encryptionKey().isPresent());
        return output.slice();
    }

    private Map<String, List<String>> headers(FileSystemSpooledSegmentHandle fileHandle) {
        return fileHandle.encryptionKey().map(this.encryptionHeadersTranslator::createHeaders).orElse((Map)ImmutableMap.of());
    }

    public SpooledSegmentHandle handle(Slice identifier, Map<String, List<String>> headers) {
        BasicSliceInput input = identifier.getInput();
        byte[] uuid = new byte[16];
        input.readBytes(uuid);
        short encodingLength = input.readShort();
        String encoding = input.readSlice((int)encodingLength).toStringUtf8();
        if (!input.readBoolean()) {
            return new FileSystemSpooledSegmentHandle(encoding, uuid, Optional.empty());
        }
        return new FileSystemSpooledSegmentHandle(encoding, uuid, Optional.of(this.encryptionHeadersTranslator.extractKey(headers)));
    }

    private Duration remainingTtl(Instant expiresAt, Duration accessTtl) {
        Duration remainingTTL = new Duration((double)java.time.Duration.between(Instant.now(), expiresAt).toMillis(), TimeUnit.MILLISECONDS);
        if (accessTtl.compareTo(remainingTTL) < 0) {
            return accessTtl;
        }
        return remainingTTL;
    }

    private void checkExpiration(FileSystemSpooledSegmentHandle handle) throws IOException {
        if (handle.expirationTime().isBefore(Instant.now())) {
            throw new IOException("Segment not found or expired");
        }
    }

    private static void checkFileExists(TrinoInputFile inputFile) throws IOException {
        if (!inputFile.exists()) {
            throw new IOException("Segment not found or expired");
        }
    }
}

