/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.s3;

import com.google.common.base.Verify;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimaps;
import com.google.common.collect.SetMultimap;
import io.airlift.units.Duration;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoFileSystemException;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.filesystem.UriLocation;
import io.trino.filesystem.encryption.EncryptionKey;
import io.trino.filesystem.s3.S3Context;
import io.trino.filesystem.s3.S3FileIterator;
import io.trino.filesystem.s3.S3FileSystemConfig;
import io.trino.filesystem.s3.S3InputFile;
import io.trino.filesystem.s3.S3Location;
import io.trino.filesystem.s3.S3OutputFile;
import io.trino.filesystem.s3.S3SseCUtils;
import io.trino.filesystem.s3.S3SseRequestConfigurator;
import java.io.IOException;
import java.net.URISyntaxException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.auth.signer.AwsS3V4Signer;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.CommonPrefix;
import software.amazon.awssdk.services.s3.model.DeleteObjectRequest;
import software.amazon.awssdk.services.s3.model.DeleteObjectsRequest;
import software.amazon.awssdk.services.s3.model.DeleteObjectsResponse;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Request;
import software.amazon.awssdk.services.s3.model.ObjectIdentifier;
import software.amazon.awssdk.services.s3.model.OptionalObjectAttributes;
import software.amazon.awssdk.services.s3.model.RequestPayer;
import software.amazon.awssdk.services.s3.model.S3Error;
import software.amazon.awssdk.services.s3.model.S3Object;
import software.amazon.awssdk.services.s3.presigner.S3Presigner;
import software.amazon.awssdk.services.s3.presigner.model.GetObjectPresignRequest;
import software.amazon.awssdk.services.s3.presigner.model.PresignedGetObjectRequest;

final class S3FileSystem
implements TrinoFileSystem {
    private final Executor uploadExecutor;
    private final S3Client client;
    private final S3Presigner preSigner;
    private final S3Context context;
    private final RequestPayer requestPayer;

    public S3FileSystem(Executor uploadExecutor, S3Client client, S3Presigner preSigner, S3Context context) {
        this.uploadExecutor = Objects.requireNonNull(uploadExecutor, "uploadExecutor is null");
        this.client = Objects.requireNonNull(client, "client is null");
        this.preSigner = Objects.requireNonNull(preSigner, "preSigner is null");
        this.context = Objects.requireNonNull(context, "context is null");
        this.requestPayer = context.requestPayer();
    }

    public TrinoInputFile newInputFile(Location location) {
        return new S3InputFile(this.client, this.context, new S3Location(location), null, null, Optional.empty());
    }

    public TrinoInputFile newInputFile(Location location, long length) {
        return new S3InputFile(this.client, this.context, new S3Location(location), length, null, Optional.empty());
    }

    public TrinoInputFile newInputFile(Location location, long length, Instant lastModified) {
        return new S3InputFile(this.client, this.context, new S3Location(location), length, lastModified, Optional.empty());
    }

    public TrinoInputFile newEncryptedInputFile(Location location, EncryptionKey key) {
        return new S3InputFile(this.client, this.context, new S3Location(location), null, null, Optional.of(key));
    }

    public TrinoInputFile newEncryptedInputFile(Location location, long length, EncryptionKey key) {
        return new S3InputFile(this.client, this.context, new S3Location(location), length, null, Optional.of(key));
    }

    public TrinoInputFile newEncryptedInputFile(Location location, long length, Instant lastModified, EncryptionKey key) {
        return new S3InputFile(this.client, this.context, new S3Location(location), length, lastModified, Optional.of(key));
    }

    public TrinoOutputFile newOutputFile(Location location) {
        return new S3OutputFile(this.uploadExecutor, this.client, this.context, new S3Location(location), Optional.empty());
    }

    public TrinoOutputFile newEncryptedOutputFile(Location location, EncryptionKey key) {
        return new S3OutputFile(this.uploadExecutor, this.client, this.context, new S3Location(location), Optional.of(key));
    }

    public void deleteFile(Location location) throws IOException {
        location.verifyValidFileLocation();
        S3Location s3Location = new S3Location(location);
        DeleteObjectRequest request = (DeleteObjectRequest)DeleteObjectRequest.builder().overrideConfiguration(this.context::applyCredentialProviderOverride).requestPayer(this.requestPayer).key(s3Location.key()).bucket(s3Location.bucket()).build();
        try {
            this.client.deleteObject(request);
        }
        catch (SdkException e) {
            throw new TrinoFileSystemException("Failed to delete file: " + String.valueOf(location), (Throwable)e);
        }
    }

    public void deleteDirectory(Location location) throws IOException {
        FileIterator iterator = this.listObjects(location, true);
        while (iterator.hasNext()) {
            ArrayList<Location> files = new ArrayList<Location>();
            while (files.size() < 1000 && iterator.hasNext()) {
                files.add(iterator.next().location());
            }
            this.deleteObjects(files);
        }
    }

    public void deleteFiles(Collection<Location> locations) throws IOException {
        locations.forEach(Location::verifyValidFileLocation);
        this.deleteObjects(locations);
    }

    private void deleteObjects(Collection<Location> locations) throws IOException {
        SetMultimap bucketToKeys = (SetMultimap)locations.stream().map(S3Location::new).collect(Multimaps.toMultimap(S3Location::bucket, S3Location::key, HashMultimap::create));
        HashMap<String, String> failures = new HashMap<String, String>();
        for (Map.Entry entry : bucketToKeys.asMap().entrySet()) {
            String bucket = (String)entry.getKey();
            Collection allKeys = (Collection)entry.getValue();
            for (List keys : Iterables.partition((Iterable)allKeys, (int)250)) {
                List<ObjectIdentifier> objects = keys.stream().map(key -> (ObjectIdentifier)ObjectIdentifier.builder().key(key).build()).toList();
                DeleteObjectsRequest request = (DeleteObjectsRequest)DeleteObjectsRequest.builder().overrideConfiguration(this.context::applyCredentialProviderOverride).requestPayer(this.requestPayer).bucket(bucket).overrideConfiguration(S3FileSystem.disableStrongIntegrityChecksums()).delete(builder -> builder.objects((Collection)objects).quiet(Boolean.valueOf(true))).build();
                try {
                    DeleteObjectsResponse response = this.client.deleteObjects(request);
                    for (S3Error error : response.errors()) {
                        failures.put("s3://%s/%s".formatted(bucket, error.key()), error.code());
                    }
                }
                catch (SdkException e) {
                    throw new TrinoFileSystemException("Error while batch deleting files", (Throwable)e);
                }
            }
        }
        if (!failures.isEmpty()) {
            throw new IOException("Failed to delete one or more files: " + String.valueOf(failures));
        }
    }

    public void renameFile(Location source, Location target) throws IOException {
        throw new IOException("S3 does not support renames");
    }

    public FileIterator listFiles(Location location) throws IOException {
        return this.listObjects(location, false);
    }

    private FileIterator listObjects(Location location, boolean includeDirectoryObjects) throws IOException {
        S3Location s3Location = new S3Location(location);
        Object key = s3Location.key();
        if (!((String)key).isEmpty() && !((String)key).endsWith("/")) {
            key = (String)key + "/";
        }
        ListObjectsV2Request request = (ListObjectsV2Request)ListObjectsV2Request.builder().overrideConfiguration(this.context::applyCredentialProviderOverride).optionalObjectAttributes(new OptionalObjectAttributes[]{OptionalObjectAttributes.RESTORE_STATUS}).requestPayer(this.requestPayer).bucket(s3Location.bucket()).prefix((String)key).build();
        try {
            Stream<S3Object> s3ObjectStream = this.client.listObjectsV2Paginator(request).contents().stream();
            if (!includeDirectoryObjects) {
                s3ObjectStream = s3ObjectStream.filter(object -> !object.key().endsWith("/"));
            }
            return new S3FileIterator(s3Location, s3ObjectStream.iterator());
        }
        catch (SdkException e) {
            throw new TrinoFileSystemException("Failed to list location: " + String.valueOf(location), (Throwable)e);
        }
    }

    public Optional<Boolean> directoryExists(Location location) throws IOException {
        S3FileSystem.validateS3Location(location);
        if (location.path().isEmpty() || this.listFiles(location).hasNext()) {
            return Optional.of(true);
        }
        return Optional.empty();
    }

    public void createDirectory(Location location) {
        S3FileSystem.validateS3Location(location);
    }

    public void renameDirectory(Location source, Location target) throws IOException {
        throw new IOException("S3 does not support directory renames");
    }

    public Set<Location> listDirectories(Location location) throws IOException {
        S3Location s3Location = new S3Location(location);
        Location baseLocation = s3Location.baseLocation();
        Object key = s3Location.key();
        if (!((String)key).isEmpty() && !((String)key).endsWith("/")) {
            key = (String)key + "/";
        }
        ListObjectsV2Request request = (ListObjectsV2Request)ListObjectsV2Request.builder().overrideConfiguration(this.context::applyCredentialProviderOverride).requestPayer(this.requestPayer).bucket(s3Location.bucket()).prefix((String)key).delimiter("/").build();
        try {
            return (Set)this.client.listObjectsV2Paginator(request).commonPrefixes().stream().map(CommonPrefix::prefix).map(arg_0 -> ((Location)baseLocation).appendPath(arg_0)).collect(ImmutableSet.toImmutableSet());
        }
        catch (SdkException e) {
            throw new TrinoFileSystemException("Failed to list location: " + String.valueOf(location), (Throwable)e);
        }
    }

    public Optional<Location> createTemporaryDirectory(Location targetPath, String temporaryPrefix, String relativePrefix) {
        S3FileSystem.validateS3Location(targetPath);
        return Optional.empty();
    }

    public Optional<UriLocation> preSignedUri(Location location, Duration ttl) throws IOException {
        return this.encryptedPreSignedUri(location, ttl, Optional.empty());
    }

    public Optional<UriLocation> encryptedPreSignedUri(Location location, Duration ttl, EncryptionKey key) throws IOException {
        return this.encryptedPreSignedUri(location, ttl, Optional.of(key));
    }

    public Optional<UriLocation> encryptedPreSignedUri(Location location, Duration ttl, Optional<EncryptionKey> key) throws IOException {
        location.verifyValidFileLocation();
        S3Location s3Location = new S3Location(location);
        Verify.verify((key.isEmpty() || this.context.s3SseContext().sseType() == S3FileSystemConfig.S3SseType.NONE ? 1 : 0) != 0, (String)"Encryption key cannot be used with SSE configuration", (Object[])new Object[0]);
        GetObjectRequest request = (GetObjectRequest)((GetObjectRequest.Builder)GetObjectRequest.builder().overrideConfiguration(this.context::applyCredentialProviderOverride).requestPayer(this.requestPayer).key(s3Location.key()).bucket(s3Location.bucket()).applyMutation(builder -> key.ifPresentOrElse(encryption -> builder.sseCustomerKeyMD5(S3SseCUtils.md5Checksum(encryption)).sseCustomerAlgorithm(encryption.algorithm()).sseCustomerKey(S3SseCUtils.encoded(encryption)), () -> S3SseRequestConfigurator.setEncryptionSettings(builder, this.context.s3SseContext())))).build();
        GetObjectPresignRequest preSignRequest = GetObjectPresignRequest.builder().signatureDuration(ttl.toJavaTime()).getObjectRequest(request).build();
        try {
            PresignedGetObjectRequest preSigned = this.preSigner.presignGetObject(preSignRequest);
            return Optional.of(new UriLocation(preSigned.url().toURI(), S3FileSystem.filterHeaders(preSigned.httpRequest().headers())));
        }
        catch (SdkException e) {
            throw new IOException("Failed to generate pre-signed URI", e);
        }
        catch (URISyntaxException e) {
            throw new TrinoFileSystemException("Failed to convert pre-signed URI to URI", (Throwable)e);
        }
    }

    private static Map<String, List<String>> filterHeaders(Map<String, List<String>> headers) {
        return headers.entrySet().stream().filter(entry -> !((String)entry.getKey()).equalsIgnoreCase("host")).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    private static void validateS3Location(Location location) {
        new S3Location(location);
    }

    static AwsRequestOverrideConfiguration disableStrongIntegrityChecksums() {
        return ((AwsRequestOverrideConfiguration.Builder)AwsRequestOverrideConfiguration.builder().signer((Signer)AwsS3V4Signer.create())).build();
    }
}

