/*
 * Decompiled with CFR 0.152.
 */
package io.trino.spi.block;

import io.airlift.slice.SizeOf;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockUtil;
import io.trino.spi.block.IntArrayList;
import io.trino.spi.block.MapHashTables;
import io.trino.spi.block.SqlMap;
import io.trino.spi.block.ValueBlock;
import io.trino.spi.type.MapType;
import jakarta.annotation.Nullable;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.function.ObjLongConsumer;

public final class MapBlock
implements ValueBlock {
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(MapBlock.class);
    private final MapType mapType;
    private final int startOffset;
    private final int positionCount;
    @Nullable
    private final boolean[] mapIsNull;
    private final int[] offsets;
    private final Block keyBlock;
    private final Block valueBlock;
    private final MapHashTables hashTables;
    private final long baseSizeInBytes;
    private volatile long valueSizeInBytes = -1L;
    private final long retainedSizeInBytes;

    public static MapBlock fromKeyValueBlock(Optional<boolean[]> mapIsNull, int[] offsets, Block keyBlock, Block valueBlock, MapType mapType) {
        return MapBlock.fromKeyValueBlock(mapIsNull, offsets, offsets.length - 1, keyBlock, valueBlock, mapType);
    }

    public static MapBlock fromKeyValueBlock(Optional<boolean[]> mapIsNull, int[] offsets, int mapCount, Block keyBlock, Block valueBlock, MapType mapType) {
        MapBlock.validateConstructorArguments(mapType, 0, mapCount, mapIsNull.orElse(null), offsets, keyBlock, valueBlock);
        return MapBlock.createMapBlockInternal(mapType, 0, mapCount, mapIsNull, offsets, keyBlock, valueBlock, new MapHashTables(mapType, MapHashTables.HashBuildMode.DUPLICATE_NOT_CHECKED, mapCount, Optional.empty()));
    }

    public static MapBlock createMapBlockInternal(MapType mapType, int startOffset, int positionCount, Optional<boolean[]> mapIsNull, int[] offsets, Block keyBlock, Block valueBlock, MapHashTables hashTables) {
        MapBlock.validateConstructorArguments(mapType, startOffset, positionCount, mapIsNull.orElse(null), offsets, keyBlock, valueBlock);
        Objects.requireNonNull(hashTables, "hashTables is null");
        return new MapBlock(mapType, startOffset, positionCount, mapIsNull.orElse(null), offsets, keyBlock, valueBlock, hashTables);
    }

    private static void validateConstructorArguments(MapType mapType, int startOffset, int positionCount, @Nullable boolean[] mapIsNull, int[] offsets, Block keyBlock, Block valueBlock) {
        if (startOffset < 0) {
            throw new IllegalArgumentException("startOffset is negative");
        }
        if (positionCount < 0) {
            throw new IllegalArgumentException("positionCount is negative");
        }
        if (mapIsNull != null && mapIsNull.length - startOffset < positionCount) {
            throw new IllegalArgumentException("isNull length is less than positionCount");
        }
        Objects.requireNonNull(offsets, "offsets is null");
        if (offsets.length - startOffset < positionCount + 1) {
            throw new IllegalArgumentException("offsets length is less than positionCount");
        }
        Objects.requireNonNull(keyBlock, "keyBlock is null");
        Objects.requireNonNull(valueBlock, "valueBlock is null");
        if (keyBlock.getPositionCount() != valueBlock.getPositionCount()) {
            throw new IllegalArgumentException(String.format("keyBlock and valueBlock has different size: %s %s", keyBlock.getPositionCount(), valueBlock.getPositionCount()));
        }
        Objects.requireNonNull(mapType, "mapType is null");
    }

    private MapBlock(MapType mapType, int startOffset, int positionCount, @Nullable boolean[] mapIsNull, int[] offsets, Block keyBlock, Block valueBlock, MapHashTables hashTables) {
        this.mapType = Objects.requireNonNull(mapType, "mapType is null");
        int[] rawHashTables = hashTables.tryGet().orElse(null);
        if (rawHashTables != null && rawHashTables.length < keyBlock.getPositionCount() * 2) {
            throw new IllegalArgumentException(String.format("keyBlock/valueBlock size does not match hash table size: %s %s", keyBlock.getPositionCount(), rawHashTables.length));
        }
        this.startOffset = startOffset;
        this.positionCount = positionCount;
        this.mapIsNull = mapIsNull;
        this.offsets = offsets;
        this.keyBlock = keyBlock;
        this.valueBlock = valueBlock;
        this.hashTables = hashTables;
        int entryCount = offsets[startOffset + positionCount] - offsets[startOffset];
        this.baseSizeInBytes = 8L * (long)entryCount + 5L * (long)this.positionCount + this.calculateSize(keyBlock);
        this.retainedSizeInBytes = (long)INSTANCE_SIZE + SizeOf.sizeOf((int[])offsets) + SizeOf.sizeOf((boolean[])mapIsNull);
    }

    Block getRawKeyBlock() {
        return this.keyBlock;
    }

    Block getRawValueBlock() {
        return this.valueBlock;
    }

    MapHashTables getHashTables() {
        return this.hashTables;
    }

    int[] getOffsets() {
        return this.offsets;
    }

    int getOffsetBase() {
        return this.startOffset;
    }

    @Override
    public boolean mayHaveNull() {
        return this.mapIsNull != null;
    }

    @Override
    public int getPositionCount() {
        return this.positionCount;
    }

    @Override
    public long getSizeInBytes() {
        if (this.valueSizeInBytes < 0L) {
            if (!this.valueBlock.isLoaded()) {
                return this.baseSizeInBytes + this.valueBlock.getSizeInBytes();
            }
            this.valueSizeInBytes = this.calculateSize(this.valueBlock);
        }
        return this.baseSizeInBytes + this.valueSizeInBytes;
    }

    private long calculateSize(Block block) {
        int entriesStart = this.offsets[this.startOffset];
        int entriesEnd = this.offsets[this.startOffset + this.positionCount];
        int entryCount = entriesEnd - entriesStart;
        return block.getRegionSizeInBytes(entriesStart, entryCount);
    }

    @Override
    public long getRetainedSizeInBytes() {
        return this.retainedSizeInBytes + this.keyBlock.getRetainedSizeInBytes() + this.valueBlock.getRetainedSizeInBytes() + this.hashTables.getRetainedSizeInBytes();
    }

    @Override
    public void retainedBytesForEachPart(ObjLongConsumer<Object> consumer) {
        consumer.accept(this.keyBlock, this.keyBlock.getRetainedSizeInBytes());
        consumer.accept(this.valueBlock, this.valueBlock.getRetainedSizeInBytes());
        consumer.accept(this.offsets, SizeOf.sizeOf((int[])this.offsets));
        if (this.mapIsNull != null) {
            consumer.accept(this.mapIsNull, SizeOf.sizeOf((boolean[])this.mapIsNull));
        }
        consumer.accept(this.hashTables, this.hashTables.getRetainedSizeInBytes());
        consumer.accept(this, INSTANCE_SIZE);
    }

    public String toString() {
        return "MapBlock{positionCount=" + this.getPositionCount() + "}";
    }

    @Override
    public boolean isLoaded() {
        return this.keyBlock.isLoaded() && this.valueBlock.isLoaded();
    }

    @Override
    public Block getLoadedBlock() {
        if (this.keyBlock != this.keyBlock.getLoadedBlock()) {
            throw new IllegalStateException();
        }
        Block loadedValueBlock = this.valueBlock.getLoadedBlock();
        if (loadedValueBlock == this.valueBlock) {
            return this;
        }
        return MapBlock.createMapBlockInternal(this.getMapType(), this.startOffset, this.positionCount, Optional.ofNullable(this.mapIsNull), this.offsets, this.keyBlock, loadedValueBlock, this.hashTables);
    }

    void ensureHashTableLoaded() {
        this.hashTables.buildAllHashTablesIfNecessary(this.keyBlock, this.offsets, this.mapIsNull);
    }

    @Override
    public MapBlock copyWithAppendedNull() {
        boolean[] newMapIsNull = BlockUtil.copyIsNullAndAppendNull(this.mapIsNull, this.startOffset, this.getPositionCount());
        int[] newOffsets = BlockUtil.copyOffsetsAndAppendNull(this.offsets, this.startOffset, this.getPositionCount());
        return MapBlock.createMapBlockInternal(this.getMapType(), this.startOffset, this.getPositionCount() + 1, Optional.of(newMapIsNull), newOffsets, this.keyBlock, this.valueBlock, this.hashTables);
    }

    MapType getMapType() {
        return this.mapType;
    }

    private int getOffset(int position) {
        return this.offsets[position + this.startOffset];
    }

    @Override
    public String getEncodingName() {
        return "MAP";
    }

    @Override
    public MapBlock copyPositions(int[] positions, int offset, int length) {
        BlockUtil.checkArrayRange(positions, offset, length);
        int[] newOffsets = new int[length + 1];
        boolean[] newMapIsNull = new boolean[length];
        IntArrayList entriesPositions = new IntArrayList();
        int newPosition = 0;
        for (int i = offset; i < offset + length; ++i) {
            int position = positions[i];
            if (this.isNull(position)) {
                newMapIsNull[newPosition] = true;
                newOffsets[newPosition + 1] = newOffsets[newPosition];
            } else {
                int entriesStartOffset = this.getOffset(position);
                int entriesEndOffset = this.getOffset(position + 1);
                int entryCount = entriesEndOffset - entriesStartOffset;
                newOffsets[newPosition + 1] = newOffsets[newPosition] + entryCount;
                for (int elementIndex = entriesStartOffset; elementIndex < entriesEndOffset; ++elementIndex) {
                    entriesPositions.add(elementIndex);
                }
            }
            ++newPosition;
        }
        int[] rawHashTables = this.hashTables.tryGet().orElse(null);
        int[] newRawHashTables = null;
        int newHashTableEntries = newOffsets[newOffsets.length - 1] * 2;
        if (rawHashTables != null) {
            newRawHashTables = new int[newHashTableEntries];
            int newHashIndex = 0;
            for (int i = offset; i < offset + length; ++i) {
                int position = positions[i];
                int entriesStartOffset = this.getOffset(position);
                int entriesEndOffset = this.getOffset(position + 1);
                for (int hashIndex = entriesStartOffset * 2; hashIndex < entriesEndOffset * 2; ++hashIndex) {
                    newRawHashTables[newHashIndex] = rawHashTables[hashIndex];
                    ++newHashIndex;
                }
            }
        }
        Block newKeys = this.keyBlock.copyPositions(entriesPositions.elements(), 0, entriesPositions.size());
        Block newValues = this.valueBlock.copyPositions(entriesPositions.elements(), 0, entriesPositions.size());
        return MapBlock.createMapBlockInternal(this.mapType, 0, length, Optional.of(newMapIsNull), newOffsets, newKeys, newValues, new MapHashTables(this.mapType, MapHashTables.HashBuildMode.DUPLICATE_NOT_CHECKED, length, Optional.ofNullable(newRawHashTables)));
    }

    @Override
    public MapBlock getRegion(int position, int length) {
        int positionCount = this.getPositionCount();
        BlockUtil.checkValidRegion(positionCount, position, length);
        return MapBlock.createMapBlockInternal(this.mapType, position + this.startOffset, length, Optional.ofNullable(this.mapIsNull), this.offsets, this.keyBlock, this.valueBlock, this.hashTables);
    }

    @Override
    public long getRegionSizeInBytes(int position, int length) {
        int positionCount = this.getPositionCount();
        BlockUtil.checkValidRegion(positionCount, position, length);
        int entriesStart = this.offsets[this.startOffset + position];
        int entriesEnd = this.offsets[this.startOffset + position + length];
        int entryCount = entriesEnd - entriesStart;
        return this.keyBlock.getRegionSizeInBytes(entriesStart, entryCount) + this.valueBlock.getRegionSizeInBytes(entriesStart, entryCount) + 5L * (long)length + 8L * (long)entryCount;
    }

    @Override
    public OptionalInt fixedSizeInBytesPerPosition() {
        return OptionalInt.empty();
    }

    private OptionalInt keyAndValueFixedSizeInBytesPerRow() {
        OptionalInt keyFixedSizePerRow = this.keyBlock.fixedSizeInBytesPerPosition();
        if (keyFixedSizePerRow.isEmpty()) {
            return OptionalInt.empty();
        }
        OptionalInt valueFixedSizePerRow = this.valueBlock.fixedSizeInBytesPerPosition();
        if (valueFixedSizePerRow.isEmpty()) {
            return OptionalInt.empty();
        }
        return OptionalInt.of(keyFixedSizePerRow.getAsInt() + valueFixedSizePerRow.getAsInt());
    }

    @Override
    public long getPositionsSizeInBytes(boolean[] positions, int selectedMapPositions) {
        long keyAndValuesSizeInBytes;
        int selectedEntryCount;
        int positionCount = this.getPositionCount();
        BlockUtil.checkValidPositions(positions, positionCount);
        if (selectedMapPositions == 0) {
            return 0L;
        }
        if (selectedMapPositions == positionCount) {
            return this.getSizeInBytes();
        }
        int[] offsets = this.offsets;
        int offsetBase = this.startOffset;
        OptionalInt fixedKeyAndValueSizePerRow = this.keyAndValueFixedSizeInBytesPerRow();
        if (fixedKeyAndValueSizePerRow.isPresent()) {
            selectedEntryCount = BlockUtil.countSelectedPositionsFromOffsets(positions, offsets, offsetBase);
            keyAndValuesSizeInBytes = (long)fixedKeyAndValueSizePerRow.getAsInt() * (long)selectedEntryCount;
        } else {
            boolean[] entryPositions = new boolean[this.keyBlock.getPositionCount()];
            selectedEntryCount = BlockUtil.countAndMarkSelectedPositionsFromOffsets(positions, offsets, offsetBase, entryPositions);
            keyAndValuesSizeInBytes = this.keyBlock.getPositionsSizeInBytes(entryPositions, selectedEntryCount) + this.valueBlock.getPositionsSizeInBytes(entryPositions, selectedEntryCount);
        }
        return keyAndValuesSizeInBytes + 5L * (long)selectedMapPositions + 8L * (long)selectedEntryCount;
    }

    @Override
    public MapBlock copyRegion(int position, int length) {
        int positionCount = this.getPositionCount();
        BlockUtil.checkValidRegion(positionCount, position, length);
        int startValueOffset = this.getOffset(position);
        int endValueOffset = this.getOffset(position + length);
        Block newKeys = this.keyBlock.copyRegion(startValueOffset, endValueOffset - startValueOffset);
        Block newValues = this.valueBlock.copyRegion(startValueOffset, endValueOffset - startValueOffset);
        int[] newOffsets = BlockUtil.compactOffsets(this.offsets, position + this.startOffset, length);
        boolean[] mapIsNull = this.mapIsNull;
        boolean[] newMapIsNull = mapIsNull == null ? null : BlockUtil.compactArray(mapIsNull, position + this.startOffset, length);
        int[] rawHashTables = this.hashTables.tryGet().orElse(null);
        int[] newRawHashTables = null;
        int expectedNewHashTableEntries = (endValueOffset - startValueOffset) * 2;
        if (rawHashTables != null) {
            newRawHashTables = BlockUtil.compactArray(rawHashTables, startValueOffset * 2, expectedNewHashTableEntries);
        }
        if (newKeys == this.keyBlock && newValues == this.valueBlock && newOffsets == this.offsets && newMapIsNull == mapIsNull && newRawHashTables == rawHashTables) {
            return this;
        }
        return MapBlock.createMapBlockInternal(this.mapType, 0, length, Optional.ofNullable(newMapIsNull), newOffsets, newKeys, newValues, new MapHashTables(this.mapType, MapHashTables.HashBuildMode.DUPLICATE_NOT_CHECKED, length, Optional.ofNullable(newRawHashTables)));
    }

    public SqlMap getMap(int position) {
        BlockUtil.checkReadablePosition(this, position);
        int startEntryOffset = this.getOffset(position);
        int endEntryOffset = this.getOffset(position + 1);
        return new SqlMap(this.mapType, this.keyBlock, this.valueBlock, new SqlMap.HashTableSupplier(this), startEntryOffset, endEntryOffset - startEntryOffset);
    }

    @Override
    public MapBlock getSingleValueBlock(int position) {
        BlockUtil.checkReadablePosition(this, position);
        int startValueOffset = this.getOffset(position);
        int endValueOffset = this.getOffset(position + 1);
        int valueLength = endValueOffset - startValueOffset;
        Block newKeys = this.keyBlock.copyRegion(startValueOffset, valueLength);
        Block newValues = this.valueBlock.copyRegion(startValueOffset, valueLength);
        int[] rawHashTables = this.hashTables.tryGet().orElse(null);
        int[] newRawHashTables = null;
        if (rawHashTables != null) {
            newRawHashTables = Arrays.copyOfRange(rawHashTables, startValueOffset * 2, endValueOffset * 2);
        }
        return MapBlock.createMapBlockInternal(this.mapType, 0, 1, Optional.of(new boolean[]{this.isNull(position)}), new int[]{0, valueLength}, newKeys, newValues, new MapHashTables(this.mapType, MapHashTables.HashBuildMode.DUPLICATE_NOT_CHECKED, 1, Optional.ofNullable(newRawHashTables)));
    }

    @Override
    public long getEstimatedDataSizeForStats(int position) {
        BlockUtil.checkReadablePosition(this, position);
        if (this.isNull(position)) {
            return 0L;
        }
        int startValueOffset = this.getOffset(position);
        int endValueOffset = this.getOffset(position + 1);
        long size = 0L;
        Block rawKeyBlock = this.keyBlock;
        Block rawValueBlock = this.valueBlock;
        for (int i = startValueOffset; i < endValueOffset; ++i) {
            size += rawKeyBlock.getEstimatedDataSizeForStats(i);
            size += rawValueBlock.getEstimatedDataSizeForStats(i);
        }
        return size;
    }

    @Override
    public boolean isNull(int position) {
        BlockUtil.checkReadablePosition(this, position);
        boolean[] mapIsNull = this.mapIsNull;
        return mapIsNull != null && mapIsNull[position + this.startOffset];
    }

    @Override
    public MapBlock getUnderlyingValueBlock() {
        return this;
    }

    public boolean isHashTablesPresent() {
        return this.hashTables.tryGet().isPresent();
    }
}

