/*
 * Decompiled with CFR 0.152.
 */
package io.trino.execution.executor.scheduler;

import com.google.common.base.Preconditions;
import com.google.common.base.Ticker;
import com.google.common.collect.ImmutableSet;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.errorprone.annotations.ThreadSafe;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import io.airlift.log.Logger;
import io.trino.execution.executor.scheduler.BlockingSchedulingQueue;
import io.trino.execution.executor.scheduler.Gate;
import io.trino.execution.executor.scheduler.Group;
import io.trino.execution.executor.scheduler.Reservation;
import io.trino.execution.executor.scheduler.Schedulable;
import io.trino.execution.executor.scheduler.SchedulerContext;
import io.trino.execution.executor.scheduler.TaskControl;
import java.util.Objects;
import java.util.Set;
import java.util.StringJoiner;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

@ThreadSafe
public final class FairScheduler
implements AutoCloseable {
    private static final Logger LOG = Logger.get(FairScheduler.class);
    public static final long QUANTUM_NANOS = TimeUnit.MILLISECONDS.toNanos(1000L);
    private final ExecutorService schedulerExecutor;
    private final ListeningExecutorService taskExecutor;
    private final BlockingSchedulingQueue<Group, TaskControl> queue = new BlockingSchedulingQueue();
    private final Reservation<TaskControl> concurrencyControl;
    private final Ticker ticker;
    private final Gate paused = new Gate(true);
    @GuardedBy(value="this")
    private boolean closed;

    public FairScheduler(int maxConcurrentTasks, String threadNameFormat, Ticker ticker) {
        this.ticker = Objects.requireNonNull(ticker, "ticker is null");
        this.concurrencyControl = new Reservation(maxConcurrentTasks);
        this.schedulerExecutor = Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("fair-scheduler-%d").setDaemon(true).build());
        this.taskExecutor = MoreExecutors.listeningDecorator((ExecutorService)Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat(threadNameFormat).setDaemon(true).build()));
    }

    public static FairScheduler newInstance(int maxConcurrentTasks) {
        return FairScheduler.newInstance(maxConcurrentTasks, Ticker.systemTicker());
    }

    public static FairScheduler newInstance(int maxConcurrentTasks, Ticker ticker) {
        FairScheduler scheduler = new FairScheduler(maxConcurrentTasks, "fair-scheduler-runner-%d", ticker);
        scheduler.start();
        return scheduler;
    }

    public void start() {
        this.schedulerExecutor.submit(this::runScheduler);
    }

    public void pause() {
        this.paused.close();
    }

    public void resume() {
        this.paused.open();
    }

    @Override
    public synchronized void close() {
        if (this.closed) {
            return;
        }
        this.closed = true;
        Set<TaskControl> tasks = this.queue.finishAll();
        for (TaskControl task : tasks) {
            task.cancel();
        }
        this.taskExecutor.shutdownNow();
        this.schedulerExecutor.shutdownNow();
    }

    public synchronized Group createGroup(String name) {
        Preconditions.checkArgument((!this.closed ? 1 : 0) != 0, (Object)"Already closed");
        Group group = new Group(name);
        this.queue.startGroup(group);
        return group;
    }

    public synchronized void removeGroup(Group group) {
        Preconditions.checkArgument((!this.closed ? 1 : 0) != 0, (Object)"Already closed");
        Set<TaskControl> tasks = this.queue.finishGroup(group);
        for (TaskControl task : tasks) {
            task.cancel();
        }
    }

    public Set<Integer> getTasks(Group group) {
        return (Set)this.queue.getTasks(group).stream().map(TaskControl::id).collect(ImmutableSet.toImmutableSet());
    }

    public synchronized ListenableFuture<Void> submit(Group group, int id, Schedulable runner) {
        Preconditions.checkArgument((!this.closed ? 1 : 0) != 0, (Object)"Already closed");
        TaskControl task = new TaskControl(group, id, this.ticker);
        return this.taskExecutor.submit(() -> this.runTask(runner, task), null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void runTask(Schedulable runner, TaskControl task) {
        task.setThread(Thread.currentThread());
        if (!this.makeRunnableAndAwait(task, 0L)) {
            return;
        }
        SchedulerContext context = new SchedulerContext(this, task);
        try {
            runner.run(context);
        }
        catch (Exception e) {
            LOG.error((Throwable)e);
        }
        finally {
            if (task.getState() == TaskControl.State.RUNNING) {
                this.concurrencyControl.release(task);
            }
            this.queue.finish(task.group(), task);
            task.transitionToFinished();
        }
    }

    private boolean makeRunnableAndAwait(TaskControl task, long deltaWeight) {
        if (!task.transitionToWaiting()) {
            return false;
        }
        if (!this.queue.enqueue(task.group(), task, deltaWeight)) {
            return false;
        }
        return this.awaitReadyAndTransitionToRunning(task);
    }

    private boolean awaitReadyAndTransitionToRunning(TaskControl task) {
        if (!task.awaitReady()) {
            if (task.isReady()) {
                this.concurrencyControl.release(task);
            }
            return false;
        }
        if (!task.transitionToRunning()) {
            this.concurrencyControl.release(task);
            return false;
        }
        return true;
    }

    boolean yield(TaskControl task) {
        Preconditions.checkState((task.getThread() == Thread.currentThread() ? 1 : 0) != 0, (Object)"yield() may only be called from the task thread");
        long delta = task.elapsed();
        if (delta < QUANTUM_NANOS) {
            return true;
        }
        this.concurrencyControl.release(task);
        return this.makeRunnableAndAwait(task, delta);
    }

    boolean block(TaskControl task, ListenableFuture<?> future) {
        Preconditions.checkState((task.getThread() == Thread.currentThread() ? 1 : 0) != 0, (Object)"block() may only be called from the task thread");
        long delta = task.elapsed();
        this.concurrencyControl.release(task);
        if (!task.transitionToBlocked()) {
            return false;
        }
        if (!this.queue.block(task.group(), task, delta)) {
            return false;
        }
        future.addListener(task::markUnblocked, MoreExecutors.directExecutor());
        task.awaitUnblock();
        return this.makeRunnableAndAwait(task, 0L);
    }

    private void runScheduler() {
        while (true) {
            try {
                while (true) {
                    this.paused.awaitOpen();
                    this.concurrencyControl.reserve();
                    TaskControl task = this.queue.dequeue(QUANTUM_NANOS);
                    this.concurrencyControl.register(task);
                    if (task.markReady()) continue;
                    this.concurrencyControl.release(task);
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return;
            }
            catch (Exception e) {
                LOG.error((Throwable)e);
                continue;
            }
            break;
        }
    }

    long getStartNanos(TaskControl task) {
        return task.getStartNanos();
    }

    long getScheduledNanos(TaskControl task) {
        return task.getScheduledNanos();
    }

    long getWaitNanos(TaskControl task) {
        return task.getWaitNanos();
    }

    long getBlockedNanos(TaskControl task) {
        return task.getBlockedNanos();
    }

    public String toString() {
        return new StringJoiner(", ", FairScheduler.class.getSimpleName() + "[", "]").add("queue=" + String.valueOf(this.queue)).add("concurrencyControl=" + String.valueOf(this.concurrencyControl)).add("closed=" + this.closed).toString();
    }
}

