/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import io.airlift.slice.SizeOf;
import io.trino.array.IntBigArray;
import io.trino.operator.IdRegistry;
import io.trino.operator.RowIdComparisonStrategy;
import io.trino.operator.RowIdHashStrategy;
import io.trino.operator.RowReference;
import io.trino.spi.Page;
import io.trino.util.LongBigArrayFIFOQueue;
import it.unimi.dsi.fastutil.ints.IntIterator;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import jakarta.annotation.Nullable;
import java.util.Arrays;

public final class RowReferencePageManager {
    private static final long INSTANCE_SIZE = SizeOf.instanceSize(RowReferencePageManager.class);
    private static final long PAGE_ACCOUNTING_INSTANCE_SIZE = SizeOf.instanceSize(PageAccounting.class);
    private static final int RESERVED_ROW_ID_FOR_CURSOR = -1;
    private final IdRegistry<PageAccounting> pages = new IdRegistry();
    private final RowIdBuffer rowIdBuffer = new RowIdBuffer();
    private final IntHashSet compactionCandidates = new IntHashSet();
    @Nullable
    private LoadCursor currentCursor;
    private long pageBytes;

    public LoadCursor add(Page page) {
        return this.add(page, 0);
    }

    public LoadCursor add(Page page, int startingPosition) {
        Preconditions.checkState((this.currentCursor == null ? 1 : 0) != 0, (Object)"Cursor still active");
        Preconditions.checkArgument((startingPosition >= 0 && startingPosition <= page.getPositionCount() ? 1 : 0) != 0, (String)"invalid startingPosition: %s", (int)startingPosition);
        PageAccounting pageAccounting = this.pages.allocateId(id -> new PageAccounting(id, page));
        pageAccounting.lockPage();
        this.currentCursor = new LoadCursor(pageAccounting, startingPosition, () -> {
            Preconditions.checkState((this.currentCursor != null ? 1 : 0) != 0);
            pageAccounting.unlockPage();
            pageAccounting.loadPageLoadIfNeeded();
            this.pageBytes += pageAccounting.sizeOf();
            this.currentCursor = null;
            this.checkPageMaintenance(pageAccounting);
        });
        return this.currentCursor;
    }

    public void dereference(long rowId) {
        PageAccounting pageAccounting = this.pages.get(this.rowIdBuffer.getPageId(rowId));
        pageAccounting.dereference(rowId);
        this.checkPageMaintenance(pageAccounting);
    }

    private void checkPageMaintenance(PageAccounting pageAccounting) {
        int pageId = pageAccounting.getPageId();
        if (pageAccounting.isPruneEligible()) {
            this.compactionCandidates.remove(pageId);
            this.pages.deallocate(pageId);
            this.pageBytes -= pageAccounting.sizeOf();
        } else if (pageAccounting.isCompactionEligible()) {
            this.compactionCandidates.add(pageId);
        }
    }

    public Page getPage(long rowId) {
        if (RowReferencePageManager.isCursorRowId(rowId)) {
            Preconditions.checkState((this.currentCursor != null ? 1 : 0) != 0, (Object)"No active cursor");
            return this.currentCursor.getPage();
        }
        int pageId = this.rowIdBuffer.getPageId(rowId);
        return this.pages.get(pageId).getPage();
    }

    public int getPosition(long rowId) {
        if (RowReferencePageManager.isCursorRowId(rowId)) {
            Preconditions.checkState((this.currentCursor != null ? 1 : 0) != 0, (Object)"No active cursor");
            return this.currentCursor.getCurrentPosition();
        }
        return this.rowIdBuffer.getPosition(rowId);
    }

    private static boolean isCursorRowId(long rowId) {
        return rowId == -1L;
    }

    public void compactIfNeeded() {
        IntIterator iterator = this.compactionCandidates.iterator();
        while (iterator.hasNext()) {
            int pageId = iterator.nextInt();
            PageAccounting pageAccounting = this.pages.get(pageId);
            this.pageBytes -= pageAccounting.sizeOf();
            pageAccounting.compact();
            this.pageBytes += pageAccounting.sizeOf();
        }
        this.compactionCandidates.clear();
    }

    @VisibleForTesting
    int getCompactionCandidateCount() {
        return this.compactionCandidates.size();
    }

    @VisibleForTesting
    long getPageBytes() {
        return this.pageBytes;
    }

    public long sizeOf() {
        return INSTANCE_SIZE + this.pageBytes + this.pages.sizeOf() + this.rowIdBuffer.sizeOf() + this.compactionCandidates.sizeOf();
    }

    private static class RowIdBuffer {
        public static final long UNKNOWN_ID = -1L;
        private static final long INSTANCE_SIZE = SizeOf.instanceSize(RowIdBuffer.class);
        private final IntBigArray buffer = new IntBigArray();
        private final LongBigArrayFIFOQueue emptySlots = new LongBigArrayFIFOQueue();
        private long capacity;

        private RowIdBuffer() {
        }

        public long allocateRowId(int pageId, int position) {
            long newRowId;
            if (!this.emptySlots.isEmpty()) {
                newRowId = this.emptySlots.dequeueLong();
            } else {
                newRowId = this.capacity++;
                this.buffer.ensureCapacity(this.capacity * 2L);
            }
            this.setPageId(newRowId, pageId);
            this.setPosition(newRowId, position);
            return newRowId;
        }

        public void deallocate(long rowId) {
            this.emptySlots.enqueue(rowId);
        }

        public int getPageId(long rowId) {
            return this.buffer.get(rowId * 2L);
        }

        public void setPageId(long rowId, int pageId) {
            this.buffer.set(rowId * 2L, pageId);
        }

        public int getPosition(long rowId) {
            return this.buffer.get(rowId * 2L + 1L);
        }

        public void setPosition(long rowId, int position) {
            this.buffer.set(rowId * 2L + 1L, position);
        }

        public long sizeOf() {
            return INSTANCE_SIZE + this.buffer.sizeOf() + this.emptySlots.sizeOf();
        }
    }

    private static class IntHashSet
    extends IntOpenHashSet {
        private static final long INSTANCE_SIZE = SizeOf.instanceSize(IntHashSet.class);

        private IntHashSet() {
        }

        public long sizeOf() {
            return INSTANCE_SIZE + SizeOf.sizeOf((int[])this.key);
        }
    }

    public static final class LoadCursor
    implements RowReference,
    AutoCloseable {
        private final PageAccounting pageAccounting;
        private final Runnable closeCallback;
        private int currentPosition;

        private LoadCursor(PageAccounting pageAccounting, int startingPosition, Runnable closeCallback) {
            this.pageAccounting = pageAccounting;
            this.currentPosition = startingPosition - 1;
            this.closeCallback = closeCallback;
        }

        private Page getPage() {
            return this.pageAccounting.getPage();
        }

        private int getCurrentPosition() {
            Preconditions.checkState((this.currentPosition >= 0 ? 1 : 0) != 0, (Object)"Not yet advanced");
            return this.currentPosition;
        }

        public boolean advance() {
            if (this.currentPosition >= this.pageAccounting.getPage().getPositionCount() - 1) {
                return false;
            }
            ++this.currentPosition;
            return true;
        }

        @Override
        public int compareTo(RowIdComparisonStrategy strategy, long rowId) {
            Preconditions.checkState((this.currentPosition >= 0 ? 1 : 0) != 0, (Object)"Not yet advanced");
            return strategy.compare(-1L, rowId);
        }

        @Override
        public boolean equals(RowIdHashStrategy strategy, long rowId) {
            Preconditions.checkState((this.currentPosition >= 0 ? 1 : 0) != 0, (Object)"Not yet advanced");
            return strategy.equals(-1L, rowId);
        }

        @Override
        public long hash(RowIdHashStrategy strategy) {
            Preconditions.checkState((this.currentPosition >= 0 ? 1 : 0) != 0, (Object)"Not yet advanced");
            return strategy.hashCode(-1L);
        }

        @Override
        public long allocateRowId() {
            Preconditions.checkState((this.currentPosition >= 0 ? 1 : 0) != 0, (Object)"Not yet advanced");
            return this.pageAccounting.referencePosition(this.currentPosition);
        }

        @Override
        public void close() {
            this.closeCallback.run();
        }
    }

    private final class PageAccounting {
        private static final int COMPACTION_MIN_FILL_MULTIPLIER = 2;
        private final int pageId;
        private Page page;
        private boolean isPageLoaded;
        private long[] rowIds;
        private boolean lockedPage = true;
        private int activePositions;

        public PageAccounting(int pageId, Page page) {
            this.pageId = pageId;
            this.page = page;
            this.rowIds = new long[page.getPositionCount()];
            Arrays.fill(this.rowIds, -1L);
        }

        public long referencePosition(int position) {
            long rowId = this.rowIds[position];
            if (rowId == -1L) {
                this.rowIds[position] = rowId = RowReferencePageManager.this.rowIdBuffer.allocateRowId(this.pageId, position);
                ++this.activePositions;
            }
            return rowId;
        }

        public void lockPage() {
            this.lockedPage = true;
        }

        public void unlockPage() {
            this.lockedPage = false;
        }

        public int getPageId() {
            return this.pageId;
        }

        public Page getPage() {
            return this.page;
        }

        public void dereference(long rowId) {
            int position = RowReferencePageManager.this.rowIdBuffer.getPosition(rowId);
            Preconditions.checkArgument((rowId == this.rowIds[position] ? 1 : 0) != 0, (Object)"rowId does not match this page");
            this.rowIds[position] = -1L;
            --this.activePositions;
            RowReferencePageManager.this.rowIdBuffer.deallocate(rowId);
        }

        public boolean isPruneEligible() {
            return !this.lockedPage && this.activePositions == 0;
        }

        public boolean isCompactionEligible() {
            return !this.lockedPage && this.activePositions * 2 < this.page.getPositionCount();
        }

        public void loadPageLoadIfNeeded() {
            if (!this.isPageLoaded && this.activePositions > 0) {
                this.page = this.page.getLoadedPage();
                this.isPageLoaded = true;
            }
        }

        public void compact() {
            int i;
            long rowId;
            Preconditions.checkState((!this.lockedPage ? 1 : 0) != 0, (Object)"Should not attempt compaction when page is locked");
            if (this.activePositions == this.page.getPositionCount()) {
                return;
            }
            this.loadPageLoadIfNeeded();
            int newIndex = 0;
            int[] positionsToKeep = new int[this.activePositions];
            long[] newRowIds = new long[this.activePositions];
            for (i = 0; i < this.page.getPositionCount() && newIndex < positionsToKeep.length; newIndex += rowId == -1L ? 0 : 1, ++i) {
                rowId = this.rowIds[i];
                positionsToKeep[newIndex] = i;
                newRowIds[newIndex] = rowId;
            }
            Verify.verify((newIndex == this.activePositions ? 1 : 0) != 0);
            for (i = 0; i < newRowIds.length; ++i) {
                RowReferencePageManager.this.rowIdBuffer.setPosition(newRowIds[i], i);
            }
            this.page = this.page.copyPositions(positionsToKeep, 0, positionsToKeep.length);
            this.rowIds = newRowIds;
        }

        public long sizeOf() {
            long loadedPageSize = this.isPageLoaded ? this.page.getSizeInBytes() : 0L;
            return PAGE_ACCOUNTING_INSTANCE_SIZE + loadedPageSize + SizeOf.sizeOf((long[])this.rowIds);
        }
    }
}

