/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator.scalar;

import io.airlift.concurrent.ThreadLocalCache;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.airlift.units.Duration;
import io.trino.operator.scalar.MathFunctions;
import io.trino.operator.scalar.QuarterOfYearDateTimeField;
import io.trino.operator.scalar.timestamptz.CurrentTimestamp;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.function.Description;
import io.trino.spi.function.LiteralParameter;
import io.trino.spi.function.LiteralParameters;
import io.trino.spi.function.ScalarFunction;
import io.trino.spi.function.SqlType;
import io.trino.spi.type.DateTimeEncoding;
import io.trino.spi.type.Int128;
import io.trino.spi.type.Int128Math;
import io.trino.spi.type.LongTimestampWithTimeZone;
import io.trino.spi.type.TimeZoneKey;
import io.trino.type.DateTimes;
import io.trino.util.DateTimeUtils;
import io.trino.util.DateTimeZoneIndex;
import java.math.BigInteger;
import java.time.DateTimeException;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.time.temporal.TemporalAccessor;
import java.util.Locale;
import java.util.concurrent.TimeUnit;
import org.joda.time.Chronology;
import org.joda.time.DateTime;
import org.joda.time.DateTimeField;
import org.joda.time.Days;
import org.joda.time.LocalDate;
import org.joda.time.ReadablePartial;
import org.joda.time.chrono.ISOChronology;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.DateTimeFormatterBuilder;
import org.joda.time.format.ISODateTimeFormat;

public final class DateTimeFunctions {
    private static final ThreadLocalCache<Slice, DateTimeFormatter> DATETIME_FORMATTER_CACHE = new ThreadLocalCache(100, DateTimeFunctions::createDateTimeFormatter);
    private static final ISOChronology UTC_CHRONOLOGY = ISOChronology.getInstanceUTC();
    private static final DateTimeField DAY_OF_WEEK = UTC_CHRONOLOGY.dayOfWeek();
    private static final DateTimeField DAY_OF_MONTH = UTC_CHRONOLOGY.dayOfMonth();
    private static final DateTimeField DAY_OF_YEAR = UTC_CHRONOLOGY.dayOfYear();
    private static final DateTimeField WEEK_OF_YEAR = UTC_CHRONOLOGY.weekOfWeekyear();
    private static final DateTimeField YEAR_OF_WEEK = UTC_CHRONOLOGY.weekyear();
    private static final DateTimeField MONTH_OF_YEAR = UTC_CHRONOLOGY.monthOfYear();
    private static final DateTimeField QUARTER = QuarterOfYearDateTimeField.QUARTER_OF_YEAR.getField((Chronology)UTC_CHRONOLOGY);
    private static final DateTimeField YEAR = UTC_CHRONOLOGY.year();
    private static final int MILLISECONDS_IN_SECOND = 1000;
    private static final int MILLISECONDS_IN_MINUTE = 60000;
    private static final int MILLISECONDS_IN_HOUR = 3600000;
    private static final int MILLISECONDS_IN_DAY = 86400000;
    private static final int PIVOT_YEAR = 2020;
    private static final Slice ISO_8601_DATE_FORMAT = Slices.utf8Slice((String)"%Y-%m-%d");

    private DateTimeFunctions() {
    }

    @ScalarFunction
    @Description(value="Current timestamp with time zone")
    @SqlType(value="timestamp(3) with time zone")
    public static long now(ConnectorSession session) {
        return CurrentTimestamp.shortTimestamp(3L, session, null);
    }

    @Description(value="Current date")
    @ScalarFunction
    @SqlType(value="date")
    public static long currentDate(ConnectorSession session) {
        ISOChronology chronology = DateTimeZoneIndex.getChronology(session.getTimeZoneKey());
        LocalDate currentDate = new DateTime(session.getStart().toEpochMilli(), (Chronology)chronology).toLocalDate();
        return Days.daysBetween((ReadablePartial)new LocalDate(1970, 1, 1), (ReadablePartial)currentDate).getDays();
    }

    @Description(value="Current time zone")
    @ScalarFunction(value="current_timezone")
    @SqlType(value="varchar")
    public static Slice currentTimeZone(ConnectorSession session) {
        return Slices.utf8Slice((String)session.getTimeZoneKey().getId());
    }

    @ScalarFunction(value="from_unixtime")
    @SqlType(value="timestamp(3) with time zone")
    public static long fromUnixTime(ConnectorSession session, @SqlType(value="double") double unixTime) {
        try {
            return DateTimeEncoding.packDateTimeWithZone((long)Math.round(unixTime * 1000.0), (TimeZoneKey)session.getTimeZoneKey());
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
    }

    @ScalarFunction(value="from_unixtime")
    @SqlType(value="timestamp(3) with time zone")
    public static long fromUnixTime(@SqlType(value="double") double unixTime, @SqlType(value="bigint") long hoursOffset, @SqlType(value="bigint") long minutesOffset) {
        try {
            TimeZoneKey timeZoneKey = TimeZoneKey.getTimeZoneKeyForOffset((long)(hoursOffset * 60L + minutesOffset));
            return DateTimeEncoding.packDateTimeWithZone((long)Math.round(unixTime * 1000.0), (TimeZoneKey)timeZoneKey);
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
    }

    @ScalarFunction(value="from_unixtime")
    @LiteralParameters(value={"x"})
    @SqlType(value="timestamp(3) with time zone")
    public static long fromUnixTime(@SqlType(value="double") double unixTime, @SqlType(value="varchar(x)") Slice zoneId) {
        try {
            return DateTimeEncoding.packDateTimeWithZone((long)Math.round(unixTime * 1000.0), (String)zoneId.toStringUtf8());
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
    }

    @ScalarFunction(value="from_unixtime_nanos")
    @SqlType(value="timestamp(9) with time zone")
    public static LongTimestampWithTimeZone fromUnixtimeNanosLong(ConnectorSession session, @SqlType(value="bigint") long unixTimeNanos) {
        long epochSeconds = Math.floorDiv(unixTimeNanos, 1000000000L);
        long nanosOfSecond = Math.floorMod(unixTimeNanos, 1000000000L);
        long picosOfSecond = nanosOfSecond * 1000L;
        return DateTimes.longTimestampWithTimeZone(epochSeconds, picosOfSecond, session.getTimeZoneKey().getZoneId());
    }

    @ScalarFunction(value="to_iso8601")
    @SqlType(value="varchar(16)")
    public static Slice toISO8601FromDate(@SqlType(value="date") long date) {
        DateTimeFormatter formatter = ISODateTimeFormat.date().withChronology((Chronology)UTC_CHRONOLOGY);
        return Slices.utf8Slice((String)formatter.print(TimeUnit.DAYS.toMillis(date)));
    }

    @ScalarFunction(value="from_iso8601_timestamp")
    @LiteralParameters(value={"x"})
    @SqlType(value="timestamp(3) with time zone")
    public static long fromISO8601Timestamp(ConnectorSession session, @SqlType(value="varchar(x)") Slice iso8601DateTime) {
        DateTimeFormatter formatter = ISODateTimeFormat.dateTimeParser().withChronology((Chronology)DateTimeZoneIndex.getChronology(session.getTimeZoneKey())).withOffsetParsed();
        try {
            return DateTimeZoneIndex.packDateTimeWithZone(DateTimeFunctions.parseDateTimeHelper(formatter, iso8601DateTime.toStringUtf8()));
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
    }

    @ScalarFunction(value="from_iso8601_timestamp_nanos")
    @LiteralParameters(value={"x"})
    @SqlType(value="timestamp(9) with time zone")
    public static LongTimestampWithTimeZone fromIso8601TimestampNanos(ConnectorSession session, @SqlType(value="varchar(x)") Slice iso8601DateTime) {
        TemporalAccessor parsedDatetime;
        java.time.format.DateTimeFormatter formatter = java.time.format.DateTimeFormatter.ISO_DATE_TIME;
        String datetimeString = iso8601DateTime.toStringUtf8();
        try {
            parsedDatetime = formatter.parseBest(datetimeString, ZonedDateTime::from, LocalDateTime::from);
        }
        catch (DateTimeParseException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
        ZonedDateTime zonedDatetime = parsedDatetime instanceof ZonedDateTime ? (ZonedDateTime)parsedDatetime : ((LocalDateTime)parsedDatetime).atZone(session.getTimeZoneKey().getZoneId());
        long picosOfSecond = (long)zonedDatetime.getNano() * 1000L;
        TimeZoneKey zone = TimeZoneKey.getTimeZoneKey((String)zonedDatetime.getZone().getId());
        return LongTimestampWithTimeZone.fromEpochSecondsAndFraction((long)zonedDatetime.toEpochSecond(), (long)picosOfSecond, (TimeZoneKey)zone);
    }

    @ScalarFunction(value="from_iso8601_date")
    @LiteralParameters(value={"x"})
    @SqlType(value="date")
    public static long fromISO8601Date(@SqlType(value="varchar(x)") Slice iso8601DateTime) {
        DateTimeFormatter formatter = ISODateTimeFormat.dateElementParser().withChronology((Chronology)UTC_CHRONOLOGY);
        DateTime dateTime = DateTimeFunctions.parseDateTimeHelper(formatter, iso8601DateTime.toStringUtf8());
        return TimeUnit.MILLISECONDS.toDays(dateTime.getMillis());
    }

    @Description(value="Truncate to the specified precision in the session timezone")
    @ScalarFunction(value="date_trunc")
    @LiteralParameters(value={"x"})
    @SqlType(value="date")
    public static long truncateDate(@SqlType(value="varchar(x)") Slice unit, @SqlType(value="date") long date) {
        long millis = DateTimeFunctions.getDateField(UTC_CHRONOLOGY, unit).roundFloor(TimeUnit.DAYS.toMillis(date));
        return TimeUnit.MILLISECONDS.toDays(millis);
    }

    @Description(value="Add the specified amount of date to the given date")
    @LiteralParameters(value={"x"})
    @ScalarFunction(value="date_add")
    @SqlType(value="date")
    public static long addFieldValueDate(@SqlType(value="varchar(x)") Slice unit, @SqlType(value="bigint") long value, @SqlType(value="date") long date) {
        long millis = DateTimeFunctions.getDateField(UTC_CHRONOLOGY, unit).add(TimeUnit.DAYS.toMillis(date), Math.toIntExact(value));
        return TimeUnit.MILLISECONDS.toDays(millis);
    }

    @Description(value="Difference of the given dates in the given unit")
    @ScalarFunction(value="date_diff")
    @LiteralParameters(value={"x"})
    @SqlType(value="bigint")
    public static long diffDate(@SqlType(value="varchar(x)") Slice unit, @SqlType(value="date") long date1, @SqlType(value="date") long date2) {
        return DateTimeFunctions.getDateField(UTC_CHRONOLOGY, unit).getDifferenceAsLong(TimeUnit.DAYS.toMillis(date2), TimeUnit.DAYS.toMillis(date1));
    }

    private static DateTimeField getDateField(ISOChronology chronology, Slice unit) {
        String unitString;
        switch (unitString = unit.toStringUtf8().toLowerCase(Locale.ENGLISH)) {
            case "day": {
                return chronology.dayOfMonth();
            }
            case "week": {
                return chronology.weekOfWeekyear();
            }
            case "month": {
                return chronology.monthOfYear();
            }
            case "quarter": {
                return QuarterOfYearDateTimeField.QUARTER_OF_YEAR.getField((Chronology)chronology);
            }
            case "year": {
                return chronology.year();
            }
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "'" + unitString + "' is not a valid DATE field");
    }

    public static DateTimeField getTimestampField(ISOChronology chronology, Slice unit) {
        String unitString;
        switch (unitString = unit.toStringUtf8().toLowerCase(Locale.ENGLISH)) {
            case "millisecond": {
                return chronology.millisOfSecond();
            }
            case "second": {
                return chronology.secondOfMinute();
            }
            case "minute": {
                return chronology.minuteOfHour();
            }
            case "hour": {
                return chronology.hourOfDay();
            }
            case "day": {
                return chronology.dayOfMonth();
            }
            case "week": {
                return chronology.weekOfWeekyear();
            }
            case "month": {
                return chronology.monthOfYear();
            }
            case "quarter": {
                return QuarterOfYearDateTimeField.QUARTER_OF_YEAR.getField((Chronology)chronology);
            }
            case "year": {
                return chronology.year();
            }
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "'" + unitString + "' is not a valid TIMESTAMP field");
    }

    @Description(value="Parses the specified date/time by the given format")
    @ScalarFunction
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="timestamp(3) with time zone")
    public static long parseDatetime(ConnectorSession session, @SqlType(value="varchar(x)") Slice datetime, @SqlType(value="varchar(y)") Slice formatString) {
        try {
            return DateTimeZoneIndex.packDateTimeWithZone(DateTimeFunctions.parseDateTimeHelper(DateTimeFormat.forPattern((String)formatString.toStringUtf8()).withChronology((Chronology)DateTimeZoneIndex.getChronology(session.getTimeZoneKey())).withOffsetParsed().withLocale(session.getLocale()), datetime.toStringUtf8()));
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
    }

    private static DateTime parseDateTimeHelper(DateTimeFormatter formatter, String datetimeString) {
        try {
            return formatter.parseDateTime(datetimeString);
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
    }

    public static Slice dateFormat(ISOChronology chronology, Locale locale, long timestamp, Slice formatString) {
        DateTimeFormatter formatter = ((DateTimeFormatter)DATETIME_FORMATTER_CACHE.get((Object)formatString)).withChronology((Chronology)chronology).withLocale(locale);
        return Slices.utf8Slice((String)formatter.print(timestamp));
    }

    @ScalarFunction
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="timestamp(3)")
    public static long dateParse(ConnectorSession session, @SqlType(value="varchar(x)") Slice dateTime, @SqlType(value="varchar(y)") Slice formatString) {
        if (ISO_8601_DATE_FORMAT.equals((Object)formatString)) {
            try {
                long days = DateTimeUtils.parseDate(dateTime.toStringUtf8());
                return DateTimes.scaleEpochMillisToMicros(days * 86400000L);
            }
            catch (ArithmeticException | IllegalArgumentException | DateTimeException e) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
            }
        }
        DateTimeFormatter formatter = ((DateTimeFormatter)DATETIME_FORMATTER_CACHE.get((Object)formatString)).withZoneUTC().withLocale(session.getLocale());
        try {
            return DateTimes.scaleEpochMillisToMicros(formatter.parseMillis(dateTime.toStringUtf8()));
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
    }

    @Description(value="Millisecond of the second of the given interval")
    @ScalarFunction(value="millisecond")
    @SqlType(value="bigint")
    public static long millisecondFromInterval(@SqlType(value="interval day to second") long milliseconds) {
        return milliseconds % 1000L;
    }

    @Description(value="Second of the minute of the given interval")
    @ScalarFunction(value="second")
    @SqlType(value="bigint")
    public static long secondFromInterval(@SqlType(value="interval day to second") long milliseconds) {
        return milliseconds % 60000L / 1000L;
    }

    @Description(value="Minute of the hour of the given interval")
    @ScalarFunction(value="minute")
    @SqlType(value="bigint")
    public static long minuteFromInterval(@SqlType(value="interval day to second") long milliseconds) {
        return milliseconds % 3600000L / 60000L;
    }

    @Description(value="Hour of the day of the given interval")
    @ScalarFunction(value="hour")
    @SqlType(value="bigint")
    public static long hourFromInterval(@SqlType(value="interval day to second") long milliseconds) {
        return milliseconds % 86400000L / 3600000L;
    }

    @Description(value="Day of the week of the given date")
    @ScalarFunction(value="day_of_week", alias={"dow"})
    @SqlType(value="bigint")
    public static long dayOfWeekFromDate(@SqlType(value="date") long date) {
        return DAY_OF_WEEK.get(TimeUnit.DAYS.toMillis(date));
    }

    @Description(value="Day of the month of the given date")
    @ScalarFunction(value="day", alias={"day_of_month"})
    @SqlType(value="bigint")
    public static long dayFromDate(@SqlType(value="date") long date) {
        return DAY_OF_MONTH.get(TimeUnit.DAYS.toMillis(date));
    }

    @Description(value="Day of the month of the given interval")
    @ScalarFunction(value="day", alias={"day_of_month"})
    @SqlType(value="bigint")
    public static long dayFromInterval(@SqlType(value="interval day to second") long milliseconds) {
        return milliseconds / 86400000L;
    }

    @Description(value="Last day of the month of the given date")
    @ScalarFunction(value="last_day_of_month")
    @SqlType(value="date")
    public static long lastDayOfMonthFromDate(@SqlType(value="date") long date) {
        long millis = UTC_CHRONOLOGY.monthOfYear().roundCeiling(TimeUnit.DAYS.toMillis(date) + 1L) - 86400000L;
        return TimeUnit.MILLISECONDS.toDays(millis);
    }

    @Description(value="Day of the year of the given date")
    @ScalarFunction(value="day_of_year", alias={"doy"})
    @SqlType(value="bigint")
    public static long dayOfYearFromDate(@SqlType(value="date") long date) {
        return DAY_OF_YEAR.get(TimeUnit.DAYS.toMillis(date));
    }

    @Description(value="Week of the year of the given date")
    @ScalarFunction(value="week", alias={"week_of_year"})
    @SqlType(value="bigint")
    public static long weekFromDate(@SqlType(value="date") long date) {
        return WEEK_OF_YEAR.get(TimeUnit.DAYS.toMillis(date));
    }

    @Description(value="Year of the ISO week of the given date")
    @ScalarFunction(value="year_of_week", alias={"yow"})
    @SqlType(value="bigint")
    public static long yearOfWeekFromDate(@SqlType(value="date") long date) {
        return YEAR_OF_WEEK.get(TimeUnit.DAYS.toMillis(date));
    }

    @Description(value="Month of the year of the given date")
    @ScalarFunction(value="month")
    @SqlType(value="bigint")
    public static long monthFromDate(@SqlType(value="date") long date) {
        return MONTH_OF_YEAR.get(TimeUnit.DAYS.toMillis(date));
    }

    @Description(value="Month of the year of the given interval")
    @ScalarFunction(value="month")
    @SqlType(value="bigint")
    public static long monthFromInterval(@SqlType(value="interval year to month") long months) {
        return months % 12L;
    }

    @Description(value="Quarter of the year of the given date")
    @ScalarFunction(value="quarter")
    @SqlType(value="bigint")
    public static long quarterFromDate(@SqlType(value="date") long date) {
        return QUARTER.get(TimeUnit.DAYS.toMillis(date));
    }

    @Description(value="Year of the given date")
    @ScalarFunction(value="year")
    @SqlType(value="bigint")
    public static long yearFromDate(@SqlType(value="date") long date) {
        return YEAR.get(TimeUnit.DAYS.toMillis(date));
    }

    @Description(value="Year of the given interval")
    @ScalarFunction(value="year")
    @SqlType(value="bigint")
    public static long yearFromInterval(@SqlType(value="interval year to month") long months) {
        return months / 12L;
    }

    public static DateTimeFormatter createDateTimeFormatter(Slice format) {
        DateTimeFormatterBuilder builder = new DateTimeFormatterBuilder();
        String formatString = format.toStringUtf8();
        boolean escaped = false;
        for (int i = 0; i < formatString.length(); ++i) {
            char character = formatString.charAt(i);
            if (escaped) {
                switch (character) {
                    case 'a': {
                        builder.appendDayOfWeekShortText();
                        break;
                    }
                    case 'b': {
                        builder.appendMonthOfYearShortText();
                        break;
                    }
                    case 'c': {
                        builder.appendMonthOfYear(1);
                        break;
                    }
                    case 'd': {
                        builder.appendDayOfMonth(2);
                        break;
                    }
                    case 'e': {
                        builder.appendDayOfMonth(1);
                        break;
                    }
                    case 'f': {
                        builder.appendFractionOfSecond(6, 9);
                        break;
                    }
                    case 'H': {
                        builder.appendHourOfDay(2);
                        break;
                    }
                    case 'I': 
                    case 'h': {
                        builder.appendClockhourOfHalfday(2);
                        break;
                    }
                    case 'i': {
                        builder.appendMinuteOfHour(2);
                        break;
                    }
                    case 'j': {
                        builder.appendDayOfYear(3);
                        break;
                    }
                    case 'k': {
                        builder.appendHourOfDay(1);
                        break;
                    }
                    case 'l': {
                        builder.appendClockhourOfHalfday(1);
                        break;
                    }
                    case 'M': {
                        builder.appendMonthOfYearText();
                        break;
                    }
                    case 'm': {
                        builder.appendMonthOfYear(2);
                        break;
                    }
                    case 'p': {
                        builder.appendHalfdayOfDayText();
                        break;
                    }
                    case 'r': {
                        builder.appendClockhourOfHalfday(2).appendLiteral(':').appendMinuteOfHour(2).appendLiteral(':').appendSecondOfMinute(2).appendLiteral(' ').appendHalfdayOfDayText();
                        break;
                    }
                    case 'S': 
                    case 's': {
                        builder.appendSecondOfMinute(2);
                        break;
                    }
                    case 'T': {
                        builder.appendHourOfDay(2).appendLiteral(':').appendMinuteOfHour(2).appendLiteral(':').appendSecondOfMinute(2);
                        break;
                    }
                    case 'v': {
                        builder.appendWeekOfWeekyear(2);
                        break;
                    }
                    case 'x': {
                        builder.appendWeekyear(4, 4);
                        break;
                    }
                    case 'W': {
                        builder.appendDayOfWeekText();
                        break;
                    }
                    case 'Y': {
                        builder.appendYear(4, 4);
                        break;
                    }
                    case 'y': {
                        builder.appendTwoDigitYear(2020);
                        break;
                    }
                    case 'D': 
                    case 'U': 
                    case 'V': 
                    case 'X': 
                    case 'u': 
                    case 'w': {
                        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("%%%s not supported in date format string", Character.valueOf(character)));
                    }
                    case '%': {
                        builder.appendLiteral('%');
                        break;
                    }
                    default: {
                        builder.appendLiteral(character);
                    }
                }
                escaped = false;
                continue;
            }
            if (character == '%') {
                escaped = true;
                continue;
            }
            builder.appendLiteral(character);
        }
        try {
            return builder.toFormatter();
        }
        catch (UnsupportedOperationException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
    }

    @Description(value="Convert duration string to an interval")
    @ScalarFunction(value="parse_duration")
    @LiteralParameters(value={"x"})
    @SqlType(value="interval day to second")
    public static long parseDuration(@SqlType(value="varchar(x)") Slice duration) {
        try {
            return Duration.valueOf((String)duration.toStringUtf8()).toMillis();
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
    }

    @ScalarFunction(value="to_milliseconds")
    @SqlType(value="bigint")
    public static long toMilliseconds(@SqlType(value="interval day to second") long value) {
        return value;
    }

    @ScalarFunction(value="from_unixtime_nanos")
    public static final class FromUnixtimeNanosDecimal {
        private FromUnixtimeNanosDecimal() {
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="timestamp(9) with time zone")
        public static LongTimestampWithTimeZone fromLong(@LiteralParameter(value="s") long scale, ConnectorSession session, @SqlType(value="decimal(p, s)") Int128 unixTimeNanos) {
            Int128 decimal = Int128Math.rescale((Int128)unixTimeNanos, (int)(-((int)scale)));
            BigInteger unixTimeNanosInt = decimal.toBigInteger();
            long epochSeconds = unixTimeNanosInt.divide(BigInteger.valueOf(1000000000L)).longValue();
            long nanosOfSecond = unixTimeNanosInt.remainder(BigInteger.valueOf(1000000000L)).longValue();
            long picosOfSecond = nanosOfSecond * 1000L;
            if (picosOfSecond < 0L) {
                --epochSeconds;
                picosOfSecond += 1000000000000L;
            }
            try {
                return DateTimes.longTimestampWithTimeZone(epochSeconds, picosOfSecond, session.getTimeZoneKey().getZoneId());
            }
            catch (ArithmeticException e) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
            }
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="timestamp(9) with time zone")
        public static LongTimestampWithTimeZone fromShort(@LiteralParameter(value="s") long scale, ConnectorSession session, @SqlType(value="decimal(p, s)") long unixTimeNanos) {
            long roundedUnixTimeNanos = MathFunctions.Round.roundShort(scale, unixTimeNanos);
            return DateTimeFunctions.fromUnixtimeNanosLong(session, roundedUnixTimeNanos);
        }
    }
}

