/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.vertica;

import com.fasterxml.jackson.annotation.JsonAlias;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.xml.JacksonXmlModule;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import com.fasterxml.jackson.dataformat.xml.annotation.JacksonXmlElementWrapper;
import com.fasterxml.jackson.dataformat.xml.annotation.JacksonXmlProperty;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import io.airlift.log.Logger;
import io.trino.plugin.jdbc.JdbcColumnHandle;
import io.trino.plugin.jdbc.JdbcTableHandle;
import io.trino.plugin.jdbc.RemoteTableName;
import io.trino.spi.connector.ColumnHandle;
import io.trino.spi.statistics.ColumnStatistics;
import io.trino.spi.statistics.DoubleRange;
import io.trino.spi.statistics.Estimate;
import io.trino.spi.statistics.TableStatistics;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.Type;
import java.sql.Connection;
import java.util.AbstractMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.jdbi.v3.core.Handle;
import org.jdbi.v3.core.Jdbi;
import org.jdbi.v3.core.statement.Query;

public final class VerticaTableStatisticsReader {
    private static final Logger log = Logger.get(VerticaTableStatisticsReader.class);
    private static final ObjectMapper OBJECT_MAPPER = new XmlMapper(new JacksonXmlModule()).disable(DeserializationFeature.FAIL_ON_IGNORED_PROPERTIES, new DeserializationFeature[]{DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES});
    private static final Estimate UNKNOWN_NULL_FRACTION_REPLACEMENT = Estimate.of((double)0.1);

    private VerticaTableStatisticsReader() {
    }

    public static TableStatistics readTableStatistics(Connection connection, JdbcTableHandle table, Supplier<List<JdbcColumnHandle>> columnSupplier) {
        Preconditions.checkArgument((boolean)table.isNamedRelation(), (String)"Relation is not a table: %s", (Object)table);
        log.debug("Reading statistics for %s", new Object[]{table});
        try (Handle handle = Jdbi.open((Connection)connection);){
            StatisticsDao statisticsDao = new StatisticsDao(handle);
            Long rowCount = statisticsDao.getRowCount(table);
            log.debug("Estimated row count of table %s is %s", new Object[]{table, rowCount});
            if (rowCount == null) {
                TableStatistics tableStatistics = TableStatistics.empty();
                return tableStatistics;
            }
            TableStatistics.Builder tableStatistics = TableStatistics.builder();
            tableStatistics.setRowCount(Estimate.of((double)rowCount.longValue()));
            Schema schema = statisticsDao.getSchemaStatistics(table);
            if (schema == null || schema.tables().size() == 0) {
                TableStatistics tableStatistics2 = TableStatistics.empty();
                return tableStatistics2;
            }
            Map columnsStatistics = (Map)((Table)Iterables.getOnlyElement(schema.tables())).columns().stream().collect(ImmutableMap.toImmutableMap(Column::columnName, Column::stats));
            Map<String, Long> columnsDataSize = statisticsDao.getColumnDataSize(table);
            for (JdbcColumnHandle column : columnSupplier.get()) {
                ColumnStatistics columnStatistics;
                ColumnStatistics.Builder columnStatisticsBuilder = io.trino.spi.statistics.ColumnStatistics.builder();
                String columnName = column.getColumnName();
                if (columnsDataSize.containsKey(columnName)) {
                    columnStatisticsBuilder.setDataSize(Estimate.of((double)columnsDataSize.get(columnName).longValue()));
                }
                if ((columnStatistics = (ColumnStatistics)columnsStatistics.get(columnName)) != null) {
                    log.debug("Reading column statistics for %s, %s from index statistics: %s", new Object[]{table, columnName, columnStatistics});
                    Optional<Long> nullsCount = columnStatistics.histogram().category().stream().filter(category -> category.bound().isNull()).map(category -> category.count().value()).findFirst();
                    long distinctValuesCount = columnStatistics.distinct().value;
                    if (nullsCount.isPresent() || columnStatistics.minValue().isNull() || columnStatistics.maxValue().isNull()) {
                        distinctValuesCount = Math.max(columnStatistics.distinct().value - 1L, 0L);
                    }
                    columnStatisticsBuilder.setDistinctValuesCount(Estimate.of((double)distinctValuesCount));
                    if (VerticaTableStatisticsReader.isNumeric(column.getColumnType())) {
                        columnStatisticsBuilder.setRange(VerticaTableStatisticsReader.createNumericRange(columnStatistics.minValue().getValue(), columnStatistics.maxValue().getValue()));
                    }
                    columnStatisticsBuilder.setNullsFraction(VerticaTableStatisticsReader.getNullsFraction(rowCount, nullsCount, columnStatistics));
                }
                tableStatistics.setColumnStatistics((ColumnHandle)column, columnStatisticsBuilder.build());
            }
            tableStatistics.setRowCount(Estimate.of((double)rowCount.longValue()));
            TableStatistics tableStatistics3 = tableStatistics.build();
            return tableStatistics3;
        }
    }

    private static Estimate getNullsFraction(Long rowCount, Optional<Long> nullsCount, ColumnStatistics columnStatistics) {
        if (nullsCount.isPresent()) {
            return Estimate.of((double)Math.min(1.0, (double)nullsCount.get().longValue() / (double)rowCount.longValue()));
        }
        if (columnStatistics.distinct().value == 2L && (columnStatistics.minValue().isNan() || columnStatistics.maxValue().isNan())) {
            return UNKNOWN_NULL_FRACTION_REPLACEMENT;
        }
        if (columnStatistics.minValue().isNull() || columnStatistics.maxValue().isNull()) {
            long nonNullCount = columnStatistics.histogram().category().stream().mapToLong(category -> category.count().value()).sum();
            return Estimate.of((double)(((double)rowCount.longValue() - (double)nonNullCount) / (double)rowCount.longValue()));
        }
        return Estimate.zero();
    }

    private static boolean isNumeric(Type type) {
        return type == BigintType.BIGINT || type == DoubleType.DOUBLE || type instanceof DecimalType;
    }

    private static Optional<DoubleRange> createNumericRange(Optional<String> minValue, Optional<String> maxValue) {
        if (minValue.isEmpty() && maxValue.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(new DoubleRange(minValue.filter(value -> !value.equals("-nan")).flatMap(VerticaTableStatisticsReader::tryParseDouble).orElse(Double.NEGATIVE_INFINITY).doubleValue(), maxValue.filter(value -> !value.equals("nan")).flatMap(VerticaTableStatisticsReader::tryParseDouble).orElse(Double.POSITIVE_INFINITY).doubleValue()));
    }

    private static Optional<Double> tryParseDouble(String value) {
        try {
            return Optional.of(Double.valueOf(value));
        }
        catch (NumberFormatException e) {
            return Optional.empty();
        }
    }

    private static class StatisticsDao {
        private final Handle handle;

        public StatisticsDao(Handle handle) {
            this.handle = Objects.requireNonNull(handle, "handle is null");
        }

        Long getRowCount(JdbcTableHandle table) {
            RemoteTableName remoteTableName = table.getRequiredNamedRelation().getRemoteTableName();
            return ((Query)((Query)this.handle.createQuery("SELECT row_count FROM v_monitor.projection_storage WHERE anchor_table_schema = :schema AND anchor_table_name = :table_name").bind("schema", (String)remoteTableName.getCatalogName().orElse(null))).bind("table_name", remoteTableName.getTableName())).mapTo(Long.class).findOne().orElse(null);
        }

        Schema getSchemaStatistics(JdbcTableHandle table) {
            RemoteTableName remoteTableName = table.getRequiredNamedRelation().getRemoteTableName();
            return (Schema)((Query)this.handle.createQuery("SELECT EXPORT_STATISTICS('', :schema_table_name)").bind("schema_table_name", String.format("%s.%s", remoteTableName.getSchemaName().orElse(null), remoteTableName.getTableName()))).map((rs, ctx) -> {
                try {
                    String exportStatistics = rs.getString("EXPORT_STATISTICS");
                    return (Schema)OBJECT_MAPPER.readValue(exportStatistics, Schema.class);
                }
                catch (JsonProcessingException e) {
                    log.warn((Throwable)e, "Failed to read statistics");
                    return null;
                }
            }).one();
        }

        Map<String, Long> getColumnDataSize(JdbcTableHandle table) {
            RemoteTableName remoteTableName = table.getRequiredNamedRelation().getRemoteTableName();
            return ((Query)((Query)this.handle.createQuery("SELECT column_name, SUM(used_bytes) AS size FROM v_monitor.column_storage WHERE anchor_table_schema = :schema AND anchor_table_name = :table_name GROUP BY column_name").bind("schema", (String)remoteTableName.getCatalogName().orElse(null))).bind("table_name", remoteTableName.getTableName())).map((rs, ctx) -> new AbstractMap.SimpleEntry<String, Long>(rs.getString("column_name"), rs.getLong("size"))).stream().collect(Collectors.toMap(AbstractMap.SimpleEntry::getKey, AbstractMap.SimpleEntry::getValue));
        }
    }

    public record Schema(@JacksonXmlProperty(localName="tables") List<Table> tables) {
    }

    public record Table(@JacksonXmlProperty(localName="schema") String schemaName, @JacksonXmlProperty(localName="name") String tableName, @JacksonXmlProperty(localName="columns") List<Column> columns) {
    }

    public record ColumnStatistics(@JacksonXmlProperty(localName="distinct") NumericValue distinct, @JacksonXmlProperty(localName="minValue") NullableValue minValue, @JacksonXmlProperty(localName="maxValue") NullableValue maxValue, @JacksonXmlProperty(localName="histogram") Histogram histogram) {
    }

    public record Histogram(@JacksonXmlElementWrapper(useWrapping=false) @JacksonXmlProperty(localName="category") List<Category> category) {

        public record Category(@JacksonXmlProperty(localName="bound") NullableValue bound, @JacksonXmlProperty(localName="count") NumericValue count) {
        }
    }

    public record NumericValue(@JacksonXmlProperty(localName="value", isAttribute=true) long value) {
    }

    public static class NullableValue {
        private final String value;
        private final String nullValue;

        @JsonCreator
        public NullableValue(String value) {
            this(null, value);
        }

        @JsonCreator
        public NullableValue(@JacksonXmlProperty(localName="nullValue", isAttribute=true) String nullValue, @JacksonXmlProperty(localName=" ") String value) {
            this.value = value;
            this.nullValue = nullValue;
        }

        public Optional<String> getValue() {
            return Optional.ofNullable(this.value);
        }

        public boolean isNull() {
            if (this.nullValue == null) {
                return false;
            }
            return this.nullValue.equals("true");
        }

        public boolean isNan() {
            if (this.value == null) {
                return false;
            }
            return this.value.equals("-nan") || this.value.equals("nan");
        }
    }

    public record Column(@JacksonXmlProperty(localName="name") String columnName, @JacksonXmlProperty(localName="dataType") String dataType, @JsonProperty(value="intStats") @JsonAlias(value={"stringStats", "floatStats", "numericStats"}) ColumnStatistics stats) {
    }
}

