/*
 * Decompiled with CFR 0.152.
 */
package io.unitycatalog.server.persist;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.unitycatalog.server.exception.BaseException;
import io.unitycatalog.server.exception.ErrorCode;
import io.unitycatalog.server.model.AwsIamRoleRequest;
import io.unitycatalog.server.model.AwsIamRoleResponse;
import io.unitycatalog.server.model.CreateCredentialRequest;
import io.unitycatalog.server.model.CredentialInfo;
import io.unitycatalog.server.model.ListCredentialsResponse;
import io.unitycatalog.server.model.UpdateCredentialRequest;
import io.unitycatalog.server.persist.Repositories;
import io.unitycatalog.server.persist.dao.CredentialDAO;
import io.unitycatalog.server.persist.utils.PagedListingHelper;
import io.unitycatalog.server.persist.utils.TransactionManager;
import io.unitycatalog.server.utils.IdentityUtils;
import io.unitycatalog.server.utils.ValidationUtils;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.Query;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CredentialRepository {
    private static final Logger LOGGER = LoggerFactory.getLogger(CredentialRepository.class);
    private final Repositories repositories;
    private final SessionFactory sessionFactory;
    private static final PagedListingHelper<CredentialDAO> LISTING_HELPER = new PagedListingHelper<CredentialDAO>(CredentialDAO.class);
    public static ObjectMapper objectMapper = new ObjectMapper();

    public CredentialRepository(Repositories repositories, SessionFactory sessionFactory) {
        this.repositories = repositories;
        this.sessionFactory = sessionFactory;
    }

    public CredentialInfo addCredential(CreateCredentialRequest createCredentialRequest) {
        ValidationUtils.validateSqlObjectName(createCredentialRequest.getName());
        String callerId = IdentityUtils.findPrincipalEmailAddress();
        UUID storageCredentialId = UUID.randomUUID();
        CredentialInfo storageCredentialInfo = new CredentialInfo().id(storageCredentialId.toString()).name(createCredentialRequest.getName()).comment(createCredentialRequest.getComment()).purpose(createCredentialRequest.getPurpose()).owner(callerId).createdAt(Instant.now().toEpochMilli()).createdBy(callerId);
        if (createCredentialRequest.getAwsIamRole() == null) {
            throw new BaseException(ErrorCode.INVALID_ARGUMENT, "Storage credential must have one of aws_iam_role, azure_service_principal, azure_managed_identity or gcp_service_account");
        }
        storageCredentialInfo.setAwsIamRole(CredentialRepository.fromAwsIamRoleRequest(createCredentialRequest.getAwsIamRole()));
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            if (this.getCredentialDAO(session, createCredentialRequest.getName()) != null) {
                throw new BaseException(ErrorCode.ALREADY_EXISTS, "Storage credential already exists: " + createCredentialRequest.getName());
            }
            session.persist((Object)CredentialDAO.from(storageCredentialInfo));
            LOGGER.info("Added storage credential: {}", (Object)storageCredentialInfo.getName());
            return storageCredentialInfo;
        }, "Failed to add storage credential", false);
    }

    public CredentialInfo getCredential(String name) {
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            CredentialDAO dao = this.getCredentialDAO(session, name);
            if (dao == null) {
                throw new BaseException(ErrorCode.NOT_FOUND, "Storage credential not found: " + name);
            }
            LOGGER.info("Retrieved storage credential: {}", (Object)name);
            return dao.toCredentialInfo();
        }, "Failed to get storage credential", true);
    }

    protected CredentialDAO getCredentialDAO(Session session, String name) {
        Query query = session.createQuery("FROM CredentialDAO WHERE name = :value", CredentialDAO.class);
        query.setParameter("value", (Object)name);
        query.setMaxResults(1);
        return (CredentialDAO)query.uniqueResult();
    }

    public ListCredentialsResponse listCredentials(Optional<Integer> maxResults, Optional<String> pageToken) {
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            List<CredentialDAO> daoList = LISTING_HELPER.listEntity(session, maxResults, pageToken, null);
            String nextPageToken = LISTING_HELPER.getNextPageToken(daoList, maxResults);
            ArrayList<CredentialInfo> results = new ArrayList<CredentialInfo>();
            for (CredentialDAO dao : daoList) {
                try {
                    results.add(dao.toCredentialInfo());
                }
                catch (Exception e) {
                    LOGGER.error("Failed to process credential: {}", (Object)dao.getName(), (Object)e);
                }
            }
            return new ListCredentialsResponse().credentials(results).nextPageToken(nextPageToken);
        }, "Failed to list storage credentials", true);
    }

    public CredentialInfo updateCredential(String name, UpdateCredentialRequest updateCredential) {
        String callerId = IdentityUtils.findPrincipalEmailAddress();
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            CredentialDAO existingCredential = this.getCredentialDAO(session, name);
            if (existingCredential == null) {
                throw new BaseException(ErrorCode.NOT_FOUND, "Storage credential not found: " + name);
            }
            if (updateCredential.getNewName() != null) {
                ValidationUtils.validateSqlObjectName(updateCredential.getNewName());
                if (this.getCredentialDAO(session, updateCredential.getNewName()) != null) {
                    throw new BaseException(ErrorCode.ALREADY_EXISTS, "Storage credential already exists: " + updateCredential.getNewName());
                }
                existingCredential.setName(updateCredential.getNewName());
            }
            CredentialRepository.updateCredentialFields(existingCredential, updateCredential);
            if (updateCredential.getComment() != null) {
                existingCredential.setComment(updateCredential.getComment());
            }
            existingCredential.setUpdatedAt(new Date());
            existingCredential.setUpdatedBy(callerId);
            session.merge((Object)existingCredential);
            LOGGER.info("Updated storage credential: {}", (Object)name);
            return existingCredential.toCredentialInfo();
        }, "Failed to update storage credential", false);
    }

    private static void updateCredentialFields(CredentialDAO existingCredential, UpdateCredentialRequest updateCredentialRequest) {
        try {
            if (updateCredentialRequest.getAwsIamRole() != null) {
                existingCredential.setCredentialType(CredentialDAO.CredentialType.AWS_IAM_ROLE);
                String jsonCredential = objectMapper.writeValueAsString((Object)CredentialRepository.fromAwsIamRoleRequest(updateCredentialRequest.getAwsIamRole()));
                existingCredential.setCredential(jsonCredential);
            }
        }
        catch (JsonProcessingException e) {
            throw new BaseException(ErrorCode.INVALID_ARGUMENT, "Failed to serialize credential: " + e.getMessage());
        }
    }

    public CredentialInfo deleteCredential(String name) {
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            CredentialDAO existingCredential = this.getCredentialDAO(session, name);
            if (existingCredential == null) {
                throw new BaseException(ErrorCode.NOT_FOUND, "Credential not found: " + name);
            }
            CredentialInfo credentialInfo = existingCredential.toCredentialInfo();
            session.remove((Object)existingCredential);
            LOGGER.info("Deleted credential: {}", (Object)name);
            return credentialInfo;
        }, "Failed to delete credential", false);
    }

    private static AwsIamRoleResponse fromAwsIamRoleRequest(AwsIamRoleRequest awsIamRoleRequest) {
        return new AwsIamRoleResponse().roleArn(awsIamRoleRequest.getRoleArn());
    }
}

