/*
 * Decompiled with CFR 0.152.
 */
package io.unitycatalog.server.persist;

import io.unitycatalog.server.exception.BaseException;
import io.unitycatalog.server.exception.ErrorCode;
import io.unitycatalog.server.model.ColumnInfo;
import io.unitycatalog.server.model.CreateTable;
import io.unitycatalog.server.model.ListTablesResponse;
import io.unitycatalog.server.model.TableInfo;
import io.unitycatalog.server.model.TableType;
import io.unitycatalog.server.persist.PropertyRepository;
import io.unitycatalog.server.persist.Repositories;
import io.unitycatalog.server.persist.dao.CatalogInfoDAO;
import io.unitycatalog.server.persist.dao.PropertyDAO;
import io.unitycatalog.server.persist.dao.SchemaInfoDAO;
import io.unitycatalog.server.persist.dao.TableInfoDAO;
import io.unitycatalog.server.persist.utils.FileOperations;
import io.unitycatalog.server.persist.utils.PagedListingHelper;
import io.unitycatalog.server.persist.utils.RepositoryUtils;
import io.unitycatalog.server.persist.utils.TransactionManager;
import io.unitycatalog.server.utils.IdentityUtils;
import io.unitycatalog.server.utils.ValidationUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.Query;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TableRepository {
    private static final Logger LOGGER = LoggerFactory.getLogger(TableRepository.class);
    private final SessionFactory sessionFactory;
    private final Repositories repositories;
    private final FileOperations fileOperations;
    private static final PagedListingHelper<TableInfoDAO> LISTING_HELPER = new PagedListingHelper<TableInfoDAO>(TableInfoDAO.class);

    public TableRepository(Repositories repositories, SessionFactory sessionFactory) {
        this.repositories = repositories;
        this.sessionFactory = sessionFactory;
        this.fileOperations = repositories.getFileOperations();
    }

    public TableInfo getTableById(String tableId) {
        LOGGER.debug("Getting table by id: {}", (Object)tableId);
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            TableInfoDAO tableInfoDAO = (TableInfoDAO)session.get(TableInfoDAO.class, (Object)UUID.fromString(tableId));
            if (tableInfoDAO == null) {
                throw new BaseException(ErrorCode.NOT_FOUND, "Table not found: " + tableId);
            }
            SchemaInfoDAO schemaInfoDAO = (SchemaInfoDAO)session.get(SchemaInfoDAO.class, (Object)tableInfoDAO.getSchemaId());
            if (schemaInfoDAO == null) {
                throw new BaseException(ErrorCode.NOT_FOUND, "Schema not found: " + String.valueOf(tableInfoDAO.getSchemaId()));
            }
            CatalogInfoDAO catalogInfoDAO = (CatalogInfoDAO)session.get(CatalogInfoDAO.class, (Object)schemaInfoDAO.getCatalogId());
            if (catalogInfoDAO == null) {
                throw new BaseException(ErrorCode.NOT_FOUND, "Catalog not found: " + String.valueOf(schemaInfoDAO.getCatalogId()));
            }
            TableInfo tableInfo = tableInfoDAO.toTableInfo(true);
            tableInfo.setSchemaName(schemaInfoDAO.getName());
            tableInfo.setCatalogName(catalogInfoDAO.getName());
            return tableInfo;
        }, "Failed to get table by ID", true);
    }

    public TableInfo getTable(String fullName) {
        LOGGER.debug("Getting table: {}", (Object)fullName);
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            String[] parts = fullName.split("\\.");
            if (parts.length != 3) {
                throw new BaseException(ErrorCode.INVALID_ARGUMENT, "Invalid table name: " + fullName);
            }
            String catalogName = parts[0];
            String schemaName = parts[1];
            String tableName = parts[2];
            TableInfoDAO tableInfoDAO = this.findTable(session, catalogName, schemaName, tableName);
            if (tableInfoDAO == null) {
                throw new BaseException(ErrorCode.NOT_FOUND, "Table not found: " + fullName);
            }
            TableInfo tableInfo = tableInfoDAO.toTableInfo(true);
            tableInfo.setCatalogName(catalogName);
            tableInfo.setSchemaName(schemaName);
            RepositoryUtils.attachProperties(tableInfo, tableInfo.getTableId(), "table", session);
            return tableInfo;
        }, "Failed to get table", true);
    }

    public String getTableUniformMetadataLocation(Session session, String catalogName, String schemaName, String tableName) {
        TableInfoDAO dao = this.findTable(session, catalogName, schemaName, tableName);
        return dao.getUniformIcebergMetadataLocation();
    }

    private TableInfoDAO findTable(Session session, String catalogName, String schemaName, String tableName) {
        UUID schemaId = this.getSchemaId(session, catalogName, schemaName);
        return this.findBySchemaIdAndName(session, schemaId, tableName);
    }

    public TableInfo createTable(CreateTable createTable) {
        ValidationUtils.validateSqlObjectName(createTable.getName());
        String callerId = IdentityUtils.findPrincipalEmailAddress();
        List<ColumnInfo> columnInfos = createTable.getColumns().stream().map(c -> c.typeText(c.getTypeText().toLowerCase(Locale.ROOT))).collect(Collectors.toList());
        Long createTime = System.currentTimeMillis();
        TableInfo tableInfo = new TableInfo().tableId(UUID.randomUUID().toString()).name(createTable.getName()).catalogName(createTable.getCatalogName()).schemaName(createTable.getSchemaName()).tableType(createTable.getTableType()).dataSourceFormat(createTable.getDataSourceFormat()).columns(columnInfos).storageLocation(FileOperations.convertRelativePathToURI(createTable.getStorageLocation())).comment(createTable.getComment()).properties(createTable.getProperties()).owner(callerId).createdAt(createTime).createdBy(callerId).updatedAt(createTime).updatedBy(callerId);
        String fullName = this.getTableFullName(tableInfo);
        LOGGER.debug("Creating table: {}", (Object)fullName);
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            String schemaName;
            String catalogName = tableInfo.getCatalogName();
            UUID schemaId = this.getSchemaId(session, catalogName, schemaName = tableInfo.getSchemaName());
            TableInfoDAO existingTable = this.findBySchemaIdAndName(session, schemaId, tableInfo.getName());
            if (existingTable != null) {
                throw new BaseException(ErrorCode.ALREADY_EXISTS, "Table already exists: " + fullName);
            }
            if (TableType.MANAGED.equals((Object)tableInfo.getTableType())) {
                throw new BaseException(ErrorCode.INVALID_ARGUMENT, "MANAGED table creation is not supported yet.");
            }
            if (tableInfo.getStorageLocation() == null) {
                throw new BaseException(ErrorCode.INVALID_ARGUMENT, "Storage location is required for external table");
            }
            TableInfoDAO tableInfoDAO = TableInfoDAO.from(tableInfo);
            tableInfoDAO.setSchemaId(schemaId);
            tableInfoDAO.getColumns().forEach(c -> {
                c.setId(UUID.randomUUID());
                c.setTable(tableInfoDAO);
            });
            PropertyDAO.from(tableInfo.getProperties(), tableInfoDAO.getId(), "table").forEach(arg_0 -> ((Session)session).persist(arg_0));
            session.persist((Object)tableInfoDAO);
            return tableInfo;
        }, "Error creating table: " + fullName, false);
    }

    public TableInfoDAO findBySchemaIdAndName(Session session, UUID schemaId, String name) {
        String hql = "FROM TableInfoDAO t WHERE t.schemaId = :schemaId AND t.name = :name";
        Query query = session.createQuery(hql, TableInfoDAO.class);
        query.setParameter("schemaId", (Object)schemaId);
        query.setParameter("name", (Object)name);
        LOGGER.debug("Finding table by schemaId: {} and name: {}", (Object)schemaId, (Object)name);
        return (TableInfoDAO)query.uniqueResult();
    }

    private String getTableFullName(TableInfo tableInfo) {
        return tableInfo.getCatalogName() + "." + tableInfo.getSchemaName() + "." + tableInfo.getName();
    }

    public UUID getSchemaId(Session session, String catalogName, String schemaName) {
        SchemaInfoDAO schemaInfo = this.repositories.getSchemaRepository().getSchemaDAO(session, catalogName, schemaName);
        if (schemaInfo == null) {
            throw new BaseException(ErrorCode.NOT_FOUND, "Schema not found: " + schemaName);
        }
        return schemaInfo.getId();
    }

    public ListTablesResponse listTables(String catalogName, String schemaName, Optional<Integer> maxResults, Optional<String> pageToken, Boolean omitProperties, Boolean omitColumns) {
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            UUID schemaId = this.getSchemaId(session, catalogName, schemaName);
            return this.listTables(session, schemaId, catalogName, schemaName, maxResults, pageToken, omitProperties, omitColumns);
        }, "Failed to list tables", true);
    }

    public ListTablesResponse listTables(Session session, UUID schemaId, String catalogName, String schemaName, Optional<Integer> maxResults, Optional<String> pageToken, Boolean omitProperties, Boolean omitColumns) {
        List<TableInfoDAO> tableInfoDAOList = LISTING_HELPER.listEntity(session, maxResults, pageToken, schemaId);
        String nextPageToken = LISTING_HELPER.getNextPageToken(tableInfoDAOList, maxResults);
        ArrayList<TableInfo> result = new ArrayList<TableInfo>();
        for (TableInfoDAO tableInfoDAO : tableInfoDAOList) {
            TableInfo tableInfo = tableInfoDAO.toTableInfo(omitColumns == false);
            if (!omitProperties.booleanValue()) {
                RepositoryUtils.attachProperties(tableInfo, tableInfo.getTableId(), "table", session);
            }
            tableInfo.setCatalogName(catalogName);
            tableInfo.setSchemaName(schemaName);
            result.add(tableInfo);
        }
        return new ListTablesResponse().tables(result).nextPageToken(nextPageToken);
    }

    public void deleteTable(String fullName) {
        TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            String[] parts = fullName.split("\\.");
            if (parts.length != 3) {
                throw new BaseException(ErrorCode.INVALID_ARGUMENT, "Invalid table name: " + fullName);
            }
            String catalogName = parts[0];
            String schemaName = parts[1];
            String tableName = parts[2];
            UUID schemaId = this.getSchemaId(session, catalogName, schemaName);
            this.deleteTable(session, schemaId, tableName);
            return null;
        }, "Failed to delete table", false);
    }

    public void deleteTable(Session session, UUID schemaId, String tableName) {
        TableInfoDAO tableInfoDAO = this.findBySchemaIdAndName(session, schemaId, tableName);
        if (tableInfoDAO == null) {
            throw new BaseException(ErrorCode.NOT_FOUND, "Table not found: " + tableName);
        }
        if (TableType.MANAGED.getValue().equals(tableInfoDAO.getType())) {
            try {
                this.fileOperations.deleteDirectory(tableInfoDAO.getUrl());
            }
            catch (Throwable e) {
                LOGGER.error("Error deleting table directory: {}", (Object)tableInfoDAO.getUrl(), (Object)e);
            }
        }
        PropertyRepository.findProperties(session, tableInfoDAO.getId(), "table").forEach(arg_0 -> ((Session)session).remove(arg_0));
        session.remove((Object)tableInfoDAO);
    }
}

