/*
 * Decompiled with CFR 0.152.
 */
package io.unitycatalog.server.persist;

import io.unitycatalog.server.exception.BaseException;
import io.unitycatalog.server.exception.ErrorCode;
import io.unitycatalog.server.model.CreateVolumeRequestContent;
import io.unitycatalog.server.model.ListVolumesResponseContent;
import io.unitycatalog.server.model.UpdateVolumeRequestContent;
import io.unitycatalog.server.model.VolumeInfo;
import io.unitycatalog.server.model.VolumeType;
import io.unitycatalog.server.persist.Repositories;
import io.unitycatalog.server.persist.dao.CatalogInfoDAO;
import io.unitycatalog.server.persist.dao.SchemaInfoDAO;
import io.unitycatalog.server.persist.dao.VolumeInfoDAO;
import io.unitycatalog.server.persist.utils.FileOperations;
import io.unitycatalog.server.persist.utils.PagedListingHelper;
import io.unitycatalog.server.persist.utils.TransactionManager;
import io.unitycatalog.server.utils.IdentityUtils;
import io.unitycatalog.server.utils.ValidationUtils;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.Query;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VolumeRepository {
    private static final Logger LOGGER = LoggerFactory.getLogger(VolumeRepository.class);
    private final Repositories repositories;
    private final SessionFactory sessionFactory;
    private final FileOperations fileOperations;
    private static final PagedListingHelper<VolumeInfoDAO> LISTING_HELPER = new PagedListingHelper<VolumeInfoDAO>(VolumeInfoDAO.class);

    public VolumeRepository(Repositories repositories, SessionFactory sessionFactory) {
        this.repositories = repositories;
        this.sessionFactory = sessionFactory;
        this.fileOperations = repositories.getFileOperations();
    }

    public VolumeInfo createVolume(CreateVolumeRequestContent createVolumeRequest) {
        ValidationUtils.validateSqlObjectName(createVolumeRequest.getName());
        String volumeFullName = createVolumeRequest.getCatalogName() + "." + createVolumeRequest.getSchemaName() + "." + createVolumeRequest.getName();
        String callerId = IdentityUtils.findPrincipalEmailAddress();
        Long createTime = System.currentTimeMillis();
        VolumeInfo volumeInfo = new VolumeInfo();
        volumeInfo.setVolumeId(UUID.randomUUID().toString());
        volumeInfo.setCatalogName(createVolumeRequest.getCatalogName());
        volumeInfo.setSchemaName(createVolumeRequest.getSchemaName());
        volumeInfo.setName(createVolumeRequest.getName());
        volumeInfo.setComment(createVolumeRequest.getComment());
        volumeInfo.setFullName(volumeFullName);
        volumeInfo.setOwner(callerId);
        volumeInfo.setCreatedAt(createTime);
        volumeInfo.setCreatedBy(callerId);
        volumeInfo.setUpdatedAt(createTime);
        volumeInfo.setUpdatedBy(callerId);
        volumeInfo.setVolumeType(createVolumeRequest.getVolumeType());
        if (VolumeType.MANAGED.equals((Object)createVolumeRequest.getVolumeType())) {
            throw new BaseException(ErrorCode.INVALID_ARGUMENT, "Managed volume creation is not supported");
        }
        if (createVolumeRequest.getStorageLocation() == null) {
            throw new BaseException(ErrorCode.INVALID_ARGUMENT, "Storage location is required for external volume");
        }
        volumeInfo.setStorageLocation(createVolumeRequest.getStorageLocation());
        VolumeInfoDAO volumeInfoDAO = VolumeInfoDAO.from(volumeInfo);
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            SchemaInfoDAO schemaInfoDAO = this.repositories.getSchemaRepository().getSchemaDAO(session, createVolumeRequest.getCatalogName(), createVolumeRequest.getSchemaName());
            if (schemaInfoDAO == null) {
                throw new BaseException(ErrorCode.NOT_FOUND, "Schema not found: " + createVolumeRequest.getCatalogName() + "." + createVolumeRequest.getSchemaName());
            }
            if (this.getVolumeDAO(session, createVolumeRequest.getCatalogName(), createVolumeRequest.getSchemaName(), createVolumeRequest.getName()) != null) {
                throw new BaseException(ErrorCode.ALREADY_EXISTS, "Volume already exists: " + volumeFullName);
            }
            volumeInfoDAO.setSchemaId(schemaInfoDAO.getId());
            session.persist((Object)volumeInfoDAO);
            LOGGER.info("Added volume: {}", (Object)volumeInfo.getName());
            return this.convertFromDAO(volumeInfoDAO, createVolumeRequest.getCatalogName(), createVolumeRequest.getSchemaName());
        }, "Failed to create volume", false);
    }

    public VolumeInfo getVolume(String fullName) {
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            String[] namespace = fullName.split("\\.");
            if (namespace.length != 3) {
                throw new BaseException(ErrorCode.INVALID_ARGUMENT, "Invalid volume name: " + fullName);
            }
            String catalogName = namespace[0];
            String schemaName = namespace[1];
            String volumeName = namespace[2];
            return this.convertFromDAO(this.getVolumeDAO(session, catalogName, schemaName, volumeName), catalogName, schemaName);
        }, "Failed to get volume", true);
    }

    public VolumeInfoDAO getVolumeDAO(Session session, String catalogName, String schemaName, String volumeName) {
        SchemaInfoDAO schemaInfo = this.repositories.getSchemaRepository().getSchemaDAO(session, catalogName, schemaName);
        if (schemaInfo == null) {
            throw new BaseException(ErrorCode.NOT_FOUND, "Schema not found: " + catalogName + "." + schemaName);
        }
        return this.getVolumeDAO(session, schemaInfo.getId(), volumeName);
    }

    public VolumeInfoDAO getVolumeDAO(Session session, UUID schemaId, String volumeName) {
        Query query = session.createQuery("FROM VolumeInfoDAO WHERE name = :name and schemaId = :schemaId", VolumeInfoDAO.class);
        query.setParameter("name", (Object)volumeName);
        query.setParameter("schemaId", (Object)schemaId);
        query.setMaxResults(1);
        return (VolumeInfoDAO)query.uniqueResult();
    }

    public VolumeInfo getVolumeById(String volumeId) {
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            VolumeInfoDAO volumeInfoDAO = (VolumeInfoDAO)session.get(VolumeInfoDAO.class, (Object)UUID.fromString(volumeId));
            if (volumeInfoDAO == null) {
                throw new BaseException(ErrorCode.NOT_FOUND, "Volume not found: " + volumeId);
            }
            SchemaInfoDAO schemaInfoDAO = (SchemaInfoDAO)session.get(SchemaInfoDAO.class, (Object)volumeInfoDAO.getSchemaId());
            if (schemaInfoDAO == null) {
                throw new BaseException(ErrorCode.NOT_FOUND, "Schema not found: " + String.valueOf(volumeInfoDAO.getSchemaId()));
            }
            CatalogInfoDAO catalogInfoDAO = (CatalogInfoDAO)session.get(CatalogInfoDAO.class, (Object)schemaInfoDAO.getCatalogId());
            if (catalogInfoDAO == null) {
                throw new BaseException(ErrorCode.NOT_FOUND, "Catalog not found: " + String.valueOf(schemaInfoDAO.getCatalogId()));
            }
            VolumeInfo volumeInfo = volumeInfoDAO.toVolumeInfo();
            volumeInfo.setSchemaName(schemaInfoDAO.getName());
            volumeInfo.setCatalogName(catalogInfoDAO.getName());
            return volumeInfo;
        }, "Failed to get volume by ID", true);
    }

    private void addNamespaceData(VolumeInfo volumeInfo, String catalogName, String schemaName) {
        volumeInfo.setCatalogName(catalogName);
        volumeInfo.setSchemaName(schemaName);
        volumeInfo.setFullName(catalogName + "." + schemaName + "." + volumeInfo.getName());
    }

    public UUID getSchemaId(Session session, String catalogName, String schemaName) {
        SchemaInfoDAO schemaInfo = this.repositories.getSchemaRepository().getSchemaDAO(session, catalogName, schemaName);
        if (schemaInfo == null) {
            throw new BaseException(ErrorCode.NOT_FOUND, "Schema not found: " + schemaName);
        }
        return schemaInfo.getId();
    }

    public ListVolumesResponseContent listVolumes(String catalogName, String schemaName, Optional<Integer> maxResults, Optional<String> pageToken, Optional<Boolean> includeBrowse) {
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            UUID schemaId = this.getSchemaId(session, catalogName, schemaName);
            return this.listVolumes(session, schemaId, catalogName, schemaName, maxResults, pageToken);
        }, "Failed to list volumes", true);
    }

    public ListVolumesResponseContent listVolumes(Session session, UUID schemaId, String catalogName, String schemaName, Optional<Integer> maxResults, Optional<String> pageToken) {
        List<VolumeInfoDAO> volumeInfoDAOList = LISTING_HELPER.listEntity(session, maxResults, pageToken, schemaId);
        String nextPageToken = LISTING_HELPER.getNextPageToken(volumeInfoDAOList, maxResults);
        ArrayList<VolumeInfo> result = new ArrayList<VolumeInfo>();
        for (VolumeInfoDAO volumeInfoDAO : volumeInfoDAOList) {
            VolumeInfo volumeInfo = volumeInfoDAO.toVolumeInfo();
            this.addNamespaceData(volumeInfo, catalogName, schemaName);
            result.add(volumeInfo);
        }
        return new ListVolumesResponseContent().volumes(result).nextPageToken(nextPageToken);
    }

    private VolumeInfo convertFromDAO(VolumeInfoDAO volumeInfoDAO, String catalogName, String schemaName) {
        VolumeInfo volumeInfo = volumeInfoDAO.toVolumeInfo();
        volumeInfo.setCatalogName(catalogName);
        volumeInfo.setSchemaName(schemaName);
        volumeInfo.setFullName(catalogName + "." + schemaName + "." + volumeInfo.getName());
        return volumeInfo;
    }

    public VolumeInfo updateVolume(String name, UpdateVolumeRequestContent updateVolumeRequest) {
        if (updateVolumeRequest.getNewName() != null) {
            ValidationUtils.validateSqlObjectName(updateVolumeRequest.getNewName());
        }
        String callerId = IdentityUtils.findPrincipalEmailAddress();
        String[] namespace = name.split("\\.");
        String catalog = namespace[0];
        String schema = namespace[1];
        String volume = namespace[2];
        return TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            VolumeInfoDAO existingVolume;
            VolumeInfoDAO volumeInfo = this.getVolumeDAO(session, catalog, schema, volume);
            if (volumeInfo == null) {
                throw new BaseException(ErrorCode.NOT_FOUND, "Volume not found: " + name);
            }
            if (updateVolumeRequest.getNewName() != null && (existingVolume = this.getVolumeDAO(session, catalog, schema, updateVolumeRequest.getNewName())) != null) {
                throw new BaseException(ErrorCode.ALREADY_EXISTS, "Volume already exists: " + updateVolumeRequest.getNewName());
            }
            if (updateVolumeRequest.getNewName() == null && updateVolumeRequest.getComment() == null) {
                return this.convertFromDAO(volumeInfo, catalog, schema);
            }
            if (updateVolumeRequest.getNewName() != null) {
                volumeInfo.setName(updateVolumeRequest.getNewName());
            }
            if (updateVolumeRequest.getComment() != null) {
                volumeInfo.setComment(updateVolumeRequest.getComment());
            }
            volumeInfo.setUpdatedAt(new Date());
            volumeInfo.setUpdatedBy(callerId);
            session.merge((Object)volumeInfo);
            LOGGER.info("Updated volume: {}", (Object)volumeInfo.getName());
            return this.convertFromDAO(volumeInfo, catalog, schema);
        }, "Failed to update volume", false);
    }

    public void deleteVolume(String name) {
        TransactionManager.executeWithTransaction(this.sessionFactory, session -> {
            String[] namespace = name.split("\\.");
            if (namespace.length != 3) {
                throw new BaseException(ErrorCode.INVALID_ARGUMENT, "Invalid volume name: " + name);
            }
            String catalog = namespace[0];
            String schema = namespace[1];
            String volume = namespace[2];
            SchemaInfoDAO schemaInfo = this.repositories.getSchemaRepository().getSchemaDAO(session, catalog, schema);
            if (schemaInfo == null) {
                throw new BaseException(ErrorCode.NOT_FOUND, "Schema not found: " + catalog + "." + schema);
            }
            this.deleteVolume(session, schemaInfo.getId(), volume);
            return null;
        }, "Failed to delete volume", false);
    }

    public void deleteVolume(Session session, UUID schemaId, String volumeName) {
        VolumeInfoDAO volumeInfoDAO = this.getVolumeDAO(session, schemaId, volumeName);
        if (volumeInfoDAO == null) {
            throw new BaseException(ErrorCode.NOT_FOUND, "Volume not found: " + volumeName);
        }
        if (VolumeType.MANAGED.getValue().equals(volumeInfoDAO.getVolumeType())) {
            try {
                this.fileOperations.deleteDirectory(volumeInfoDAO.getStorageLocation());
            }
            catch (Exception e) {
                LOGGER.error("Error deleting volume directory", (Throwable)e);
            }
        }
        session.remove((Object)volumeInfoDAO);
        LOGGER.info("Deleted volume: {}", (Object)volumeInfoDAO.getName());
    }
}

