/*
 * Decompiled with CFR 0.152.
 */
package io.unitycatalog.server.persist.utils;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicSessionCredentials;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import io.unitycatalog.server.exception.BaseException;
import io.unitycatalog.server.exception.ErrorCode;
import io.unitycatalog.server.persist.utils.UriUtils;
import io.unitycatalog.server.utils.Constants;
import io.unitycatalog.server.utils.ServerProperties;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Comparator;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileOperations {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileOperations.class);
    private final ServerProperties serverProperties;
    private static String modelStorageRootCached;
    private static String modelStorageRootPropertyCached;

    public FileOperations(ServerProperties serverProperties) {
        this.serverProperties = serverProperties;
    }

    private static void reset() {
        modelStorageRootPropertyCached = null;
        modelStorageRootCached = null;
    }

    private String getModelStorageRoot() {
        String currentModelStorageRoot = this.serverProperties.get(ServerProperties.Property.MODEL_STORAGE_ROOT);
        if (modelStorageRootPropertyCached != currentModelStorageRoot) {
            FileOperations.reset();
        }
        if (modelStorageRootCached != null) {
            return modelStorageRootCached;
        }
        String modelStorageRoot = currentModelStorageRoot;
        if (modelStorageRoot == null) {
            modelStorageRoot = System.getProperty("user.dir");
        }
        if (!UriUtils.isValidURI(modelStorageRoot)) {
            modelStorageRoot = Paths.get(modelStorageRoot, new String[0]).toUri().toString();
        }
        while (modelStorageRoot.endsWith("/")) {
            modelStorageRoot = modelStorageRoot.substring(0, modelStorageRoot.length() - 1);
        }
        modelStorageRootCached = modelStorageRoot;
        modelStorageRootPropertyCached = currentModelStorageRoot;
        return modelStorageRoot;
    }

    private String getModelDirectoryURI(String entityFullName) {
        return this.getModelStorageRoot() + "/" + entityFullName.replace(".", "/");
    }

    public String getModelStorageLocation(String catalogId, String schemaId, String modelId) {
        return this.getModelDirectoryURI(catalogId + "." + schemaId + ".models." + modelId);
    }

    public String getModelVersionStorageLocation(String catalogId, String schemaId, String modelId, String versionId) {
        return this.getModelDirectoryURI(catalogId + "." + schemaId + ".models." + modelId + ".versions." + versionId);
    }

    private static URI createURI(String uri) {
        if (uri.startsWith("s3://") || uri.startsWith("file:")) {
            return URI.create(uri);
        }
        return Paths.get(uri, new String[0]).toUri();
    }

    public void deleteDirectory(String path) {
        URI directoryUri = FileOperations.createURI(path);
        FileOperations.validateURI(directoryUri);
        if (directoryUri.getScheme() == null || directoryUri.getScheme().equals("file")) {
            try {
                FileOperations.deleteLocalDirectory(Paths.get(directoryUri));
            }
            catch (IOException | RuntimeException e) {
                throw new BaseException(ErrorCode.INTERNAL, "Failed to delete directory: " + path, e);
            }
        } else if (directoryUri.getScheme().equals("s3")) {
            this.modifyS3Directory(directoryUri, false);
        } else {
            throw new BaseException(ErrorCode.INVALID_ARGUMENT, "Unsupported URI scheme: " + directoryUri.getScheme());
        }
    }

    private static void deleteLocalDirectory(Path dirPath) throws IOException {
        if (Files.exists(dirPath, new LinkOption[0])) {
            try (Stream<Path> walk = Files.walk(dirPath, FileVisitOption.FOLLOW_LINKS);){
                walk.sorted(Comparator.reverseOrder()).forEach(path -> {
                    try {
                        Files.delete(path);
                    }
                    catch (IOException e) {
                        throw new RuntimeException("Failed to delete " + String.valueOf(path), e);
                    }
                });
            }
        } else {
            throw new IOException("Directory does not exist: " + String.valueOf(dirPath));
        }
    }

    private URI modifyS3Directory(URI parsedUri, boolean createOrDelete) {
        ObjectListing listing;
        String bucketName = parsedUri.getHost();
        Object path = parsedUri.getPath().substring(1);
        String accessKey = this.serverProperties.getProperty("aws.s3.accessKey");
        String secretKey = this.serverProperties.getProperty("aws.s3.secretKey");
        String sessionToken = this.serverProperties.getProperty("aws.s3.sessionToken");
        String region = this.serverProperties.getProperty("aws.region");
        BasicSessionCredentials sessionCredentials = new BasicSessionCredentials(accessKey, secretKey, sessionToken);
        AmazonS3 s3Client = (AmazonS3)((AmazonS3ClientBuilder)((AmazonS3ClientBuilder)AmazonS3ClientBuilder.standard().withCredentials((AWSCredentialsProvider)new AWSStaticCredentialsProvider((AWSCredentials)sessionCredentials))).withRegion(region)).build();
        if (createOrDelete) {
            if (!((String)path).endsWith("/")) {
                path = (String)path + "/";
            }
            if (s3Client.doesObjectExist(bucketName, (String)path)) {
                throw new BaseException(ErrorCode.ALREADY_EXISTS, "Directory already exists: " + (String)path);
            }
            try {
                byte[] emptyContent = new byte[]{};
                ByteArrayInputStream emptyContentStream = new ByteArrayInputStream(emptyContent);
                ObjectMetadata metadata = new ObjectMetadata();
                metadata.setContentLength(0L);
                s3Client.putObject(new PutObjectRequest(bucketName, (String)path, (InputStream)emptyContentStream, metadata));
                LOGGER.debug("Directory created successfully: {}", path);
                return URI.create(String.format("s3://%s/%s", bucketName, path));
            }
            catch (Exception e) {
                throw new BaseException(ErrorCode.INTERNAL, "Failed to create directory: " + (String)path, e);
            }
        }
        ListObjectsRequest req = new ListObjectsRequest().withBucketName(bucketName).withPrefix((String)path);
        do {
            listing = s3Client.listObjects(req);
            listing.getObjectSummaries().forEach(object -> s3Client.deleteObject(bucketName, object.getKey()));
            req.setMarker(listing.getNextMarker());
        } while (listing.isTruncated());
        return URI.create(String.format("s3://%s/%s", bucketName, path));
    }

    private static URI adjustFileUri(URI fileUri) {
        Object uriString = fileUri.toString();
        if (((String)uriString).startsWith("file:/") && !((String)uriString).startsWith("file:///")) {
            uriString = "file://" + ((String)uriString).substring(5);
        }
        return URI.create((String)uriString);
    }

    public static String convertRelativePathToURI(String url) {
        if (url == null) {
            return null;
        }
        if (FileOperations.isSupportedCloudStorageUri(url)) {
            return url;
        }
        return FileOperations.adjustFileUri(FileOperations.createURI(url)).toString();
    }

    public static boolean isSupportedCloudStorageUri(String url) {
        String scheme = URI.create(url).getScheme();
        return scheme != null && Constants.SUPPORTED_SCHEMES.contains(scheme);
    }

    private static void validateURI(URI uri) {
        if (uri.getScheme() == null) {
            throw new BaseException(ErrorCode.INVALID_ARGUMENT, "Invalid path: " + uri.getPath());
        }
        URI normalized = uri.normalize();
        if (!normalized.getPath().startsWith(uri.getPath())) {
            throw new BaseException(ErrorCode.INVALID_ARGUMENT, "Normalization failed: " + uri.getPath());
        }
    }

    public static void assertValidLocation(String location) {
        FileOperations.validateURI(URI.create(location));
    }
}

