package io.uok.spacex.cache;

import com.google.common.collect.Maps;
import io.uok.spacex.redis.serializer.KryoRedisSerializer;
import java.time.Duration;
import java.util.Map;
import org.springframework.data.redis.cache.RedisCache;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.cache.RedisCacheWriter;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.serializer.RedisSerializationContext.SerializationPair;

public class SpacexRedisCacheManager extends RedisCacheManager {

  private String namespace;
  private Map<String, RedisCache> spacexRedisCacheMap = Maps.newConcurrentMap();

  public SpacexRedisCacheManager(RedisConnectionFactory redisConnectionFactory, String namespace,
      long defaultExpiration) {
    super(RedisCacheWriter.lockingRedisCacheWriter(redisConnectionFactory),
        newConfig(namespace, defaultExpiration));
    this.namespace = namespace;
  }

  protected static RedisCacheConfiguration newConfig(String namespace, long duration) {
    return RedisCacheConfiguration.defaultCacheConfig()
        .entryTtl(
            duration == 0 ? Duration.ZERO : Duration.ofSeconds(duration))
        .computePrefixWith(name -> "$cache::" + namespace + "::" + name + "::")
        .serializeValuesWith(SerializationPair.fromSerializer(new KryoRedisSerializer()));
  }

  protected RedisCache getMissingCache(String name) {
    if (spacexRedisCacheMap.containsKey(name)) {
      return spacexRedisCacheMap.get(name);
    } else {
      return super.getMissingCache(name);
    }
  }

  public void addRedisCache(String name, long expireSeconds) {
    spacexRedisCacheMap
        .put(name,
            createRedisCache(name, newConfig(namespace, expireSeconds)));
  }
}

