package io.uok.spacex.core.service.impl;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.querydsl.core.QueryResults;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQueryFactory;
import io.uok.spacex.core.dao.DictionaryDAO;
import io.uok.spacex.core.dao.DictionaryItemDAO;
import io.uok.spacex.core.domain.DictionaryDO;
import io.uok.spacex.core.domain.DictionaryItemDO;
import io.uok.spacex.core.domain.QDictionaryDO;
import io.uok.spacex.core.domain.QDictionaryItemDO;
import io.uok.spacex.core.dto.DictItemDTO;
import io.uok.spacex.core.service.DictionarySV;
import io.uok.spacex.exception.BusinessException;
import java.util.List;
import javax.annotation.Resource;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

public class DictionarySVImpl implements DictionarySV {

  @Resource
  private DictionaryDAO dictionaryDAO;

  @Resource
  private DictionaryItemDAO dictionaryItemDAO;

  @Resource
  private JPAQueryFactory jpaQueryFactory;


  @Override
  public void add(DictionaryDO dictionary) {
    // 判断code是否重复
    QDictionaryDO qDictionary = QDictionaryDO.dictionaryDO;
    boolean exists = dictionaryDAO.exists(qDictionary.code.eq(dictionary.getCode()));
    if (exists) {
      throw new BusinessException("字典编码已经存在，添加失败.");
    }
    dictionaryDAO.save(dictionary);
  }

  @Override
  public void update(DictionaryDO dictionary) {
    // 判断code是否重复
    QDictionaryDO qDictionary = QDictionaryDO.dictionaryDO;
    Predicate predicate = qDictionary.code.eq(dictionary.getCode())
        .and(qDictionary.id.ne(dictionary.getId()));
    boolean exists = dictionaryDAO.exists(predicate);
    if (exists) {
      throw new BusinessException("字典编码已经存在，修改失败.");
    }
    dictionaryDAO.save(dictionary);
  }

  @Override
  public DictionaryDO get(String code) {
    return dictionaryDAO.findByCode(code);
  }

  @Override
  public void delete(Long id) {
    dictionaryDAO.deleteById(id);
  }

  @Override
  public Page<DictionaryDO> find(String name, String code, String description, Pageable pageable) {
    QDictionaryDO qDictionary = QDictionaryDO.dictionaryDO;
    List<Predicate> predicates = Lists.newArrayList();
    if (!Strings.isNullOrEmpty(name)) {
      predicates.add(qDictionary.name.like(name));
    }
    if (!Strings.isNullOrEmpty(code)) {
      predicates.add(qDictionary.code.eq(code));
    }
    if (!Strings.isNullOrEmpty(description)) {
      predicates.add(qDictionary.description.like(description));
    }

    return dictionaryDAO.findAll(ExpressionUtils.allOf(predicates), pageable);
  }

  @Override
  public List<DictItemDTO> getItems(String code) {
    QDictionaryDO qDictionary = QDictionaryDO.dictionaryDO;
    QDictionaryItemDO qDictionaryItem = QDictionaryItemDO.dictionaryItemDO;

    QueryResults<DictItemDTO> results = jpaQueryFactory.select(Projections
        .bean(DictItemDTO.class, qDictionary.code, qDictionaryItem.label,
            qDictionaryItem.value))
        .from(qDictionary)
        .leftJoin(qDictionaryItem)
        .on(qDictionary.id.eq(qDictionaryItem.dictionaryId))
        .where(qDictionary.code.eq(code))
        .where(qDictionaryItem.status.eq(1))
        .fetchResults();

    return results.getResults();
  }

  @Override
  public void addItem(DictionaryItemDO dictionaryItem) {
    QDictionaryItemDO qDictionaryItem = QDictionaryItemDO.dictionaryItemDO;
    boolean exists = dictionaryItemDAO
        .exists(qDictionaryItem.dictionaryId.eq(dictionaryItem.getDictionaryId())
            .and(qDictionaryItem.label.eq(dictionaryItem.getLabel())
                .or(qDictionaryItem.value.eq(dictionaryItem.getValue()))));
    if (exists) {
      throw new BusinessException("字典项值或者文本存在重复，添加失败.");
    }
    dictionaryItemDAO.save(dictionaryItem);
  }

  @Override
  public void updateItem(DictionaryItemDO dictionaryItem) {
    QDictionaryItemDO qDictionaryItem = QDictionaryItemDO.dictionaryItemDO;
    boolean exists = dictionaryItemDAO.exists(
        qDictionaryItem.id.ne(dictionaryItem.getId())
            .and(qDictionaryItem.dictionaryId.eq(dictionaryItem.getDictionaryId()))
            .and(
                qDictionaryItem.label.eq(dictionaryItem.getLabel())
                    .or(
                        qDictionaryItem.value.eq(dictionaryItem.getValue())
                    )
            )
    );
    if (exists) {
      throw new BusinessException("字典项值或者文本存在重复，添加失败.");
    }
    dictionaryItemDAO.save(dictionaryItem);
  }

  @Override
  public void deleteItem(Long itemId) {
    dictionaryItemDAO.deleteById(itemId);
  }

  @Override
  public List<DictionaryItemDO> findItems(Long id) {
    QDictionaryItemDO qDictionaryItem = QDictionaryItemDO.dictionaryItemDO;
    return Lists.newArrayList(dictionaryItemDAO.findAll(qDictionaryItem.dictionaryId.eq(id)));
  }
}
