package io.uok.spacex.redis;

import com.google.common.base.Splitter;
import java.util.List;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.data.redis.RedisAutoConfiguration;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.connection.RedisClusterConfiguration;
import org.springframework.data.redis.connection.RedisPassword;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;

@Configuration
@ConfigurationProperties(prefix = "spacex.redis")
@ConditionalOnProperty(name = "spacex.redis.enable", havingValue = "true", matchIfMissing = true)
@AutoConfigureBefore(RedisAutoConfiguration.class)
public class RedisConfig {

  @Value("${spacex.redis.servers}")
  private String servers;
  @Value("${spacex.redis.password:}")
  private String password;

  private static final Splitter serverSplitter = Splitter.on(',').trimResults();
  private static final Splitter splitter = Splitter.on(':');


  @Bean
  public LettuceConnectionFactory redisConnectionFactory() {
    List<String> serverList = serverSplitter.splitToList(servers);
    if (1 < serverList.size()) {
      RedisClusterConfiguration config = new RedisClusterConfiguration();
      for (String server : serverList) {
        List<String> hostPort = splitter.splitToList(server);
        config.clusterNode(hostPort.get(0), Integer.parseInt(hostPort.get(1)));
      }
      config.setMaxRedirects(3);
      if (password != null) {
        config.setPassword(RedisPassword.of(password));
      }
      return new LettuceConnectionFactory(config);
    } else {
      List<String> hostPort = splitter.splitToList(serverList.get(0));
      RedisStandaloneConfiguration config = new RedisStandaloneConfiguration();
      config.setHostName(hostPort.get(0));
      config.setPort(Integer.parseInt(hostPort.get(1)));
      if (password != null) {
        config.setPassword(RedisPassword.of(password));
      }
      config.setDatabase(0);
      return new LettuceConnectionFactory(config);
    }
  }
}
