package io.uok.spacex.task.service.impl;

import com.google.common.collect.Lists;
import io.uok.spacex.exception.SystemException;
import io.uok.spacex.task.JobState;
import io.uok.spacex.task.dto.JobDTO;
import io.uok.spacex.task.service.TaskSV;
import java.util.List;
import java.util.Set;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.quartz.CronExpression;
import org.quartz.CronScheduleBuilder;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.JobExecutionContext;
import org.quartz.JobKey;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.impl.matchers.GroupMatcher;
import org.springframework.context.annotation.Lazy;
import org.springframework.util.CollectionUtils;

@Slf4j
public class TaskSVImpl implements TaskSV {

  @Resource
  @Lazy
  private Scheduler scheduler;

  @Override
  public List<JobDTO> getAllJobs() {
    try {
      List<JobDTO> jobs = Lists.newArrayList();
      List<String> groups = scheduler.getJobGroupNames();
      if (CollectionUtils.isEmpty(groups)) {
        return null;
      }
      for (String group : groups) {
        Set<JobKey> jobKeys = scheduler.getJobKeys(GroupMatcher.jobGroupEquals(group));
        for (JobKey jobKey : jobKeys) {
          JobDetail jobDetail = scheduler.getJobDetail(jobKey);
          List<CronTrigger> triggers = (List<CronTrigger>) scheduler.getTriggersOfJob(jobKey);
          if (!CollectionUtils.isEmpty(triggers)) {
            for (CronTrigger trigger : triggers) {
              JobDTO jobDTO = newJobDTO(jobDetail, trigger);
              jobs.add(jobDTO);
            }
          }
        }
      }

      return jobs;
    } catch (SchedulerException e) {
      log.error("获取全部任务失败", e);
      throw new SystemException("获取全部任务失败");
    }
  }

  private JobDTO newJobDTO(JobDetail jobDetail, CronTrigger trigger) throws SchedulerException {
    JobDTO jobDTO = new JobDTO();
    jobDTO.setName(jobDetail.getKey().getName());
    jobDTO.setClazz(jobDetail.getDescription());
    jobDTO.setCron(trigger.getCronExpression());
    jobDTO
        .setState(JobState.valueOf(scheduler.getTriggerState(trigger.getKey()).name()));
    jobDTO.setPreviousFireTime(trigger.getPreviousFireTime());
    jobDTO.setNextFireTime(trigger.getNextFireTime());
    return jobDTO;
  }

  @Override
  public List<JobDTO> getExecutingJobs() {
    try {
      List<JobDTO> jobs = Lists.newArrayList();
      List<JobExecutionContext> jobExecutionContexts = scheduler.getCurrentlyExecutingJobs();
      if (CollectionUtils.isEmpty(jobExecutionContexts)) {
        return null;
      }

      for (JobExecutionContext jobExecutionContext : jobExecutionContexts) {
        JobDTO jobDTO = newJobDTO(jobExecutionContext.getJobDetail(),
            (CronTrigger) jobExecutionContext.getTrigger());
        jobs.add(jobDTO);
      }
      return jobs;
    } catch (SchedulerException e) {
      log.error("获取全部任务失败", e);
      throw new SystemException("获取全部任务失败");
    }
  }

  @Override
  public void pauseAllJobs() {
    try {
      scheduler.pauseAll();
    } catch (SchedulerException e) {
      log.error("暂停全部任务失败", e);
      throw new SystemException("暂停全部任务失败");
    }
  }

  @Override
  public void resumeAllJobs() {
    try {
      scheduler.resumeAll();
    } catch (SchedulerException e) {
      log.error("恢复全部任务失败", e);
      throw new SystemException("恢复全部任务失败");
    }
  }

  @Override
  public void pauseJob(String jobName) {
    try {
      scheduler.pauseJob(new JobKey(jobName));
    } catch (SchedulerException e) {
      log.error("暂停任务[{}]失败", jobName, e);
      throw new SystemException("暂停任务" + jobName + "失败");
    }
  }

  @Override
  public void resumeJob(String jobName) {
    try {
      scheduler.resumeJob(new JobKey(jobName));
    } catch (SchedulerException e) {
      log.error("恢复任务[{}]失败", jobName, e);
      throw new SystemException("恢复任务" + jobName + "失败");
    }
  }

  @Override
  public JobDTO getJob(String jobName) {
    try {
      JobKey jobKey = new JobKey(jobName);
      JobDetail jobDetail = scheduler.getJobDetail(jobKey);
      List<CronTrigger> triggers = (List<CronTrigger>) scheduler.getTriggersOfJob(jobKey);
      if (!CollectionUtils.isEmpty(triggers)) {
        for (CronTrigger trigger : triggers) {
          return newJobDTO(jobDetail, trigger);
        }
      }
    } catch (SchedulerException e) {
      log.error("获取任务[{}]失败", jobName, e);
      throw new SystemException("获取任务" + jobName + "失败");
    }
    return null;
  }

  @Override
  public boolean updateJobCron(String jobName, String cron) {
    if (!CronExpression.isValidExpression(cron)) {
      log.error("Cron表达式不正确({})", cron);
      return false;
    }
    try {

      List<CronTrigger> cronTriggers = (List<CronTrigger>) scheduler
          .getTriggersOfJob(new JobKey(jobName));
      if (CollectionUtils.isEmpty(cronTriggers)) {
        log.error("不存在任务{}，更新cron[{}] 失败", jobName, cron);
        return false;
      }
      for (CronTrigger cronTrigger : cronTriggers) {
        Trigger newTrigger = cronTrigger.getTriggerBuilder()
            .withSchedule(CronScheduleBuilder.cronSchedule(cron)).build();
        scheduler.rescheduleJob(cronTrigger.getKey(), newTrigger);
      }
      return true;
    } catch (SchedulerException e) {
      log.error("更新任务失败:{}", jobName, e);
    }
    return false;
  }
}
