/*
 * Decompiled with CFR 0.152.
 */
package io.valkey.executors;

import io.valkey.CommandObject;
import io.valkey.Connection;
import io.valkey.annots.VisibleForTesting;
import io.valkey.exceptions.JedisConnectionException;
import io.valkey.exceptions.JedisException;
import io.valkey.exceptions.JedisRedirectionException;
import io.valkey.executors.CommandExecutor;
import io.valkey.providers.RedirectConnectionProvider;
import io.valkey.util.IOUtils;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RedirectCommandExecutor
implements CommandExecutor {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    public final RedirectConnectionProvider provider;
    protected final int maxAttempts;
    protected final Duration maxTotalRetriesDuration;

    public RedirectCommandExecutor(RedirectConnectionProvider provider, int maxAttempts, Duration maxTotalRetriesDuration) {
        this.provider = provider;
        this.maxAttempts = maxAttempts;
        this.maxTotalRetriesDuration = maxTotalRetriesDuration;
    }

    @Override
    public void close() {
        this.provider.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public final <T> T executeCommand(CommandObject<T> commandObject) {
        Instant deadline = Instant.now().plus(this.maxTotalRetriesDuration);
        int consecutiveConnectionFailures = 0;
        JedisException lastException = null;
        for (int attemptsLeft = this.maxAttempts; attemptsLeft > 0; --attemptsLeft) {
            Connection connection = null;
            try {
                connection = this.provider.getConnection();
                T t = this.execute(connection, commandObject);
                return t;
            }
            catch (JedisConnectionException jce) {
                lastException = jce;
                this.log.debug("Failed connecting to Redis: {}", (Object)connection, (Object)jce);
                boolean reset = this.handleConnectionProblem(attemptsLeft - 1, ++consecutiveConnectionFailures, deadline);
                if (reset) {
                    consecutiveConnectionFailures = 0;
                }
            }
            catch (JedisRedirectionException jre) {
                if (lastException == null || lastException instanceof JedisRedirectionException) {
                    lastException = jre;
                }
                this.log.debug("Redirected by server to {}", (Object)jre.getTargetNode());
                consecutiveConnectionFailures = 0;
                this.provider.renewPool(connection, jre.getTargetNode());
            }
            finally {
                IOUtils.closeQuietly(connection);
            }
            if (!Instant.now().isAfter(deadline)) continue;
            throw new JedisException("Redirect retry deadline exceeded.");
        }
        JedisException maxRedirectException = new JedisException("No more redirect attempts left.");
        maxRedirectException.addSuppressed(lastException);
        throw maxRedirectException;
    }

    @VisibleForTesting
    protected <T> T execute(Connection connection, CommandObject<T> commandObject) {
        return connection.executeCommand(commandObject);
    }

    private boolean handleConnectionProblem(int attemptsLeft, int consecutiveConnectionFailures, Instant doneDeadline) {
        if (this.maxAttempts < 3) {
            if (attemptsLeft == 0) {
                this.provider.renewPool(null, null);
                return true;
            }
            return false;
        }
        if (consecutiveConnectionFailures < 2) {
            return false;
        }
        this.sleep(RedirectCommandExecutor.getBackoffSleepMillis(attemptsLeft, doneDeadline));
        this.provider.renewPool(null, null);
        return true;
    }

    private static long getBackoffSleepMillis(int attemptsLeft, Instant deadline) {
        if (attemptsLeft <= 0) {
            return 0L;
        }
        long millisLeft = Duration.between(Instant.now(), deadline).toMillis();
        if (millisLeft < 0L) {
            throw new JedisException("Redirect retry deadline exceeded.");
        }
        long maxBackOff = millisLeft / (long)(attemptsLeft * attemptsLeft);
        return ThreadLocalRandom.current().nextLong(maxBackOff + 1L);
    }

    @VisibleForTesting
    protected void sleep(long sleepMillis) {
        try {
            TimeUnit.MILLISECONDS.sleep(sleepMillis);
        }
        catch (InterruptedException e) {
            throw new JedisException(e);
        }
    }
}

