/*
 * Copyright (c) 2016 The original author or authors
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Apache License v2.0 which accompanies this distribution.
 *
 *      The Eclipse Public License is available at
 *      http://www.eclipse.org/legal/epl-v10.html
 *
 *      The Apache License v2.0 is available at
 *      http://www.opensource.org/licenses/apache2.0.php
 *
 * You may elect to redistribute this code under either of these licenses.
 */
package io.vertx.ext.consul;

import io.vertx.codegen.annotations.DataObject;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Defines a prepared query.
 *
 * @author <a href="mailto:ruslan.sennov@gmail.com">Ruslan Sennov</a>
 */
@DataObject
public class PreparedQueryDefinition {

  private String id;
  private String name;
  private String session;
  private String service;
  private String token;
  private String dnsTtl;
  private int nearestN;
  private List<String> dcs;
  private List<String> tags;
  private boolean passing;
  private Map<String, String> meta;
  private String templateType;
  private String templateRegexp;

  /**
   * Default constructor
   */
  public PreparedQueryDefinition() {}

  /**
   * Constructor from JSON
   *
   * @param json the JSON
   */
  public PreparedQueryDefinition(JsonObject json) {
    id = json.getString("ID");
    name = json.getString("Name");
    session = json.getString("Session");
    token = json.getString("Token");
    service = json.getJsonObject("Service").getString("Service");
    nearestN = json.getJsonObject("Service").getJsonObject("Failover").getInteger("NearestN");
    JsonArray dcsJson = json.getJsonObject("Service").getJsonObject("Failover").getJsonArray("Datacenters");
    if (dcsJson != null) {
      dcs = dcsJson.stream().map(o -> (String) o).collect(Collectors.toList());
    }
    passing = json.getJsonObject("Service").getBoolean("OnlyPassing");
    JsonArray tagsJson = json.getJsonObject("Service").getJsonArray("Tags");
    if (tagsJson != null) {
      tags = tagsJson.stream().map(o -> (String) o).collect(Collectors.toList());
    }
    JsonObject metaJson = json.getJsonObject("Service").getJsonObject("NodeMeta");
    if (metaJson != null) {
      meta = metaJson.stream().collect(Collectors.toMap(Map.Entry::getKey, e -> (String) e.getValue()));
    }
    JsonObject templateJson = json.getJsonObject("Template");
    if (templateJson != null) {
      templateType = templateJson.getString("Type");
      templateRegexp = templateJson.getString("Regexp");
    }
    dnsTtl = json.getJsonObject("DNS").getString("TTL");
  }

  /**
   * Convert to JSON
   *
   * @return the JSON
   */
  public JsonObject toJson() {
    return new JsonObject()
      .put("ID", id)
      .put("Name", name)
      .put("Session", session)
      .put("Token", token)
      .put("Template", new JsonObject()
        .put("Type", templateType)
        .put("Regexp", templateRegexp))
      .put("Service", new JsonObject()
        .put("Service", service)
        .put("Failover", new JsonObject()
          .put("NearestN", nearestN)
          .put("Datacenters", dcs))
        .put("OnlyPassing", passing)
        .put("Tags", tags)
        .put("NodeMeta", meta))
      .put("DNS", new JsonObject()
        .put("TTL", dnsTtl));
  }

  /**
   * Get ID of the query
   *
   * @return ID of the query
   */
  public String getId() {
    return id;
  }

  /**
   * Set ID of the query, always generated by Consul
   *
   * @param id ID of the query
   * @return reference to this, for fluency
   */
  public PreparedQueryDefinition setId(String id) {
    this.id = id;
    return this;
  }

  /**
   * Get an optional friendly name that can be used to execute a query instead of using its ID
   *
   * @return name of the query
   */
  public String getName() {
    return name;
  }

  /**
   * Set an optional friendly name that can be used to execute a query instead of using its ID
   *
   * @param name name of the query
   * @return reference to this, for fluency
   */
  public PreparedQueryDefinition setName(String name) {
    this.name = name;
    return this;
  }

  /**
   * Get the ID of an existing session. This provides a way to automatically remove a prepared query when
   * the given session is invalidated. If not given the prepared query must be manually removed when no longer needed.
   *
   * @return id of session
   */
  public String getSession() {
    return session;
  }

  /**
   * Set the ID of an existing session. This provides a way to automatically remove a prepared query when
   * the given session is invalidated. If not given the prepared query must be manually removed when no longer needed.
   *
   * @param session id of session
   * @return reference to this, for fluency
   */
  public PreparedQueryDefinition setSession(String session) {
    this.session = session;
    return this;
  }

  /**
   * Get the name of the service to query
   *
   * @return service name
   */
  public String getService() {
    return service;
  }

  /**
   * Set the name of the service to query
   *
   * @param service service name
   * @return reference to this, for fluency
   */
  public PreparedQueryDefinition setService(String service) {
    this.service = service;
    return this;
  }

  /**
   * Get the ACL token to use each time the query is executed. This allows queries to be executed by clients
   * with lesser or even no ACL Token, so this should be used with care.
   *
   * @return the ACL token
   */
  public String getToken() {
    return token;
  }

  /**
   * Set the ACL token to use each time the query is executed. This allows queries to be executed by clients
   * with lesser or even no ACL Token, so this should be used with care.
   *
   * @param token the ACL token
   * @return reference to this, for fluency
   */
  public PreparedQueryDefinition setToken(String token) {
    this.token = token;
    return this;
  }

  /**
   * Get the TTL duration when query results are served over DNS. If this is specified,
   * it will take precedence over any Consul agent-specific configuration.
   *
   * @return the TTL duration
   */
  public String getDnsTtl() {
    return dnsTtl;
  }

  /**
   * Set the TTL duration when query results are served over DNS. If this is specified,
   * it will take precedence over any Consul agent-specific configuration.
   *
   * @param dnsTtl the TTL duration
   * @return reference to this, for fluency
   */
  public PreparedQueryDefinition setDnsTtl(String dnsTtl) {
    this.dnsTtl = dnsTtl;
    return this;
  }

  /**
   * Specifies that the query will be forwarded to up to NearestN other datacenters based on their estimated network
   * round trip time using Network Coordinates from the WAN gossip pool. The median round trip time from the server
   * handling the query to the servers in the remote datacenter is used to determine the priority.
   *
   * @return number of nearest datacenters to query
   */
  public int getNearestN() {
    return nearestN;
  }

  /**
   * Specifies that the query will be forwarded to up to NearestN other datacenters based on their estimated network
   * round trip time using Network Coordinates from the WAN gossip pool. The median round trip time from the server
   * handling the query to the servers in the remote datacenter is used to determine the priority.
   *
   * @param nearestN number of nearest datacenters to query
   * @return reference to this, for fluency
   */
  public PreparedQueryDefinition setNearestN(int nearestN) {
    this.nearestN = nearestN;
    return this;
  }

  /**
   * Specifies a fixed list of remote datacenters to forward the query to if there are no healthy nodes
   * in the local datacenter. Datacenters are queried in the order given in the list. If this option is combined
   * with NearestN, then the NearestN queries will be performed first, followed by the list given by Datacenters.
   * A given datacenter will only be queried one time during a failover, even if it is selected by both NearestN
   * and is listed in Datacenters.
   *
   * @return the list of remote datacenters
   */
  public List<String> getDcs() {
    return dcs;
  }

  /**
   * Specifies a fixed list of remote datacenters to forward the query to if there are no healthy nodes
   * in the local datacenter. Datacenters are queried in the order given in the list. If this option is combined
   * with NearestN, then the NearestN queries will be performed first, followed by the list given by Datacenters.
   * A given datacenter will only be queried one time during a failover, even if it is selected by both NearestN
   * and is listed in Datacenters.
   *
   * @param dcs the list of remote datacenters
   * @return reference to this, for fluency
   */
  public PreparedQueryDefinition setDcs(List<String> dcs) {
    this.dcs = dcs;
    return this;
  }

  /**
   * Get a list of service tags to filter the query results. For a service to pass the tag filter
   * it must have all of the required tags, and none of the excluded tags (prefixed with `!`).
   *
   * @return list of service tags
   */
  public List<String> getTags() {
    return tags;
  }

  /**
   * Set a list of service tags to filter the query results. For a service to pass the tag filter
   * it must have all of the required tags, and none of the excluded tags (prefixed with `!`).
   *
   * @param tags list of service tags
   * @return reference to this, for fluency
   */
  public PreparedQueryDefinition setTags(List<String> tags) {
    this.tags = tags;
    return this;
  }

  /**
   * Specifies the behavior of the query's health check filtering. If this is set to false, the results will include
   * nodes with checks in the passing as well as the warning states. If this is set to true,
   * only nodes with checks in the passing state will be returned.
   *
   * @return the passing flag
   */
  public boolean getPassing() {
    return passing;
  }

  /**
   * Specifies the behavior of the query's health check filtering. If this is set to false, the results will include
   * nodes with checks in the passing as well as the warning states. If this is set to true,
   * only nodes with checks in the passing state will be returned.
   *
   * @param passing the passing flag
   * @return reference to this, for fluency
   */
  public PreparedQueryDefinition setPassing(boolean passing) {
    this.passing = passing;
    return this;
  }

  /**
   * Get a list of user-defined key/value pairs that will be used for filtering the query results to nodes
   * with the given metadata values present.
   *
   * @return list of key/value pairs
   */
  public Map<String, String> getMeta() {
    return meta;
  }

  /**
   * Set a list of user-defined key/value pairs that will be used for filtering the query results to nodes
   * with the given metadata values present.
   *
   * @param meta list of key/value pairs
   * @return reference to this, for fluency
   */
  public PreparedQueryDefinition setMeta(Map<String, String> meta) {
    this.meta = meta;
    return this;
  }

  /**
   * The template type, which must be {@code name_prefix_match}. This means that the template will apply to
   * any query lookup with a name whose prefix matches the Name field of the template.
   *
   * @return the query type
   * @see <a href="https://www.consul.io/api/query.html#prepared-query-templates">Prepared Query Templates</a> endpoint
   */
  public String getTemplateType() {
    return templateType;
  }

  /**
   * The template type, which must be {@code name_prefix_match}. This means that the template will apply to
   * any query lookup with a name whose prefix matches the Name field of the template.
   *
   * @param templateType the query type
   * @return reference to this, for fluency
   * @see <a href="https://www.consul.io/api/query.html#prepared-query-templates">Prepared Query Templates</a> endpoint
   */
  public PreparedQueryDefinition setTemplateType(String templateType) {
    this.templateType = templateType;
    return this;
  }

  /**
   * Get regular expression which is used to extract fields from the entire name, once this template is selected.
   *
   * @return a regular expression
   * @see <a href="https://www.consul.io/api/query.html#prepared-query-templates">Prepared Query Templates</a> endpoint
   */
  public String getTemplateRegexp() {
    return templateRegexp;
  }

  /**
   * Set regular expression which is used to extract fields from the entire name, once this template is selected.
   *
   * @param templateRegexp a regular expression
   * @return reference to this, for fluency
   * @see <a href="https://www.consul.io/api/query.html#prepared-query-templates">Prepared Query Templates</a> endpoint
   */
  public PreparedQueryDefinition setTemplateRegexp(String templateRegexp) {
    this.templateRegexp = templateRegexp;
    return this;
  }
}
