/*
 * Copyright (c) 2013 The Netty Project
 * ------------------------------------
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Apache License v2.0 which accompanies this distribution.
 *
 *     The Eclipse Public License is available at
 *     http://www.eclipse.org/legal/epl-v10.html
 *
 *     The Apache License v2.0 is available at
 *     http://www.opensource.org/licenses/apache2.0.php
 *
 * You may elect to redistribute this code under either of these licenses.
 */
package io.vertx.core.dns.impl.netty;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufHolder;

/**
 * A DNS response packet which is sent to a client after a server receives a
 * query.
 */
public class DnsResponse extends DnsMessage<DnsResponseHeader> implements ByteBufHolder {

  private final ByteBuf rawPacket;
  private final int originalIndex;

  public DnsResponse(ByteBuf rawPacket) {
    this.rawPacket = rawPacket;
    this.originalIndex = rawPacket.readerIndex();
  }

  /**
   * Returns the non-decoded DNS response packet.
   */
  @Override
  public ByteBuf content() {
    return rawPacket;
  }

  @Override
  public int refCnt() {
    return rawPacket.refCnt();
  }

  @Override
  public boolean release() {
    return rawPacket.release();
  }

  /**
   * Returns a deep copy of this DNS response.
   */
  @Override
  public DnsResponse copy() {
    return replace(rawPacket.copy());
  }

  /**
   * Returns a duplicate of this DNS response.
   */
  @Override
  public DnsResponse duplicate() {
    return replace(rawPacket.duplicate());
  }

  @Override
  public DnsResponse retain() {
    rawPacket.retain();
    return this;
  }

  @Override
  public DnsResponse retain(int increment) {
    rawPacket.retain();
    return this;
  }

  @Override
  public DnsResponse retainedDuplicate() {
    return replace(rawPacket.retainedDuplicate());
  }

  @Override
  public DnsResponse replace(ByteBuf content) {
    return DnsResponseDecoder.decodeResponse(content, content.alloc());
  }

  @Override
  public boolean release(int decrement) {
    return rawPacket.release();
  }

  /**
   * Returns the original index at which the DNS response packet starts for the {@link io.netty.buffer.ByteBuf}
   * stored in this {@link io.netty.buffer.ByteBufHolder}.
   */
  public int originalIndex() {
    return originalIndex;
  }

  @Override
  public ByteBufHolder touch() {
    rawPacket.touch();
    return this;
  }

  @Override
  public ByteBufHolder touch(Object hint) {
    rawPacket.touch(hint);
    return this;
  }
}
