/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.test.core;

import io.netty.buffer.ByteBuf;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.json.DecodeException;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import io.vertx.test.core.TestUtils;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import org.junit.Assert;
import org.junit.Test;

public class BufferTest {
    private static final int MEDIUM_MAX_VALUE = 0x1000000;
    private final int numSets = 100;

    @Test
    public void testConstructorArguments() throws Exception {
        TestUtils.assertIllegalArgumentException(() -> Buffer.buffer((int)-1));
        TestUtils.assertNullPointerException(() -> Buffer.buffer((byte[])null));
        TestUtils.assertNullPointerException(() -> Buffer.buffer((String)null));
        TestUtils.assertNullPointerException(() -> Buffer.buffer((ByteBuf)null));
        TestUtils.assertNullPointerException(() -> Buffer.buffer(null, (String)"UTF-8"));
        TestUtils.assertNullPointerException(() -> Buffer.buffer((String)"", null));
    }

    @Test
    public void testSetGetInt() throws Exception {
        int i;
        int size = 10;
        Buffer buffer = Buffer.buffer((int)10);
        for (i = 0; i < 10; ++i) {
            buffer.setInt(i * 4, (i + 1) * 10);
        }
        for (i = 0; i < 10; ++i) {
            Assert.assertEquals((long)((i + 1) * 10), (long)buffer.getInt(i * 4));
        }
    }

    @Test
    public void testAppendBuff() throws Exception {
        int bytesLen = 100;
        byte[] bytes = TestUtils.randomByteArray(bytesLen);
        Buffer toAppend = Buffer.buffer((byte[])bytes);
        Buffer b = Buffer.buffer();
        b.appendBuffer(toAppend);
        Assert.assertEquals((long)b.length(), (long)bytes.length);
        Assert.assertTrue((boolean)TestUtils.byteArraysEqual(bytes, b.getBytes()));
        b.appendBuffer(toAppend);
        Assert.assertEquals((long)b.length(), (long)(2 * bytes.length));
        TestUtils.assertNullPointerException(() -> b.appendBuffer(null));
    }

    @Test
    public void testAppendBytes() throws Exception {
        int bytesLen = 100;
        byte[] bytes = TestUtils.randomByteArray(bytesLen);
        Buffer b = Buffer.buffer();
        b.appendBytes(bytes);
        Assert.assertEquals((long)b.length(), (long)bytes.length);
        Assert.assertTrue((boolean)TestUtils.byteArraysEqual(bytes, b.getBytes()));
        b.appendBytes(bytes);
        Assert.assertEquals((long)b.length(), (long)(2 * bytes.length));
        TestUtils.assertNullPointerException(() -> b.appendBytes(null));
    }

    @Test
    public void testAppendBytesWithOffsetAndLen() throws Exception {
        int bytesLen = 100;
        byte[] bytes = TestUtils.randomByteArray(bytesLen);
        int len = bytesLen - 2;
        Buffer b = Buffer.buffer();
        b.appendBytes(bytes, 1, len);
        Assert.assertEquals((long)b.length(), (long)len);
        byte[] copy = new byte[len];
        System.arraycopy(bytes, 1, copy, 0, len);
        Assert.assertTrue((boolean)TestUtils.byteArraysEqual(copy, b.getBytes()));
        b.appendBytes(bytes, 1, len);
        Assert.assertEquals((long)b.length(), (long)(2 * len));
        TestUtils.assertNullPointerException(() -> b.appendBytes(null, 1, len));
    }

    @Test
    public void testAppendBufferWithOffsetAndLen() throws Exception {
        int bytesLen = 100;
        byte[] bytes = TestUtils.randomByteArray(bytesLen);
        Buffer src = Buffer.buffer((byte[])bytes);
        int len = bytesLen - 2;
        Buffer b = Buffer.buffer();
        b.appendBuffer(src, 1, len);
        Assert.assertEquals((long)b.length(), (long)len);
        byte[] copy = new byte[len];
        System.arraycopy(bytes, 1, copy, 0, len);
        Assert.assertTrue((boolean)TestUtils.byteArraysEqual(copy, b.getBytes()));
        b.appendBuffer(src, 1, len);
        Assert.assertEquals((long)b.length(), (long)(2 * len));
        TestUtils.assertNullPointerException(() -> b.appendBuffer(null, 1, len));
    }

    @Test
    public void testAppendByte() throws Exception {
        int i;
        int bytesLen = 100;
        byte[] bytes = TestUtils.randomByteArray(bytesLen);
        Buffer b = Buffer.buffer();
        for (i = 0; i < bytesLen; ++i) {
            b.appendByte(bytes[i]);
        }
        Assert.assertEquals((long)b.length(), (long)bytes.length);
        Assert.assertTrue((boolean)TestUtils.byteArraysEqual(bytes, b.getBytes()));
        for (i = 0; i < bytesLen; ++i) {
            b.appendByte(bytes[i]);
        }
        Assert.assertEquals((long)b.length(), (long)(2 * bytes.length));
    }

    @Test
    public void testAppendByte2() throws Exception {
        int bytesLen = 100;
        Buffer b = Buffer.buffer((byte[])TestUtils.randomByteArray(bytesLen));
        b.setByte(b.length(), (byte)9);
    }

    @Test
    public void testAppendUnsignedByte() {
        Buffer b = Buffer.buffer((byte[])TestUtils.randomByteArray(100));
        b.appendUnsignedByte((short)190);
        Assert.assertEquals((long)101L, (long)b.length());
    }

    @Test
    public void testAppendShort() {
        Buffer b = Buffer.buffer((byte[])TestUtils.randomByteArray(100));
        b.appendShort((short)Short.MAX_VALUE);
        Assert.assertEquals((long)102L, (long)b.length());
        b.appendShortLE((short)Short.MAX_VALUE);
        Assert.assertEquals((long)104L, (long)b.length());
    }

    @Test
    public void testAppendUnsignedShort() {
        Buffer b = Buffer.buffer((byte[])TestUtils.randomByteArray(100));
        b.appendUnsignedShort(49150);
        Assert.assertEquals((long)102L, (long)b.length());
        b.appendUnsignedShortLE(49150);
        Assert.assertEquals((long)104L, (long)b.length());
    }

    @Test
    public void testAppendInt() {
        Buffer b = Buffer.buffer((byte[])TestUtils.randomByteArray(100));
        b.appendInt(Integer.MAX_VALUE);
        Assert.assertEquals((long)104L, (long)b.length());
        b.appendIntLE(Integer.MAX_VALUE);
        Assert.assertEquals((long)108L, (long)b.length());
    }

    @Test
    public void testAppendUnsignedInt() {
        Buffer b = Buffer.buffer((byte[])TestUtils.randomByteArray(100));
        b.appendUnsignedInt(0xBFFFFFFEL);
        Assert.assertEquals((long)104L, (long)b.length());
        b.appendUnsignedIntLE(0xBFFFFFFEL);
        Assert.assertEquals((long)108L, (long)b.length());
    }

    @Test
    public void testAppendMedium() {
        Buffer b = Buffer.buffer((byte[])TestUtils.randomByteArray(100));
        b.appendMedium(0x1000000);
        Assert.assertEquals((long)103L, (long)b.length());
        b.appendMediumLE(0x1000000);
        Assert.assertEquals((long)106L, (long)b.length());
    }

    @Test
    public void testAppendLong() {
        Buffer b = Buffer.buffer((byte[])TestUtils.randomByteArray(100));
        b.appendLong(Long.MAX_VALUE);
        Assert.assertEquals((long)108L, (long)b.length());
        b.appendLongLE(Long.MAX_VALUE);
        Assert.assertEquals((long)116L, (long)b.length());
    }

    @Test
    public void testAppendString1() throws Exception {
        String str = TestUtils.randomUnicodeString(100);
        byte[] sb = str.getBytes("UTF-8");
        Buffer b = Buffer.buffer();
        b.appendString(str);
        Assert.assertEquals((long)b.length(), (long)sb.length);
        Assert.assertTrue((boolean)str.equals(b.toString("UTF-8")));
        Assert.assertTrue((boolean)str.equals(b.toString(StandardCharsets.UTF_8)));
        TestUtils.assertNullPointerException(() -> b.appendString(null));
        TestUtils.assertNullPointerException(() -> b.appendString(null, "UTF-8"));
        TestUtils.assertNullPointerException(() -> b.appendString("", null));
    }

    @Test
    public void testAppendString2() throws Exception {
    }

    @Test
    public void testLE() {
        this.checkBEAndLE(2, Buffer.buffer().appendShort((short)Short.MAX_VALUE), Buffer.buffer().appendShortLE((short)Short.MAX_VALUE));
        this.checkBEAndLE(2, Buffer.buffer().appendUnsignedShort(Short.MAX_VALUE), Buffer.buffer().appendUnsignedShortLE(Short.MAX_VALUE));
        this.checkBEAndLE(3, Buffer.buffer().appendMedium(0x3FFFFFFF), Buffer.buffer().appendMediumLE(0x3FFFFFFF));
        this.checkBEAndLE(4, Buffer.buffer().appendInt(Integer.MAX_VALUE), Buffer.buffer().appendIntLE(Integer.MAX_VALUE));
        this.checkBEAndLE(4, Buffer.buffer().appendUnsignedInt(Integer.MAX_VALUE), Buffer.buffer().appendUnsignedIntLE(Integer.MAX_VALUE));
        this.checkBEAndLE(8, Buffer.buffer().appendLong(Long.MAX_VALUE), Buffer.buffer().appendLongLE(Long.MAX_VALUE));
    }

    private void checkBEAndLE(int size, Buffer big, Buffer little) {
        for (int i = 0; i < size; ++i) {
            byte bigByte = big.getByte(i);
            byte littleByte = little.getByte(size - 1 - i);
            Assert.assertEquals((long)bigByte, (long)littleByte);
        }
    }

    @Test
    public void testGetOutOfBounds() throws Exception {
        int bytesLen = 100;
        byte[] bytes = TestUtils.randomByteArray(bytesLen);
        Buffer b = Buffer.buffer((byte[])bytes);
        TestUtils.assertIndexOutOfBoundsException(() -> b.getByte(bytesLen));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getByte(bytesLen + 1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getByte(bytesLen + 100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getByte(-1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getByte(-100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getInt(bytesLen));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getInt(bytesLen + 1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getInt(bytesLen + 100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getInt(-1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getInt(-100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getLong(bytesLen));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getLong(bytesLen + 1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getLong(bytesLen + 100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getLong(-1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getLong(-100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getFloat(bytesLen));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getFloat(bytesLen + 1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getFloat(bytesLen + 100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getFloat(-1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getFloat(-100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getDouble(bytesLen));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getDouble(bytesLen + 1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getDouble(bytesLen + 100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getDouble(-1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getDouble(-100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getShort(bytesLen));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getShort(bytesLen + 1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getShort(bytesLen + 100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getShort(-1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getShort(-100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getBytes(bytesLen + 1, bytesLen + 1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getBytes(bytesLen + 100, bytesLen + 100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getBytes(-1, -1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getBytes(-100, -100));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getString(-1, bytesLen));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getString(0, bytesLen + 1));
        TestUtils.assertIllegalArgumentException(() -> b.getString(2, 1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getString(-1, bytesLen, "UTF-8"));
        TestUtils.assertIndexOutOfBoundsException(() -> b.getString(0, bytesLen + 1, "UTF-8"));
        TestUtils.assertIllegalArgumentException(() -> b.getString(2, 1, "UTF-8"));
    }

    @Test
    public void testSetOutOfBounds() throws Exception {
        Buffer b = Buffer.buffer((int)100);
        TestUtils.assertIndexOutOfBoundsException(() -> b.setByte(-1, (byte)0));
        TestUtils.assertIndexOutOfBoundsException(() -> b.setInt(-1, 0));
        TestUtils.assertIndexOutOfBoundsException(() -> b.setLong(-1, 0L));
        TestUtils.assertIndexOutOfBoundsException(() -> b.setDouble(-1, 0.0));
        TestUtils.assertIndexOutOfBoundsException(() -> b.setFloat(-1, 0.0f));
        TestUtils.assertIndexOutOfBoundsException(() -> b.setShort(-1, (short)0));
        TestUtils.assertIndexOutOfBoundsException(() -> b.setBuffer(-1, b));
        TestUtils.assertIndexOutOfBoundsException(() -> b.setBuffer(0, b, -1, 0));
        TestUtils.assertIndexOutOfBoundsException(() -> b.setBuffer(0, b, 0, -1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.setBytes(-1, TestUtils.randomByteArray(1)));
        TestUtils.assertIndexOutOfBoundsException(() -> b.setBytes(-1, TestUtils.randomByteArray(1), -1, 0));
        TestUtils.assertIndexOutOfBoundsException(() -> b.setBytes(-1, TestUtils.randomByteArray(1), 0, -1));
        TestUtils.assertIndexOutOfBoundsException(() -> b.setString(-1, ""));
        TestUtils.assertIndexOutOfBoundsException(() -> b.setString(-1, "", "UTF-8"));
    }

    @Test
    public void testGetByte() throws Exception {
        int bytesLen = 100;
        byte[] bytes = TestUtils.randomByteArray(bytesLen);
        Buffer b = Buffer.buffer((byte[])bytes);
        for (int i = 0; i < bytesLen; ++i) {
            Assert.assertEquals((long)bytes[i], (long)b.getByte(i));
        }
    }

    @Test
    public void testGetUnsignedByte() throws Exception {
        int bytesLen = 100;
        byte[] bytes = TestUtils.randomByteArray(bytesLen);
        Buffer b = Buffer.buffer((byte[])bytes);
        for (int i = 0; i < bytesLen; ++i) {
            Assert.assertEquals((long)Byte.toUnsignedLong(bytes[i]), (long)b.getUnsignedByte(i));
        }
    }

    private void testGetSetInt(boolean isLE) throws Exception {
        int i;
        int numInts = 100;
        Buffer b = Buffer.buffer((int)(numInts * 4));
        for (i = 0; i < numInts; ++i) {
            if (isLE) {
                b.setIntLE(i * 4, i);
                continue;
            }
            b.setInt(i * 4, i);
        }
        for (i = 0; i < numInts; ++i) {
            if (isLE) {
                Assert.assertEquals((long)i, (long)b.getIntLE(i * 4));
                continue;
            }
            Assert.assertEquals((long)i, (long)b.getInt(i * 4));
        }
    }

    @Test
    public void testGetInt() throws Exception {
        this.testGetSetInt(false);
    }

    @Test
    public void testGetIntLE() throws Exception {
        this.testGetSetInt(true);
    }

    private void testGetSetUnsignedInt(boolean isLE) throws Exception {
        int i;
        int numInts = 100;
        Buffer b = Buffer.buffer((int)(numInts * 4));
        for (i = 0; i < numInts; ++i) {
            if (isLE) {
                b.setUnsignedIntLE(i * 4, (long)((int)(Integer.MAX_VALUE + (long)i)));
                continue;
            }
            b.setUnsignedInt(i * 4, (long)((int)(Integer.MAX_VALUE + (long)i)));
        }
        for (i = 0; i < numInts; ++i) {
            if (isLE) {
                Assert.assertEquals((long)Integer.toUnsignedLong(Integer.MAX_VALUE + i), (long)b.getUnsignedIntLE(i * 4));
                continue;
            }
            Assert.assertEquals((long)Integer.toUnsignedLong(Integer.MAX_VALUE + i), (long)b.getUnsignedInt(i * 4));
        }
    }

    @Test
    public void testGetUnsignedInt() throws Exception {
        this.testGetSetUnsignedInt(false);
    }

    @Test
    public void testGetUnsignedIntLE() throws Exception {
        this.testGetSetUnsignedInt(true);
    }

    private void testGetSetMedium(boolean isLE) throws Exception {
        int i;
        int numInts = 100;
        Buffer b = Buffer.buffer((int)(numInts * 3));
        for (i = 0; i < numInts; ++i) {
            if (isLE) {
                b.setMediumLE(i * 3, 0x1000000 + i);
                continue;
            }
            b.setMedium(i * 3, 0x1000000 + i);
        }
        for (i = 0; i < numInts; ++i) {
            if (isLE) {
                Assert.assertEquals((long)(0x1000000 + i - 0x1000000), (long)b.getMediumLE(i * 3));
                continue;
            }
            Assert.assertEquals((long)(0x1000000 + i - 0x1000000), (long)b.getMedium(i * 3));
        }
    }

    @Test
    public void testGetMedium() throws Exception {
        this.testGetSetMedium(false);
    }

    @Test
    public void testGetMediumLE() throws Exception {
        this.testGetSetMedium(true);
    }

    private void testGetSetUnsignedMedium(boolean isLE) throws Exception {
        int i;
        int numInts = 100;
        int MEDIUM_MAX_VALUE = 0x1000000 - numInts;
        Buffer b = Buffer.buffer((int)(numInts * 3));
        for (i = 0; i < numInts; ++i) {
            if (isLE) {
                b.setMediumLE(i * 3, MEDIUM_MAX_VALUE + i);
                continue;
            }
            b.setMedium(i * 3, MEDIUM_MAX_VALUE + i);
        }
        for (i = 0; i < numInts; ++i) {
            if (isLE) {
                Assert.assertEquals((long)Integer.toUnsignedLong(MEDIUM_MAX_VALUE + i), (long)b.getUnsignedMediumLE(i * 3));
                continue;
            }
            Assert.assertEquals((long)Integer.toUnsignedLong(MEDIUM_MAX_VALUE + i), (long)b.getUnsignedMedium(i * 3));
        }
    }

    @Test
    public void testGetUnsignedMedium() throws Exception {
        this.testGetSetUnsignedMedium(false);
    }

    @Test
    public void testGetUnsignedMediumLE() throws Exception {
        this.testGetSetUnsignedMedium(true);
    }

    private void testGetSetLong(boolean isLE) throws Exception {
        int i;
        int numLongs = 100;
        Buffer b = Buffer.buffer((int)(numLongs * 8));
        for (i = 0; i < numLongs; ++i) {
            if (isLE) {
                b.setLongLE(i * 8, (long)i);
                continue;
            }
            b.setLong(i * 8, (long)i);
        }
        for (i = 0; i < numLongs; ++i) {
            if (isLE) {
                Assert.assertEquals((long)i, (long)b.getLongLE(i * 8));
                continue;
            }
            Assert.assertEquals((long)i, (long)b.getLong(i * 8));
        }
    }

    @Test
    public void testGetLong() throws Exception {
        this.testGetSetLong(false);
    }

    @Test
    public void testGetLongLE() throws Exception {
        this.testGetSetLong(true);
    }

    @Test
    public void testGetFloat() throws Exception {
        int i;
        int numFloats = 100;
        Buffer b = Buffer.buffer((int)(numFloats * 4));
        for (i = 0; i < numFloats; ++i) {
            b.setFloat(i * 4, (float)i);
        }
        for (i = 0; i < numFloats; ++i) {
            Assert.assertEquals((float)i, (float)b.getFloat(i * 4), (float)0.0f);
        }
    }

    @Test
    public void testGetDouble() throws Exception {
        int i;
        int numDoubles = 100;
        Buffer b = Buffer.buffer((int)(numDoubles * 8));
        for (i = 0; i < numDoubles; ++i) {
            b.setDouble(i * 8, (double)i);
        }
        for (i = 0; i < numDoubles; ++i) {
            Assert.assertEquals((double)i, (double)b.getDouble(i * 8), (double)0.0);
        }
    }

    private void testGetSetShort(boolean isLE) throws Exception {
        short i;
        short numShorts = 100;
        Buffer b = Buffer.buffer((int)(numShorts * 2));
        for (i = 0; i < numShorts; i = (short)(i + 1)) {
            if (isLE) {
                b.setShortLE(i * 2, i);
                continue;
            }
            b.setShort(i * 2, i);
        }
        for (i = 0; i < numShorts; i = (short)(i + 1)) {
            if (isLE) {
                Assert.assertEquals((long)i, (long)b.getShortLE(i * 2));
                continue;
            }
            Assert.assertEquals((long)i, (long)b.getShort(i * 2));
        }
    }

    @Test
    public void testGetShort() throws Exception {
        this.testGetSetShort(false);
    }

    @Test
    public void testGetShortLE() throws Exception {
        this.testGetSetShort(true);
    }

    private void testGetSetUnsignedShort(boolean isLE) throws Exception {
        int i;
        int numShorts = 100;
        Buffer b = Buffer.buffer((int)(numShorts * 2));
        for (i = 0; i < numShorts; i = (int)((short)(i + 1))) {
            if (isLE) {
                b.setUnsignedShortLE(i * 2, (int)((short)(Short.MAX_VALUE + i)));
                continue;
            }
            b.setUnsignedShort(i * 2, (int)((short)(Short.MAX_VALUE + i)));
        }
        for (i = 0; i < numShorts; i = (int)((short)(i + 1))) {
            if (isLE) {
                Assert.assertEquals((long)Integer.toUnsignedLong(Short.MAX_VALUE + i), (long)b.getUnsignedShortLE(i * 2));
                continue;
            }
            Assert.assertEquals((long)Integer.toUnsignedLong(Short.MAX_VALUE + i), (long)b.getUnsignedShort(i * 2));
        }
    }

    @Test
    public void testGetUnsignedShort() throws Exception {
        this.testGetSetUnsignedShort(false);
    }

    @Test
    public void testGetUnsignedShortLE() throws Exception {
        this.testGetSetUnsignedShort(true);
    }

    @Test
    public void testGetString() throws Exception {
        String str = TestUtils.randomAlphaString(100);
        Buffer b = Buffer.buffer((String)str, (String)"UTF-8");
        Assert.assertEquals((long)100L, (long)b.length());
        String substr = b.getString(10, 20);
        Assert.assertEquals((Object)str.substring(10, 20), (Object)substr);
        substr = b.getString(10, 20, "UTF-8");
        Assert.assertEquals((Object)str.substring(10, 20), (Object)substr);
    }

    @Test
    public void testGetBytes() throws Exception {
        byte[] bytes = TestUtils.randomByteArray(100);
        Buffer b = Buffer.buffer((byte[])bytes);
        Assert.assertTrue((boolean)TestUtils.byteArraysEqual(bytes, b.getBytes()));
    }

    @Test
    public void testGetBytes2() throws Exception {
        byte[] bytes = TestUtils.randomByteArray(100);
        Buffer b = Buffer.buffer((byte[])bytes);
        byte[] sub = new byte[bytes.length / 2];
        System.arraycopy(bytes, bytes.length / 4, sub, 0, bytes.length / 2);
        Assert.assertTrue((boolean)TestUtils.byteArraysEqual(sub, b.getBytes(bytes.length / 4, bytes.length / 4 + bytes.length / 2)));
    }

    @Test
    public void testGetBytesWithByteArray() throws Exception {
        byte[] bytes = TestUtils.randomByteArray(100);
        Buffer b = Buffer.buffer((byte[])bytes);
        byte[] sub = new byte[bytes.length / 2];
        System.arraycopy(bytes, bytes.length / 4, sub, 0, bytes.length / 2);
        byte[] result = new byte[bytes.length / 2];
        b.getBytes(bytes.length / 4, bytes.length / 4 + bytes.length / 2, result);
        Assert.assertTrue((boolean)TestUtils.byteArraysEqual(sub, result));
    }

    @Test(expected=IndexOutOfBoundsException.class)
    public void testGetBytesWithTooSmallByteArray() throws Exception {
        byte[] bytes = TestUtils.randomByteArray(100);
        Buffer b = Buffer.buffer((byte[])bytes);
        byte[] result = new byte[bytes.length / 4];
        b.getBytes(bytes.length / 4, bytes.length / 4 + bytes.length / 2, result);
    }

    @Test
    public void testGetBytesWithByteArrayFull() throws Exception {
        byte[] bytes = TestUtils.randomByteArray(100);
        Buffer b = Buffer.buffer((byte[])bytes);
        byte[] sub = new byte[bytes.length];
        System.arraycopy(bytes, bytes.length / 4, sub, 12, bytes.length / 2);
        byte[] result = new byte[bytes.length];
        b.getBytes(bytes.length / 4, bytes.length / 4 + bytes.length / 2, result, 12);
        Assert.assertTrue((boolean)TestUtils.byteArraysEqual(sub, result));
    }

    @Test(expected=IndexOutOfBoundsException.class)
    public void testGetBytesWithBadOffset() throws Exception {
        byte[] bytes = TestUtils.randomByteArray(100);
        Buffer b = Buffer.buffer((byte[])bytes);
        byte[] result = new byte[bytes.length / 2];
        b.getBytes(bytes.length / 4, bytes.length / 4 + bytes.length / 2, result, -1);
    }

    @Test
    public void testSetInt() throws Exception {
        this.testSetInt(Buffer.buffer((int)400));
    }

    @Test
    public void testSetIntExpandBuffer() throws Exception {
        this.testSetInt(Buffer.buffer());
    }

    private void testSetInt(Buffer buff) throws Exception {
        int i;
        for (i = 0; i < 100; ++i) {
            buff.setInt(i * 4, i);
        }
        for (i = 0; i < 100; ++i) {
            Assert.assertEquals((long)i, (long)buff.getInt(i * 4));
        }
    }

    @Test
    public void testSetUnsignedInt() throws Exception {
        this.testSetUnsignedInt(Buffer.buffer((int)400));
    }

    @Test
    public void testSetUnsignedIntExpandBuffer() throws Exception {
        this.testSetUnsignedInt(Buffer.buffer());
    }

    private void testSetUnsignedInt(Buffer buff) throws Exception {
        long val;
        int i;
        for (i = 0; i < 100; ++i) {
            val = Integer.toUnsignedLong(Integer.MAX_VALUE + i);
            buff.setUnsignedInt(i * 4, val);
        }
        for (i = 0; i < 100; ++i) {
            val = Integer.toUnsignedLong(Integer.MAX_VALUE + i);
            Assert.assertEquals((long)val, (long)buff.getUnsignedInt(i * 4));
        }
    }

    @Test
    public void testSetLong() throws Exception {
        this.testSetLong(Buffer.buffer((int)800));
    }

    @Test
    public void testSetLongExpandBuffer() throws Exception {
        this.testSetLong(Buffer.buffer());
    }

    private void testSetLong(Buffer buff) throws Exception {
        int i;
        for (i = 0; i < 100; ++i) {
            buff.setLong(i * 8, (long)i);
        }
        for (i = 0; i < 100; ++i) {
            Assert.assertEquals((long)i, (long)buff.getLong(i * 8));
        }
    }

    @Test
    public void testSetByte() throws Exception {
        this.testSetByte(Buffer.buffer((int)100));
    }

    @Test
    public void testSetByteExpandBuffer() throws Exception {
        this.testSetByte(Buffer.buffer());
    }

    private void testSetByte(Buffer buff) throws Exception {
        int i;
        for (i = 0; i < 100; ++i) {
            buff.setByte(i, (byte)i);
        }
        for (i = 0; i < 100; ++i) {
            Assert.assertEquals((long)i, (long)buff.getByte(i));
        }
    }

    @Test
    public void testSetUnsignedByte() throws Exception {
        this.testSetUnsignedByte(Buffer.buffer((int)100));
    }

    @Test
    public void testSetUnsignedByteExpandBuffer() throws Exception {
        this.testSetUnsignedByte(Buffer.buffer());
    }

    private void testSetUnsignedByte(Buffer buff) throws Exception {
        int val;
        int i;
        for (i = 0; i < 100; ++i) {
            val = 127 + i;
            buff.setUnsignedByte(i, (short)val);
        }
        for (i = 0; i < 100; ++i) {
            val = 127 + i;
            Assert.assertEquals((long)val, (long)buff.getUnsignedByte(i));
        }
    }

    @Test
    public void testSetFloat() throws Exception {
        this.testSetFloat(Buffer.buffer((int)400));
    }

    @Test
    public void testSetFloatExpandBuffer() throws Exception {
        this.testSetFloat(Buffer.buffer());
    }

    private void testSetFloat(Buffer buff) throws Exception {
        int i;
        for (i = 0; i < 100; ++i) {
            buff.setFloat(i * 4, (float)i);
        }
        for (i = 0; i < 100; ++i) {
            Assert.assertEquals((float)i, (float)buff.getFloat(i * 4), (float)0.0f);
        }
    }

    @Test
    public void testSetDouble() throws Exception {
        this.testSetDouble(Buffer.buffer((int)800));
    }

    @Test
    public void testSetDoubleExpandBuffer() throws Exception {
        this.testSetDouble(Buffer.buffer());
    }

    private void testSetDouble(Buffer buff) throws Exception {
        int i;
        for (i = 0; i < 100; ++i) {
            buff.setDouble(i * 8, (double)i);
        }
        for (i = 0; i < 100; ++i) {
            Assert.assertEquals((double)i, (double)buff.getDouble(i * 8), (double)0.0);
        }
    }

    @Test
    public void testSetShort() throws Exception {
        this.testSetShort(Buffer.buffer((int)200));
    }

    @Test
    public void testSetShortExpandBuffer() throws Exception {
        this.testSetShort(Buffer.buffer());
    }

    private void testSetShort(Buffer buff) throws Exception {
        int i;
        for (i = 0; i < 100; ++i) {
            buff.setShort(i * 2, (short)i);
        }
        for (i = 0; i < 100; ++i) {
            Assert.assertEquals((long)i, (long)buff.getShort(i * 2));
        }
    }

    @Test
    public void testSetUnsignedShort() throws Exception {
        this.testSetUnsignedShort(Buffer.buffer((int)200));
    }

    @Test
    public void testSetUnsignedShortExpandBuffer() throws Exception {
        this.testSetUnsignedShort(Buffer.buffer());
    }

    private void testSetUnsignedShort(Buffer buff) throws Exception {
        int val;
        int i;
        for (i = 0; i < 100; ++i) {
            val = Short.MAX_VALUE + i;
            buff.setUnsignedShort(i * 2, val);
        }
        for (i = 0; i < 100; ++i) {
            val = Short.MAX_VALUE + i;
            Assert.assertEquals((long)val, (long)buff.getUnsignedShort(i * 2));
        }
    }

    @Test
    public void testSetBytesBuffer() throws Exception {
        this.testSetBytesBuffer(Buffer.buffer((int)150));
        TestUtils.assertNullPointerException(() -> Buffer.buffer((int)150).setBytes(0, (ByteBuffer)null));
    }

    @Test
    public void testSetBytesBufferExpandBuffer() throws Exception {
        this.testSetShort(Buffer.buffer());
    }

    private void testSetBytesBuffer(Buffer buff) throws Exception {
        Buffer b = TestUtils.randomBuffer(100);
        buff.setBuffer(50, b);
        byte[] b2 = buff.getBytes(50, 150);
        Assert.assertEquals((Object)b, (Object)Buffer.buffer((byte[])b2));
        byte[] b3 = TestUtils.randomByteArray(100);
        buff.setBytes(50, b3);
        byte[] b4 = buff.getBytes(50, 150);
        Assert.assertEquals((Object)Buffer.buffer((byte[])b3), (Object)Buffer.buffer((byte[])b4));
    }

    @Test
    public void testSetBytesWithOffsetAndLen() throws Exception {
        int bytesLen = 100;
        byte[] bytes = TestUtils.randomByteArray(bytesLen);
        int len = bytesLen - 2;
        Buffer b = Buffer.buffer();
        b.setByte(0, (byte)48);
        b.setBytes(1, bytes, 1, len);
        Assert.assertEquals((long)b.length(), (long)(len + 1));
        byte[] copy = new byte[len];
        System.arraycopy(bytes, 1, copy, 0, len);
        Assert.assertTrue((boolean)TestUtils.byteArraysEqual(copy, b.getBytes(1, b.length())));
        b.setBytes(b.length(), bytes, 1, len);
        Assert.assertEquals((long)b.length(), (long)(2 * len + 1));
        TestUtils.assertNullPointerException(() -> Buffer.buffer((int)150).setBytes(0, (byte[])null));
        TestUtils.assertNullPointerException(() -> Buffer.buffer((int)150).setBytes(0, null, 1, len));
    }

    @Test
    public void testSetBufferWithOffsetAndLen() throws Exception {
        int bytesLen = 100;
        byte[] bytes = TestUtils.randomByteArray(bytesLen);
        Buffer src = Buffer.buffer((byte[])bytes);
        int len = bytesLen - 2;
        Buffer b = Buffer.buffer();
        b.setByte(0, (byte)48);
        b.setBuffer(1, src, 1, len);
        Assert.assertEquals((long)b.length(), (long)(len + 1));
        byte[] copy = new byte[len];
        System.arraycopy(bytes, 1, copy, 0, len);
        Assert.assertTrue((boolean)TestUtils.byteArraysEqual(copy, b.getBytes(1, b.length())));
        b.setBuffer(b.length(), src, 1, len);
        Assert.assertEquals((long)b.length(), (long)(2 * len + 1));
        TestUtils.assertNullPointerException(() -> b.setBuffer(1, null));
        TestUtils.assertNullPointerException(() -> b.setBuffer(1, null, 0, len));
    }

    @Test
    public void testSetBytesString() throws Exception {
        this.testSetBytesString(Buffer.buffer((int)150));
    }

    @Test
    public void testSetBytesStringExpandBuffer() throws Exception {
        this.testSetBytesString(Buffer.buffer());
    }

    private void testSetBytesString(Buffer buff) throws Exception {
        String str = TestUtils.randomUnicodeString(100);
        buff.setString(50, str);
        byte[] b1 = buff.getBytes(50, buff.length());
        String str2 = new String(b1, "UTF-8");
        Assert.assertEquals((Object)str, (Object)str2);
        TestUtils.assertNullPointerException(() -> Buffer.buffer((int)150).setString(0, null));
        TestUtils.assertNullPointerException(() -> Buffer.buffer((int)150).setString(0, null, "UTF-8"));
    }

    @Test
    public void testToString() throws Exception {
        String str = TestUtils.randomUnicodeString(100);
        Buffer buff = Buffer.buffer((String)str);
        Assert.assertEquals((Object)str, (Object)buff.toString());
    }

    @Test
    public void testCopy() throws Exception {
        Buffer buff = TestUtils.randomBuffer(100);
        Assert.assertEquals((Object)buff, (Object)buff.copy());
        Buffer copy = buff.getBuffer(0, buff.length());
        Assert.assertEquals((Object)buff, (Object)copy);
        Buffer copy2 = buff.copy();
        buff.setInt(0, 1);
        Assert.assertEquals((Object)copy, (Object)copy2);
    }

    @Test
    public void testCreateBuffers() throws Exception {
        Buffer buff = Buffer.buffer((int)1000);
        Assert.assertEquals((long)0L, (long)buff.length());
        String str = TestUtils.randomUnicodeString(100);
        buff = Buffer.buffer((String)str);
        Assert.assertEquals((long)buff.length(), (long)str.getBytes("UTF-8").length);
        Assert.assertEquals((Object)str, (Object)buff.toString());
        byte[] bytes = TestUtils.randomByteArray(100);
        buff = Buffer.buffer((byte[])bytes);
        Assert.assertEquals((long)buff.length(), (long)bytes.length);
        Assert.assertEquals((Object)Buffer.buffer((byte[])bytes), (Object)Buffer.buffer((byte[])buff.getBytes()));
    }

    @Test
    public void testSlice1() throws Exception {
        Buffer buff = TestUtils.randomBuffer(100);
        Buffer sliced = buff.slice();
        Assert.assertEquals((Object)buff, (Object)sliced);
        long rand = TestUtils.randomLong();
        sliced.setLong(0, rand);
        Assert.assertEquals((long)rand, (long)buff.getLong(0));
        buff.appendString(TestUtils.randomUnicodeString(100));
        Assert.assertEquals((long)100L, (long)sliced.length());
    }

    @Test
    public void testSlice2() throws Exception {
        Buffer buff = TestUtils.randomBuffer(100);
        Buffer sliced = buff.slice(10, 20);
        for (int i = 0; i < 10; ++i) {
            Assert.assertEquals((long)buff.getByte(10 + i), (long)sliced.getByte(i));
        }
        long rand = TestUtils.randomLong();
        sliced.setLong(0, rand);
        Assert.assertEquals((long)rand, (long)buff.getLong(10));
        buff.appendString(TestUtils.randomUnicodeString(100));
        Assert.assertEquals((long)10L, (long)sliced.length());
    }

    @Test
    public void testToJsonObject() throws Exception {
        JsonObject obj = new JsonObject();
        obj.put("wibble", "wibble_value");
        obj.put("foo", Integer.valueOf(5));
        obj.put("bar", Boolean.valueOf(true));
        Buffer buff = Buffer.buffer((String)obj.encode());
        Assert.assertEquals((Object)obj, (Object)buff.toJsonObject());
        buff = Buffer.buffer((String)TestUtils.randomAlphaString(10));
        try {
            buff.toJsonObject();
            Assert.fail();
        }
        catch (DecodeException decodeException) {
            // empty catch block
        }
    }

    @Test
    public void testToJsonArray() throws Exception {
        JsonArray arr = new JsonArray();
        arr.add("wibble");
        arr.add(Integer.valueOf(5));
        arr.add(Boolean.valueOf(true));
        Buffer buff = Buffer.buffer((String)arr.encode());
        Assert.assertEquals((Object)arr, (Object)buff.toJsonArray());
        buff = Buffer.buffer((String)TestUtils.randomAlphaString(10));
        try {
            buff.toJsonObject();
            Assert.fail();
        }
        catch (DecodeException decodeException) {
            // empty catch block
        }
    }
}

