/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.test.core;

import io.netty.channel.socket.SocketChannel;
import io.vertx.core.Context;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.Http2Settings;
import io.vertx.core.http.HttpClientOptions;
import io.vertx.core.http.HttpClientRequest;
import io.vertx.core.http.HttpServerOptions;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.http.HttpServerResponse;
import io.vertx.core.http.HttpVersion;
import io.vertx.core.http.StreamResetException;
import io.vertx.core.http.impl.Http2ServerConnection;
import io.vertx.core.net.OpenSSLEngineOptions;
import io.vertx.core.net.PemKeyCertOptions;
import io.vertx.core.net.SSLEngineOptions;
import io.vertx.test.core.Http2TestBase;
import io.vertx.test.core.HttpTest;
import io.vertx.test.core.TestUtils;
import io.vertx.test.core.tls.Cert;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.junit.Test;

public class Http2Test
extends HttpTest {
    @Override
    public void setUp() throws Exception {
        super.setUp();
        this.client = this.vertx.createHttpClient(this.createBaseClientOptions());
        this.server = this.vertx.createHttpServer(this.createBaseServerOptions().setHandle100ContinueAutomatically(true));
    }

    @Override
    protected HttpServerOptions createBaseServerOptions() {
        return Http2TestBase.createHttp2ServerOptions(8080, "localhost");
    }

    @Override
    protected HttpClientOptions createBaseClientOptions() {
        return Http2TestBase.createHttp2ClientOptions();
    }

    @Override
    public void testCloseHandlerNotCalledWhenConnectionClosedAfterEnd() throws Exception {
        this.testCloseHandlerNotCalledWhenConnectionClosedAfterEnd(1);
    }

    @Test
    public void testServerResponseWriteBufferFromOtherThread() throws Exception {
        this.server.requestHandler(req -> this.runAsync(() -> req.response().write("hello ").end("world"))).listen(this.onSuccess(v -> this.client.get(8080, "localhost", "/somepath", resp -> {
            this.assertEquals(200L, resp.statusCode());
            resp.bodyHandler(buff -> {
                this.assertEquals(Buffer.buffer((String)"hello world"), buff);
                this.testComplete();
            });
        }).exceptionHandler(this::fail).end()));
        this.await();
    }

    @Test
    public void testServerResponseResetFromOtherThread() throws Exception {
        this.server.requestHandler(req -> this.runAsync(() -> req.response().reset(0L))).listen(this.onSuccess(v -> this.client.get(8080, "localhost", "/somepath", resp -> this.fail()).exceptionHandler(err -> {
            this.assertTrue(err instanceof StreamResetException);
            this.testComplete();
        }).sendHead()));
        this.await();
    }

    void runAsync(Runnable runnable) {
        new Thread(() -> {
            try {
                runnable.run();
            }
            catch (Exception e) {
                this.fail(e);
            }
        }).start();
    }

    @Test
    public void testClientRequestWriteFromOtherThread() throws Exception {
        CountDownLatch latch1 = new CountDownLatch(1);
        CountDownLatch latch2 = new CountDownLatch(1);
        this.server.requestHandler(req -> {
            latch2.countDown();
            req.endHandler(v -> req.response().end());
        }).listen(this.onSuccess(v -> latch1.countDown()));
        this.awaitLatch(latch1);
        HttpClientRequest req2 = this.client.get(8080, "localhost", "/somepath", resp -> {
            this.assertEquals(200L, resp.statusCode());
            this.testComplete();
        }).setChunked(true).sendHead();
        this.awaitLatch(latch2);
        req2.write("hello ").end("world");
        this.await();
    }

    @Test
    public void testServerOpenSSL() throws Exception {
        HttpServerOptions opts = new HttpServerOptions().setPort(4043).setHost("localhost").setUseAlpn(true).setSsl(true).addEnabledCipherSuite("TLS_RSA_WITH_AES_128_CBC_SHA").setPemKeyCertOptions((PemKeyCertOptions)Cert.SERVER_PEM.get()).setSslEngineOptions((SSLEngineOptions)new OpenSSLEngineOptions());
        this.server.close();
        this.client.close();
        this.client = this.vertx.createHttpClient(this.createBaseClientOptions());
        this.server = this.vertx.createHttpServer(opts);
        this.server.requestHandler(req -> req.response().end());
        CountDownLatch latch = new CountDownLatch(1);
        System.out.println("starting");
        try {
            this.server.listen(this.onSuccess(v -> latch.countDown()));
        }
        catch (Throwable e) {
            e.printStackTrace();
        }
        System.out.println("listening");
        this.awaitLatch(latch);
        this.client.get(4043, "localhost", "/somepath", resp -> {
            this.assertEquals(200L, resp.statusCode());
            this.testComplete();
        }).exceptionHandler(this::fail).end();
        this.await();
    }

    @Test
    public void testServerStreamPausedWhenConnectionIsPaused() throws Exception {
        CountDownLatch fullLatch = new CountDownLatch(1);
        CompletableFuture resumeLatch = new CompletableFuture();
        this.server.requestHandler(req -> {
            HttpServerResponse resp = req.response();
            switch (req.path()) {
                case "/0": {
                    this.vertx.setPeriodic(1L, timerID -> {
                        if (resp.writeQueueFull()) {
                            this.vertx.cancelTimer(timerID.longValue());
                            fullLatch.countDown();
                        } else {
                            resp.write(Buffer.buffer((String)TestUtils.randomAlphaString(512)));
                        }
                    });
                    break;
                }
                case "/1": {
                    this.assertTrue(resp.writeQueueFull());
                    resp.drainHandler(v -> resp.end());
                    resumeLatch.complete(null);
                }
            }
        });
        this.startServer();
        this.client.getNow(8080, "localhost", "/0", resp -> {
            resp.pause();
            Context ctx = this.vertx.getOrCreateContext();
            resumeLatch.thenAccept(v1 -> ctx.runOnContext(v2 -> {
                resp.endHandler(v -> this.testComplete());
                resp.resume();
            }));
        });
        this.awaitLatch(fullLatch);
        this.client.getNow(8080, "localhost", "/1", resp -> resp.endHandler(v -> this.complete()));
        resumeLatch.get(20L, TimeUnit.SECONDS);
        this.await();
    }

    @Test
    public void testClientStreamPausedWhenConnectionIsPaused() throws Exception {
        this.waitFor(2);
        Buffer buffer = TestUtils.randomBuffer(512);
        CompletableFuture resumeLatch = new CompletableFuture();
        this.server.requestHandler(req -> {
            switch (req.path()) {
                case "/0": {
                    req.pause();
                    resumeLatch.thenAccept(v -> req.resume());
                    req.endHandler(v -> req.response().end());
                    break;
                }
                case "/1": {
                    req.bodyHandler(v -> {
                        this.assertEquals(v, buffer);
                        req.response().end();
                    });
                }
            }
        });
        this.startServer();
        HttpClientRequest req1 = this.client.get(8080, "localhost", "/0", resp -> this.complete()).setChunked(true);
        while (!req1.writeQueueFull()) {
            req1.write(Buffer.buffer((String)TestUtils.randomAlphaString(512)));
            Thread.sleep(1L);
        }
        HttpClientRequest req2 = this.client.get(8080, "localhost", "/1", resp -> this.complete()).setChunked(true);
        this.assertFalse(req2.writeQueueFull());
        req2.sendHead(v -> {
            this.assertTrue(req2.writeQueueFull());
            resumeLatch.complete(null);
        });
        resumeLatch.get(20L, TimeUnit.SECONDS);
        this.assertWaitUntil(() -> !req2.writeQueueFull());
        req1.end();
        req2.end(buffer);
        this.await();
    }

    @Test
    public void testResetClientRequestNotYetSent() throws Exception {
        this.waitFor(2);
        this.server.close();
        this.server = this.vertx.createHttpServer(this.createBaseServerOptions().setInitialSettings(new Http2Settings().setMaxConcurrentStreams(1L)));
        AtomicInteger numReq = new AtomicInteger();
        this.server.requestHandler(req -> {
            this.assertEquals(0L, numReq.getAndIncrement());
            req.response().end();
            this.complete();
        });
        this.startServer();
        HttpClientRequest post = this.client.post(8080, "localhost", "some-uri", resp -> this.fail());
        post.setChunked(true).write(TestUtils.randomBuffer(1024));
        this.assertTrue(post.reset());
        this.client.getNow(8080, "localhost", "some-uri", resp -> {
            this.assertEquals(1L, numReq.get());
            this.complete();
        });
        this.await();
    }

    @Test
    public void testDiscardConnectionWhenChannelBecomesInactive() throws Exception {
        AtomicInteger count = new AtomicInteger();
        this.server.requestHandler(req -> {
            if (count.getAndIncrement() == 0) {
                Http2ServerConnection a = (Http2ServerConnection)req.connection();
                SocketChannel channel = (SocketChannel)a.channel();
                channel.shutdown();
            } else {
                req.response().end();
            }
        });
        this.startServer();
        AtomicBoolean closed = new AtomicBoolean();
        this.client.get(8080, "localhost", "some-uri", resp -> this.fail()).connectionHandler(conn -> conn.closeHandler(v -> closed.set(true))).end();
        this.assertWaitUntil(closed::get);
        this.client.get(8080, "localhost", "some-uri", resp -> this.testComplete()).exceptionHandler(err -> this.fail()).end();
        this.await();
    }

    @Test
    public void testClientDoesNotSupportAlpn() throws Exception {
        this.waitFor(2);
        this.server.requestHandler(req -> {
            this.assertEquals(HttpVersion.HTTP_1_1, req.version());
            req.response().end();
            this.complete();
        });
        this.startServer();
        this.client.close();
        this.client = this.vertx.createHttpClient(this.createBaseClientOptions().setProtocolVersion(HttpVersion.HTTP_1_1).setUseAlpn(false));
        this.client.get(8080, "localhost", "some-uri", resp -> {
            this.assertEquals(HttpVersion.HTTP_1_1, resp.version());
            this.complete();
        }).exceptionHandler(this::fail).end();
        this.await();
    }

    @Test
    public void testServerDoesNotSupportAlpn() throws Exception {
        this.waitFor(2);
        this.server.close();
        this.server = this.vertx.createHttpServer(this.createBaseServerOptions().setUseAlpn(false));
        this.server.requestHandler(req -> {
            this.assertEquals(HttpVersion.HTTP_1_1, req.version());
            req.response().end();
            this.complete();
        });
        this.startServer();
        this.client.get(8080, "localhost", "some-uri", resp -> {
            this.assertEquals(HttpVersion.HTTP_1_1, resp.version());
            this.complete();
        }).exceptionHandler(this::fail).end();
        this.await();
    }

    @Test
    public void testClientMakeRequestHttp2WithSSLWithoutAlpn() throws Exception {
        this.client.close();
        this.client = this.vertx.createHttpClient(this.createBaseClientOptions().setUseAlpn(false));
        try {
            this.client.get(8080, "localhost", "some-uri");
            this.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test
    public void testServePendingRequests() throws Exception {
        int n = 10;
        this.waitFor(n);
        LinkedList requests = new LinkedList();
        HashSet connections = new HashSet();
        this.server.requestHandler(req -> {
            requests.add(req);
            connections.add(req.connection());
            this.assertEquals(1L, connections.size());
            if (requests.size() == n) {
                while (requests.size() > 0) {
                    ((HttpServerRequest)requests.removeFirst()).response().end();
                }
            }
        });
        this.startServer();
        for (int i = 0; i < n; ++i) {
            this.client.get(8080, "localhost", "some-uri", resp -> this.complete()).end();
        }
        this.await();
    }

    @Test
    public void testInitialMaxConcurrentStreamZero() throws Exception {
        AtomicLong concurrency = new AtomicLong();
        this.server.close();
        this.server = this.vertx.createHttpServer(this.createBaseServerOptions().setInitialSettings(new Http2Settings().setMaxConcurrentStreams(0L)));
        this.server.requestHandler(req -> {
            this.assertEquals(10L, concurrency.get());
            req.response().end();
        });
        this.server.connectionHandler(conn -> this.vertx.setTimer(500L, id -> conn.updateSettings(new Http2Settings().setMaxConcurrentStreams(10L))));
        this.startServer();
        this.client.get(8080, "localhost", "some-uri", resp -> this.testComplete()).connectionHandler(conn -> {
            this.assertEquals(0L, conn.remoteSettings().getMaxConcurrentStreams());
            conn.remoteSettingsHandler(settings -> concurrency.set(settings.getMaxConcurrentStreams()));
        }).setTimeout(10000L).exceptionHandler(err -> this.fail((Throwable)err)).end();
        this.await();
    }

    @Test
    public void testFoo() throws Exception {
        this.waitFor(2);
        this.server.requestHandler(req -> {
            HttpServerResponse resp = req.response();
            resp.write("Hello");
            resp.end("World");
            this.assertNull(resp.headers().get("content-length"));
            this.complete();
        });
        this.startServer();
        this.client.getNow(8080, "localhost", "some-uri", resp -> {
            this.assertNull(resp.getHeader("content-length"));
            resp.bodyHandler(body -> {
                this.assertEquals("HelloWorld", body.toString());
                this.complete();
            });
        });
        this.await();
    }
}

