/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.test.core.net;

import io.netty.channel.Channel;
import io.netty.channel.embedded.EmbeddedChannel;
import io.vertx.core.Vertx;
import io.vertx.core.http.impl.pool.ConnectionListener;
import io.vertx.core.http.impl.pool.ConnectionProvider;
import io.vertx.core.http.impl.pool.Pool;
import io.vertx.core.http.impl.pool.Waiter;
import io.vertx.core.impl.ContextImpl;
import io.vertx.core.impl.ContextInternal;
import io.vertx.test.core.VertxTestBase;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicReference;
import org.junit.Test;

public class ConnectionPoolTest
extends VertxTestBase {
    @Test
    public void testConnectSuccess() {
        FakeConnectionProvider connector = new FakeConnectionProvider();
        final FakeConnectionManager mgr = new FakeConnectionManager(3, 4, connector);
        final AtomicReference initLock = new AtomicReference();
        final AtomicReference handleLock = new AtomicReference();
        FakeWaiter waiter = new FakeWaiter(){

            @Override
            public synchronized void initConnection(ContextInternal ctx, FakeConnection conn) {
                ConnectionPoolTest.this.assertNull(Vertx.currentContext());
                ConnectionPoolTest.this.assertSame(ctx, this.context);
                Pool<FakeConnection> pool = mgr.pool();
                initLock.set(Thread.holdsLock(pool));
                super.initConnection(ctx, conn);
            }

            @Override
            public synchronized boolean handleConnection(ContextInternal ctx, FakeConnection conn) throws Exception {
                ConnectionPoolTest.this.assertNull(Vertx.currentContext());
                ConnectionPoolTest.this.assertSame(ctx, this.context);
                Pool<FakeConnection> pool = mgr.pool();
                handleLock.set(Thread.holdsLock(pool));
                return super.handleConnection(ctx, conn);
            }
        };
        mgr.getConnection(waiter);
        FakeConnection conn = connector.assertRequest();
        conn.connect();
        this.assertWaitUntil(waiter::isComplete);
        this.assertEquals(Boolean.FALSE, handleLock.get());
        this.assertEquals(Boolean.FALSE, initLock.get());
        this.assertWaitUntil(() -> waiter.isInitialized(conn));
        waiter.assertSuccess(conn);
        waiter.recycle();
        this.assertEquals(0L, mgr.size());
        this.assertTrue(mgr.closed());
    }

    @Test
    public void testConnectFailure() {
        FakeConnectionProvider connector = new FakeConnectionProvider();
        final FakeConnectionManager mgr = new FakeConnectionManager(3, 4, connector);
        final AtomicReference holdsLock = new AtomicReference();
        FakeWaiter waiter = new FakeWaiter(){

            @Override
            public synchronized void handleFailure(ContextInternal ctx, Throwable failure) {
                ConnectionPoolTest.this.assertNull(Vertx.currentContext());
                ConnectionPoolTest.this.assertSame(ctx, this.context);
                Pool<FakeConnection> pool = mgr.pool();
                holdsLock.set(Thread.holdsLock(pool));
                super.handleFailure(ctx, failure);
            }
        };
        mgr.getConnection(waiter);
        FakeConnection conn = connector.assertRequest();
        Throwable failure = new Throwable();
        conn.fail(failure);
        this.assertWaitUntil(waiter::isComplete);
        this.assertEquals(Boolean.FALSE, holdsLock.get());
        waiter.assertNotInitialized();
        waiter.assertFailure(failure);
    }

    @Test
    public void testConnectPoolEmptyWaiterCancelledAfterConnectRequest() {
        FakeConnectionProvider connector = new FakeConnectionProvider();
        FakeConnectionManager mgr = new FakeConnectionManager(3, 3, connector);
        FakeWaiter waiter = new FakeWaiter();
        mgr.getConnection(waiter);
        FakeConnection conn = connector.assertRequest();
        waiter.cancel();
        conn.connect();
        this.assertWaitUntil(() -> mgr.size() == 1);
        this.assertWaitUntil(() -> waiter.isInitialized(conn));
        this.assertWaitUntil(waiter::isComplete);
        this.assertFalse(waiter.isSuccess());
        this.assertFalse(waiter.isFailure());
        this.assertTrue(mgr.contains(conn));
    }

    @Test
    public void testConnectionFailure() {
        FakeConnectionProvider connector = new FakeConnectionProvider();
        FakeConnectionManager mgr = new FakeConnectionManager(3, 3, connector);
        FakeWaiter waiter = new FakeWaiter();
        mgr.getConnection(waiter);
        FakeConnection conn = connector.assertRequest();
        Exception expected = new Exception();
        conn.fail(expected);
        this.assertWaitUntil(waiter::isComplete);
        waiter.assertFailure(expected);
        this.assertTrue(waiter.isFailure());
        this.assertWaitUntil(mgr::closed);
    }

    @Test
    public void testRecycleConnection() {
        FakeConnectionProvider connector = new FakeConnectionProvider();
        FakeConnectionManager mgr = new FakeConnectionManager(3, 1, connector);
        FakeWaiter waiter1 = new FakeWaiter();
        mgr.getConnection(waiter1);
        FakeConnection conn = connector.assertRequest();
        conn.connect();
        this.assertWaitUntil(waiter1::isComplete);
        FakeWaiter waiter2 = new FakeWaiter();
        mgr.getConnection(waiter2);
        connector.assertRequests(0);
        waiter1.recycle();
        this.assertWaitUntil(waiter2::isComplete);
        waiter2.assertSuccess(conn);
    }

    @Test
    public void testRecycleDiscardedConnection() {
        FakeConnectionProvider connector = new FakeConnectionProvider();
        FakeConnectionManager mgr = new FakeConnectionManager(3, 1, connector);
        FakeWaiter waiter1 = new FakeWaiter();
        mgr.getConnection(waiter1);
        FakeConnection conn = connector.assertRequest();
        conn.connect();
        this.assertWaitUntil(waiter1::isComplete);
        FakeWaiter waiter2 = new FakeWaiter();
        mgr.getConnection(waiter2);
        conn.close();
        waiter1.recycle();
        this.assertWaitUntil(() -> connector.requests() == 1);
        this.assertFalse(mgr.closed());
        FakeConnection conn2 = connector.assertRequest();
        conn2.connect();
        this.assertWaitUntil(waiter2::isSuccess);
    }

    @Test
    public void testWaiterThrowsException() {
        FakeConnectionProvider connector = new FakeConnectionProvider();
        FakeConnectionManager mgr = new FakeConnectionManager(3, 1, connector);
        final Exception failure = new Exception();
        FakeWaiter waiter = new FakeWaiter(){

            @Override
            public synchronized boolean handleConnection(ContextInternal ctx, FakeConnection conn) throws Exception {
                throw failure;
            }
        };
        mgr.getConnection(waiter);
        FakeConnection conn = connector.assertRequest();
        conn.connect();
        this.assertEquals(0L, mgr.size());
    }

    @Test
    public void testEndpointLifecycle() {
        FakeConnectionProvider connector = new FakeConnectionProvider();
        FakeConnectionManager mgr = new FakeConnectionManager(3, 1, connector);
        FakeWaiter waiter1 = new FakeWaiter();
        mgr.getConnection(waiter1);
        FakeConnection conn = connector.assertRequest();
        conn.connect();
        this.assertWaitUntil(waiter1::isSuccess);
        conn.close();
        this.assertWaitUntil(mgr::closed);
        FakeWaiter waiter2 = new FakeWaiter();
        mgr.getConnection(waiter2);
        this.assertEquals(2L, mgr.sequence());
    }

    @Test
    public void testDontCloseEndpointWithInflightRequest() {
        FakeConnectionProvider connector = new FakeConnectionProvider();
        FakeConnectionManager mgr = new FakeConnectionManager(3, 2, connector);
        FakeWaiter waiter1 = new FakeWaiter();
        mgr.getConnection(waiter1);
        FakeConnection conn = connector.assertRequest();
        conn.connect();
        this.assertWaitUntil(waiter1::isComplete);
        FakeWaiter waiter2 = new FakeWaiter();
        mgr.getConnection(waiter2);
        conn.close();
        this.assertWaitUntil(() -> !mgr.contains(conn));
        this.assertFalse(mgr.closed());
    }

    @Test
    public void testInitialConcurrency() {
        int n = 10;
        FakeConnectionProvider connector = new FakeConnectionProvider();
        FakeConnectionManager mgr = new FakeConnectionManager(-1, 1, connector);
        ArrayList<FakeWaiter> waiters = new ArrayList<FakeWaiter>();
        for (int i = 0; i < n; ++i) {
            FakeWaiter waiter2 = new FakeWaiter();
            mgr.getConnection(waiter2);
            waiters.add(waiter2);
        }
        FakeConnection conn = connector.assertRequest();
        conn.concurrency(n).connect();
        waiters.forEach(waiter -> this.assertWaitUntil(waiter::isSuccess));
        waiters.forEach(FakeWaiter::recycle);
    }

    @Test
    public void testInitialNoConcurrency() {
        int n = 10;
        FakeConnectionProvider connector = new FakeConnectionProvider();
        FakeConnectionManager mgr = new FakeConnectionManager(-1, 1, connector);
        ArrayList<FakeWaiter> waiters = new ArrayList<FakeWaiter>();
        for (int i = 0; i < n; ++i) {
            FakeWaiter waiter2 = new FakeWaiter();
            mgr.getConnection(waiter2);
            waiters.add(waiter2);
        }
        FakeConnection conn = connector.assertRequest();
        conn.concurrency(0L).connect().awaitConnected();
        conn.concurrency(n - 1);
        this.assertWaitUntil(() -> waiters.stream().filter(FakeWaiter::isSuccess).count() == (long)(n - 1));
        waiters.stream().filter(FakeWaiter::isSuccess).findFirst().get().recycle();
        waiters.forEach(waiter -> this.assertWaitUntil(waiter::isSuccess));
    }

    @Test
    public void testRecycleWithoutDispose() {
        FakeConnectionProvider connector = new FakeConnectionProvider();
        FakeConnectionManager mgr = new FakeConnectionManager(-1, 1, connector);
        FakeWaiter waiter1 = new FakeWaiter();
        mgr.getConnection(waiter1);
        FakeConnection conn = connector.assertRequest();
        conn.connect();
        this.assertWaitUntil(waiter1::isSuccess);
        conn.recycle(false);
        FakeWaiter waiter2 = new FakeWaiter();
        mgr.getConnection(waiter2);
        this.assertWaitUntil(waiter2::isSuccess);
        waiter2.assertSuccess(conn);
        conn.recycle(true);
        this.assertEquals(0L, mgr.size());
    }

    @Test
    public void testDiscardWaiterWhenFull() {
        FakeConnectionProvider connector = new FakeConnectionProvider();
        FakeConnectionManager mgr = new FakeConnectionManager(2, 1, connector);
        FakeWaiter waiter1 = new FakeWaiter();
        mgr.getConnection(waiter1);
        FakeConnection conn = connector.assertRequest();
        FakeWaiter waiter2 = new FakeWaiter();
        mgr.getConnection(waiter2);
        FakeWaiter waiter3 = new FakeWaiter();
        mgr.getConnection(waiter3);
        FakeWaiter waiter4 = new FakeWaiter();
        mgr.getConnection(waiter4);
        this.assertWaitUntil(waiter4::isFailure);
    }

    @Test
    public void testDiscardConnectionDuringInit() {
        FakeConnectionProvider connector = new FakeConnectionProvider();
        FakeConnectionManager mgr = new FakeConnectionManager(2, 1, connector);
        FakeWaiter waiter1 = new FakeWaiter(){

            @Override
            public synchronized void initConnection(ContextInternal ctx, FakeConnection conn) {
                super.initConnection(ctx, conn);
                conn.close();
            }
        };
        mgr.getConnection(waiter1);
        FakeConnection conn = connector.assertRequest();
        conn.connect();
        this.assertWaitUntil(() -> connector.requests() == 1);
        this.assertFalse(mgr.closed());
    }

    @Test
    public void testStress() {
        int i;
        int numActors = 16;
        int numConnections = 1000;
        FakeConnectionProvider connector = new FakeConnectionProvider(){

            @Override
            public long connect(ConnectionListener<FakeConnection> listener, ContextImpl context) {
                int i = ThreadLocalRandom.current().nextInt(100);
                FakeConnection conn = new FakeConnection(context, listener);
                if (i < 10) {
                    conn.fail(new Exception("Could not connect"));
                } else {
                    conn.connect();
                }
                return 1L;
            }
        };
        FakeConnectionManager mgr = new FakeConnectionManager(-1, 16, connector);
        Thread[] actors = new Thread[numActors];
        for (i = 0; i < numActors; ++i) {
            actors[i] = new Thread(() -> {
                final CountDownLatch latch = new CountDownLatch(numConnections);
                for (int i1 = 0; i1 < numConnections; ++i1) {
                    mgr.getConnection(new Waiter<FakeConnection>((ContextImpl)this.vertx.getOrCreateContext()){

                        public void handleFailure(ContextInternal ctx, Throwable failure) {
                            latch.countDown();
                        }

                        public void initConnection(ContextInternal ctx, FakeConnection conn) {
                        }

                        public boolean handleConnection(ContextInternal ctx, FakeConnection conn) throws Exception {
                            int action = ThreadLocalRandom.current().nextInt(100);
                            if (action < -1) {
                                latch.countDown();
                                return false;
                            }
                            ConnectionPoolTest.this.vertx.setTimer(10L, id -> {
                                if (action < 15) {
                                    conn.close();
                                } else {
                                    conn.recycle();
                                }
                                latch.countDown();
                            });
                            return true;
                        }
                    });
                }
                try {
                    latch.await();
                }
                catch (InterruptedException e) {
                    e.printStackTrace();
                }
            });
            actors[i].start();
        }
        for (i = 0; i < actors.length; ++i) {
            try {
                actors[i].join();
                continue;
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
        this.assertWaitUntil(() -> mgr.closed());
        this.assertEquals(0L, mgr.size());
        this.assertEquals(0L, mgr.pool.waitersCount());
        this.assertEquals(0L, mgr.pool.waitersInQueue());
        this.assertEquals(0L, mgr.pool.weight());
        this.assertEquals(0L, mgr.pool.capacity());
    }

    class FakeConnectionProvider
    implements ConnectionProvider<FakeConnection> {
        private final ArrayDeque<FakeConnection> pendingRequests = new ArrayDeque();

        FakeConnectionProvider() {
        }

        void assertRequests(int expectedSize) {
            ConnectionPoolTest.this.assertEquals(expectedSize, this.pendingRequests.size());
        }

        int requests() {
            return this.pendingRequests.size();
        }

        FakeConnection assertRequest() {
            ConnectionPoolTest.this.assertNotNull(this.pendingRequests);
            ConnectionPoolTest.this.assertTrue(this.pendingRequests.size() > 0);
            FakeConnection request = this.pendingRequests.poll();
            ConnectionPoolTest.this.assertNotNull(request);
            return request;
        }

        public long connect(ConnectionListener<FakeConnection> listener, ContextImpl context) {
            this.pendingRequests.add(new FakeConnection(context, listener));
            return 1L;
        }

        public void close(FakeConnection conn) {
            conn.listener.onDiscard();
        }
    }

    class FakeConnection {
        private static final int DISCONNECTED = 0;
        private static final int CONNECTING = 1;
        private static final int CONNECTED = 2;
        private static final int CLOSED = 3;
        private final ContextImpl context;
        private final ConnectionListener<FakeConnection> listener;
        private final Channel channel = new EmbeddedChannel();
        private long inflight;
        private long concurrency = 1L;
        private int status = 0;

        FakeConnection(ContextImpl context, ConnectionListener<FakeConnection> listener) {
            this.context = context;
            this.listener = listener;
        }

        synchronized void close() {
            if (this.status != 2) {
                throw new IllegalStateException();
            }
            this.status = 3;
            this.listener.onDiscard();
        }

        synchronized long recycle(boolean dispose) {
            return this.recycle(1, dispose);
        }

        synchronized long recycle() {
            return this.recycle(true);
        }

        synchronized long recycle(int capacity, boolean dispose) {
            this.inflight -= (long)capacity;
            this.listener.onRecycle(capacity, dispose);
            return this.inflight;
        }

        synchronized FakeConnection concurrency(long value) {
            if (value < 0L) {
                throw new IllegalArgumentException("Invalid concurrency");
            }
            if (this.status == 2) {
                if (this.concurrency != value) {
                    this.concurrency = value;
                    this.listener.onConcurrencyChange(value);
                }
            } else {
                this.concurrency = value;
            }
            return this;
        }

        FakeConnection awaitConnected() {
            ConnectionPoolTest.this.waitUntil(() -> {
                FakeConnection fakeConnection = this;
                synchronized (fakeConnection) {
                    return this.status == 2;
                }
            });
            return this;
        }

        synchronized FakeConnection connect() {
            if (this.status != 0) {
                throw new IllegalStateException();
            }
            this.status = 1;
            this.context.nettyEventLoop().execute(() -> {
                FakeConnection fakeConnection = this;
                synchronized (fakeConnection) {
                    this.status = 2;
                    this.listener.onConnectSuccess((Object)this, this.concurrency, this.channel, this.context, 1L, 1L);
                }
            });
            return this;
        }

        void fail(Throwable err) {
            this.context.nettyEventLoop().execute(() -> this.listener.onConnectFailure(this.context, err, 1L));
        }
    }

    class FakeWaiter
    extends Waiter<FakeConnection> {
        private FakeConnection init;
        private boolean cancelled;
        private boolean completed;
        private Object result;

        FakeWaiter() {
            super((ContextImpl)ConnectionPoolTest.this.vertx.getOrCreateContext());
        }

        synchronized boolean cancel() {
            if (this.completed) {
                return false;
            }
            this.cancelled = true;
            return true;
        }

        synchronized boolean isInitialized(FakeConnection conn) {
            return this.init == conn;
        }

        synchronized void assertNotInitialized() {
            ConnectionPoolTest.this.assertSame(null, this.init);
        }

        synchronized void assertSuccess(FakeConnection conn) {
            ConnectionPoolTest.this.assertSame(conn, this.result);
        }

        synchronized void assertFailure(Throwable failure) {
            ConnectionPoolTest.this.assertSame(failure, this.result);
        }

        synchronized boolean isComplete() {
            return this.completed;
        }

        synchronized boolean isSuccess() {
            return this.completed && this.result instanceof FakeConnection;
        }

        synchronized boolean isFailure() {
            return this.completed && this.result instanceof Throwable;
        }

        public synchronized void handleFailure(ContextInternal ctx, Throwable failure) {
            ConnectionPoolTest.this.assertFalse(this.completed);
            this.completed = true;
            this.result = failure;
        }

        public synchronized void initConnection(ContextInternal ctx, FakeConnection conn) {
            ConnectionPoolTest.this.assertNull(this.init);
            ConnectionPoolTest.this.assertNotNull(conn);
            this.init = conn;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public synchronized boolean handleConnection(ContextInternal ctx, FakeConnection conn) throws Exception {
            ConnectionPoolTest.this.assertFalse(this.completed);
            this.completed = true;
            if (this.cancelled) {
                return false;
            }
            FakeConnection fakeConnection = conn;
            synchronized (fakeConnection) {
                conn.inflight++;
            }
            this.result = conn;
            return true;
        }

        long recycle() {
            FakeConnection conn = (FakeConnection)this.result;
            return conn.recycle();
        }
    }

    class FakeConnectionManager {
        private final FakeConnectionProvider connector;
        private final int queueMaxSize;
        private final int maxPoolSize;
        private Pool<FakeConnection> pool;
        private Set<FakeConnection> active = new HashSet<FakeConnection>();
        private boolean closed = true;
        private int seq;

        FakeConnectionManager(int queueMaxSize, int maxPoolSize, FakeConnectionProvider connector) {
            this.queueMaxSize = queueMaxSize;
            this.maxPoolSize = maxPoolSize;
            this.connector = connector;
        }

        synchronized int sequence() {
            return this.seq;
        }

        synchronized boolean closed() {
            return this.closed;
        }

        synchronized boolean contains(FakeConnection conn) {
            return this.active.contains(conn);
        }

        synchronized int size() {
            return this.active.size();
        }

        synchronized Pool<FakeConnection> pool() {
            return this.pool;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void getConnection(Waiter<FakeConnection> waiter) {
            FakeConnectionManager fakeConnectionManager = this;
            synchronized (fakeConnectionManager) {
                if (this.closed) {
                    ++this.seq;
                    this.closed = false;
                    this.pool = new Pool((ConnectionProvider)this.connector, this.queueMaxSize, (long)this.maxPoolSize, v -> {
                        FakeConnectionManager fakeConnectionManager = this;
                        synchronized (fakeConnectionManager) {
                            this.closed = true;
                        }
                    }, (channel, conn) -> {
                        FakeConnectionManager fakeConnectionManager = this;
                        synchronized (fakeConnectionManager) {
                            this.active.add((FakeConnection)conn);
                        }
                    }, (channel, conn) -> {
                        FakeConnectionManager fakeConnectionManager = this;
                        synchronized (fakeConnectionManager) {
                            this.active.remove(conn);
                        }
                    });
                }
            }
            this.pool.getConnection(waiter);
        }
    }
}

