/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.core.parsetools;

import io.vertx.core.Handler;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.parsetools.RecordParser;
import io.vertx.test.core.TestUtils;
import io.vertx.test.fakestream.FakeStream;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.junit.Assert;
import org.junit.Test;

public class RecordParserTest {
    @Test
    public void testIllegalArguments() {
        TestUtils.assertNullPointerException(() -> RecordParser.newDelimited((Buffer)null, handler -> {}));
        TestUtils.assertNullPointerException(() -> RecordParser.newDelimited((String)null, handler -> {}));
        RecordParser parser = RecordParser.newDelimited((String)"", handler -> {});
        TestUtils.assertNullPointerException(() -> parser.setOutput(null));
        TestUtils.assertNullPointerException(() -> parser.delimitedMode((Buffer)null));
        TestUtils.assertNullPointerException(() -> parser.delimitedMode((String)null));
        TestUtils.assertIllegalArgumentException(() -> parser.maxRecordSize(-1));
    }

    @Test
    public void testDelimited() {
        this.delimited(Buffer.buffer().appendByte((byte)10));
        this.delimited(Buffer.buffer().appendByte((byte)13).appendByte((byte)10));
        this.delimited(Buffer.buffer((byte[])new byte[]{0, 3, 2, 5, 6, 4, 6}));
    }

    @Test
    public void testFixed() {
        int i;
        int lines = 50;
        Buffer[] expected = new Buffer[lines];
        List<Buffer> lineList = this.generateLines(lines, false, (byte)0);
        expected = lineList.toArray(expected);
        int totLength = lines * (lines - 1) / 2;
        Buffer inp = Buffer.buffer((int)totLength);
        for (i = 0; i < lines; ++i) {
            inp.appendBuffer(expected[i]);
        }
        for (i = 1; i < inp.length() * 2; ++i) {
            this.doTestFixed(inp, new Integer[]{i}, expected);
        }
        List<Integer> chunkSizes = this.generateChunkSizes(lines);
        for (int i2 = 0; i2 < 10; ++i2) {
            Collections.shuffle(chunkSizes);
            this.doTestFixed(inp, chunkSizes.toArray(new Integer[0]), expected);
        }
    }

    @Test
    public void testMixed() {
        int lines = 8;
        final ArrayList<Object> types = new ArrayList<Object>();
        class MyHandler
        implements Handler<Buffer> {
            RecordParser parser = RecordParser.newFixed((int)10, (Handler)this);
            int pos;

            MyHandler() {
            }

            public void handle(Buffer buff) {
                if (this.pos < 8) {
                    Object type = types.get(this.pos);
                    if (type instanceof byte[]) {
                        byte[] bytes = (byte[])type;
                        this.parser.delimitedMode(Buffer.buffer((byte[])bytes));
                    } else {
                        int length = (Integer)type;
                        this.parser.fixedSizeMode(length);
                    }
                }
            }
        }
        MyHandler out = new MyHandler();
        Buffer[] expected = new Buffer[8];
        Buffer input = Buffer.buffer((int)100);
        expected[0] = TestUtils.randomBuffer(10);
        input.appendBuffer(expected[0]);
        types.add(expected[0].length());
        expected[1] = TestUtils.randomBuffer(100);
        input.appendBuffer(expected[1]);
        types.add(expected[1].length());
        byte[] delim = new byte[]{23, -120, 100, 3};
        expected[2] = TestUtils.randomBuffer(50, true, delim[0]);
        input.appendBuffer(expected[2]);
        types.add(delim);
        input.appendBuffer(Buffer.buffer((byte[])delim));
        expected[3] = TestUtils.randomBuffer(1000);
        input.appendBuffer(expected[3]);
        types.add(expected[3].length());
        expected[4] = TestUtils.randomBuffer(230, true, delim[0]);
        input.appendBuffer(expected[4]);
        types.add(delim);
        input.appendBuffer(Buffer.buffer((byte[])delim));
        delim = new byte[]{17};
        expected[5] = TestUtils.randomBuffer(341, true, delim[0]);
        input.appendBuffer(expected[5]);
        types.add(delim);
        input.appendBuffer(Buffer.buffer((byte[])delim));
        delim = new byte[]{54, -32, 0};
        expected[6] = TestUtils.randomBuffer(1234, true, delim[0]);
        input.appendBuffer(expected[6]);
        types.add(delim);
        input.appendBuffer(Buffer.buffer((byte[])delim));
        expected[7] = TestUtils.randomBuffer(100);
        input.appendBuffer(expected[7]);
        types.add(expected[7].length());
        this.feedChunks(input, out.parser, new Integer[]{50, 10, 3});
    }

    private void delimited(Buffer delim) {
        int i;
        int lines = 50;
        Buffer[] expected = new Buffer[lines];
        List<Buffer> lineList = this.generateLines(lines, true, delim.getByte(0));
        expected = lineList.toArray(expected);
        int totLength = lines * (lines - 1) / 2;
        Buffer inp = Buffer.buffer((int)(totLength + lines * delim.length()));
        for (i = 0; i < lines; ++i) {
            inp.appendBuffer(expected[i]);
            inp.appendBuffer(delim);
        }
        for (i = 1; i < inp.length() * 2; ++i) {
            this.doTestDelimited(inp, delim, new Integer[]{i}, expected);
        }
        List<Integer> chunkSizes = this.generateChunkSizes(lines);
        for (int i2 = 0; i2 < 10; ++i2) {
            Collections.shuffle(chunkSizes);
            this.doTestDelimited(inp, delim, chunkSizes.toArray(new Integer[0]), expected);
        }
    }

    private void doTestDelimited(Buffer input, Buffer delim, Integer[] chunkSizes, Buffer ... expected) {
        final Buffer[] results = new Buffer[expected.length];
        Handler<Buffer> out = new Handler<Buffer>(){
            int pos;

            public void handle(Buffer buff) {
                results[this.pos++] = buff;
            }
        };
        RecordParser parser = RecordParser.newDelimited((Buffer)delim, (Handler)out);
        this.feedChunks(input, parser, chunkSizes);
        this.checkResults(expected, results);
    }

    private void doTestFixed(Buffer input, Integer[] chunkSizes, final Buffer ... expected) {
        final Buffer[] results = new Buffer[expected.length];
        class MyHandler
        implements Handler<Buffer> {
            int pos;
            RecordParser parser;

            MyHandler() {
                this.parser = RecordParser.newFixed((int)expected[0].length(), (Handler)this);
            }

            public void handle(Buffer buff) {
                results[this.pos++] = buff;
                if (this.pos < expected.length) {
                    this.parser.fixedSizeMode(expected[this.pos].length());
                }
            }
        }
        MyHandler out = new MyHandler();
        this.feedChunks(input, out.parser, chunkSizes);
        this.checkResults(expected, results);
    }

    private void feedChunks(Buffer input, RecordParser parser, Integer[] chunkSizes) {
        int chunkSize;
        int chunkPos = 0;
        for (int pos = 0; pos < input.length(); pos += chunkSize) {
            int end;
            chunkSize = chunkSizes[chunkPos++];
            if (chunkPos == chunkSizes.length) {
                chunkPos = 0;
            }
            end = (end = pos + chunkSize) <= input.length() ? end : input.length();
            Buffer sub = input.getBuffer(pos, end);
            parser.handle(sub);
        }
    }

    private void checkResults(Buffer[] expected, Buffer[] results) {
        for (int i = 0; i < expected.length; ++i) {
            Assert.assertEquals((String)("Expected:" + expected[i] + " length:" + expected[i].length() + " Actual:" + results[i] + " length:" + results[i].length()), (Object)expected[i], (Object)results[i]);
        }
    }

    private List<Buffer> generateLines(int lines, boolean delim, byte delimByte) {
        ArrayList<Buffer> lineList = new ArrayList<Buffer>();
        for (int i = 0; i < lines; ++i) {
            lineList.add(TestUtils.randomBuffer(i + 1, delim, delimByte));
        }
        Collections.shuffle(lineList);
        return lineList;
    }

    private List<Integer> generateChunkSizes(int lines) {
        ArrayList<Integer> chunkSizes = new ArrayList<Integer>();
        for (int i = 1; i < lines / 5; ++i) {
            chunkSizes.add(i);
        }
        return chunkSizes;
    }

    @Test
    public void testSpreadDelimiter() {
        this.doTestDelimited(Buffer.buffer((String)"start-a-b-c-dddabc"), Buffer.buffer((String)"abc"), new Integer[]{18}, Buffer.buffer((String)"start-a-b-c-ddd"));
        this.doTestDelimited(Buffer.buffer((String)"start-abc-dddabc"), Buffer.buffer((String)"abc"), new Integer[]{18}, Buffer.buffer((String)"start-"), Buffer.buffer((String)"-ddd"));
        this.doTestDelimited(Buffer.buffer((String)"start-ab-c-dddabc"), Buffer.buffer((String)"abc"), new Integer[]{18}, Buffer.buffer((String)"start-ab-c-ddd"));
    }

    @Test
    public void testDelimitedMaxRecordSize() {
        this.doTestDelimitedMaxRecordSize(Buffer.buffer((String)"ABCD\nEFGH\n"), Buffer.buffer((String)"\n"), new Integer[]{2}, 4, null, Buffer.buffer((String)"ABCD"), Buffer.buffer((String)"EFGH"));
        this.doTestDelimitedMaxRecordSize(Buffer.buffer((String)"A\nBC10\nDEFGHIJKLM\n"), Buffer.buffer((String)"\n"), new Integer[]{2}, 10, null, Buffer.buffer((String)"A"), Buffer.buffer((String)"BC10"), Buffer.buffer((String)"DEFGHIJKLM"));
        this.doTestDelimitedMaxRecordSize(Buffer.buffer((String)"AB\nC\n\nDEFG\n\n"), Buffer.buffer((String)"\n\n"), new Integer[]{2}, 4, null, Buffer.buffer((String)"AB\nC"), Buffer.buffer((String)"DEFG"));
        this.doTestDelimitedMaxRecordSize(Buffer.buffer((String)"AB--C---D-"), Buffer.buffer((String)"-"), new Integer[]{3}, 2, null, Buffer.buffer((String)"AB"), Buffer.buffer((String)""), Buffer.buffer((String)"C"), Buffer.buffer((String)""), Buffer.buffer((String)""), Buffer.buffer((String)"D"));
        try {
            this.doTestDelimitedMaxRecordSize(Buffer.buffer((String)"ABCD--"), Buffer.buffer((String)"--"), new Integer[]{2}, 3, null, Buffer.buffer());
            Assert.fail((String)"should throw exception");
        }
        catch (IllegalStateException illegalStateException) {
            // empty catch block
        }
        AtomicBoolean handled = new AtomicBoolean();
        Handler exHandler = throwable -> handled.set(true);
        this.doTestDelimitedMaxRecordSize(Buffer.buffer((String)"ABCD--"), Buffer.buffer((String)"--"), new Integer[]{2}, 3, (Handler<Throwable>)exHandler, Buffer.buffer((String)"ABCD"));
        Assert.assertTrue((boolean)handled.get());
    }

    private void doTestDelimitedMaxRecordSize(Buffer input, Buffer delim, Integer[] chunkSizes, int maxRecordSize, Handler<Throwable> exHandler, Buffer ... expected) {
        final Buffer[] results = new Buffer[expected.length];
        Handler<Buffer> out = new Handler<Buffer>(){
            int pos;

            public void handle(Buffer buff) {
                results[this.pos++] = buff;
            }
        };
        RecordParser parser = RecordParser.newDelimited((Buffer)delim, (Handler)out);
        parser.maxRecordSize(maxRecordSize);
        parser.exceptionHandler(exHandler);
        this.feedChunks(input, parser, chunkSizes);
        this.checkResults(expected, results);
    }

    @Test
    public void testWrapReadStream() {
        FakeStream<Buffer> stream = new FakeStream<Buffer>();
        RecordParser parser = RecordParser.newDelimited((String)"\r\n", stream);
        AtomicInteger ends = new AtomicInteger();
        parser.endHandler(v -> ends.incrementAndGet());
        ArrayDeque records = new ArrayDeque();
        parser.handler(record -> records.add(record.toString()));
        Assert.assertFalse((boolean)stream.isPaused());
        parser.pause();
        parser.resume();
        stream.emit(Buffer.buffer((String)"first\r\nsecond\r\nthird"));
        Assert.assertEquals((Object)"first", records.poll());
        Assert.assertEquals((Object)"second", records.poll());
        Assert.assertNull(records.poll());
        stream.emit(Buffer.buffer((String)"\r\n"));
        Assert.assertEquals((Object)"third", records.poll());
        Assert.assertNull(records.poll());
        Assert.assertEquals((long)0L, (long)ends.get());
        Throwable cause = new Throwable();
        stream.fail(cause);
        ArrayList failures = new ArrayList();
        parser.exceptionHandler(failures::add);
        stream.fail(cause);
        Assert.assertEquals(Collections.singletonList(cause), failures);
        Assert.assertFalse((boolean)stream.isPaused());
        parser.pause();
        Assert.assertFalse((boolean)stream.isPaused());
        int count = 0;
        do {
            stream.emit(Buffer.buffer((String)("item-" + count++ + "\r\n")));
        } while (!stream.isPaused());
        Assert.assertNull(records.poll());
        parser.resume();
        for (int i = 0; i < count; ++i) {
            Assert.assertEquals((Object)("item-" + i), records.poll());
        }
        Assert.assertNull(records.poll());
        Assert.assertFalse((boolean)stream.isPaused());
        stream.end();
        Assert.assertEquals((long)1L, (long)ends.get());
    }

    @Test
    public void testPausedStreamShouldNotPauseOnIncompleteMatch() {
        FakeStream<Buffer> stream = new FakeStream<Buffer>();
        RecordParser parser = RecordParser.newDelimited((String)"\r\n", stream);
        parser.handler(event -> {});
        parser.pause().fetch(1L);
        stream.emit(Buffer.buffer((String)"abc"));
        Assert.assertFalse((boolean)stream.isPaused());
    }

    @Test
    public void testSuspendParsing() {
        FakeStream<Buffer> stream = new FakeStream<Buffer>();
        RecordParser parser = RecordParser.newDelimited((String)"\r\n", stream);
        ArrayList emitted = new ArrayList();
        parser.handler(emitted::add);
        parser.pause().fetch(1L);
        stream.emit(Buffer.buffer((String)"abc\r\ndef\r\n"));
        parser.fetch(1L);
        Assert.assertEquals(Arrays.asList(Buffer.buffer((String)"abc"), Buffer.buffer((String)"def")), emitted);
    }

    @Test
    public void testParseEmptyChunkOnFetch() {
        FakeStream<Buffer> stream = new FakeStream<Buffer>();
        RecordParser parser = RecordParser.newDelimited((String)"\r\n", stream);
        ArrayList emitted = new ArrayList();
        parser.handler(emitted::add);
        parser.pause();
        stream.emit(Buffer.buffer((String)"abc\r\n\r\n"));
        parser.fetch(1L);
        Assert.assertEquals(Collections.singletonList(Buffer.buffer((String)"abc")), emitted);
        parser.fetch(1L);
        Assert.assertEquals(Arrays.asList(Buffer.buffer((String)"abc"), Buffer.buffer()), emitted);
    }

    @Test
    public void testSwitchModeResetsState() {
        FakeStream<Buffer> stream = new FakeStream<Buffer>();
        RecordParser parser = RecordParser.newDelimited((String)"\r\n", stream);
        ArrayList emitted = new ArrayList();
        parser.handler(emitted::add);
        parser.pause();
        stream.emit(Buffer.buffer((String)"3\r\nabc\r\n"));
        parser.fetch(1L);
        Assert.assertEquals(Collections.singletonList(Buffer.buffer((String)"3")), emitted);
        parser.fixedSizeMode(5);
        parser.fetch(1L);
        Assert.assertEquals(Arrays.asList(Buffer.buffer((String)"3"), Buffer.buffer((String)"abc\r\n")), emitted);
    }

    @Test
    public void testEndOfWrappedStreamWithPauseWithFinalDelimiter() throws Exception {
        this.endOfWrappedStream(true, true);
    }

    @Test
    public void testEndOfWrappedStreamWithPauseWithoutFinalDelimiter() throws Exception {
        this.endOfWrappedStream(true, false);
    }

    @Test
    public void testEndOfWrappedStreamWithoutPauseWithoutFinalDelimiter() throws Exception {
        this.endOfWrappedStream(false, false);
    }

    @Test
    public void testEndOfWrappedStreamWithoutPauseWithFinalDelimiter() throws Exception {
        this.endOfWrappedStream(false, true);
    }

    private void endOfWrappedStream(boolean pauseParser, boolean finalDelimiter) throws Exception {
        FakeStream<Buffer> stream = new FakeStream<Buffer>();
        RecordParser parser = RecordParser.newDelimited((String)"\n", stream);
        List emitted = Collections.synchronizedList(new ArrayList());
        CountDownLatch endLatch = new CountDownLatch(1);
        parser.endHandler(v -> {
            Assert.assertEquals(Arrays.asList(Buffer.buffer((String)"toto"), Buffer.buffer((String)"titi")), (Object)emitted);
            endLatch.countDown();
        });
        parser.handler(emitted::add);
        if (pauseParser) {
            parser.pause();
        }
        stream.emit(Buffer.buffer((String)"toto\ntit"));
        stream.emit(Buffer.buffer((String)(finalDelimiter ? "i\n" : "i")));
        stream.end();
        if (pauseParser) {
            parser.fetch(1L);
            parser.resume();
        }
        endLatch.await();
    }

    @Test
    public void testParserIsNotReentrant() throws Exception {
        RecordParser recordParser = RecordParser.newDelimited((String)"\n");
        AtomicInteger state = new AtomicInteger(0);
        StringBuffer emitted = new StringBuffer();
        CountDownLatch latch = new CountDownLatch(2);
        recordParser.handler(buff -> {
            emitted.append(buff.toString());
            if (state.compareAndSet(0, 1)) {
                recordParser.handle(Buffer.buffer((String)"bar\n"));
                Assert.assertEquals((String)"Parser shouldn't trigger parsing while emitting", (Object)"foo", (Object)emitted.toString());
                latch.countDown();
            } else if (state.compareAndSet(1, 2)) {
                Assert.assertEquals((Object)"foobar", (Object)emitted.toString());
                latch.countDown();
            }
        });
        recordParser.handle(Buffer.buffer((String)"foo\n"));
        latch.await();
    }
}

