/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.core.file;

import io.vertx.core.Vertx;
import io.vertx.core.VertxOptions;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.file.FileSystemOptions;
import io.vertx.core.file.impl.FileResolver;
import io.vertx.core.http.HttpClientOptions;
import io.vertx.core.http.HttpClientRequest;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.HttpServerOptions;
import io.vertx.core.impl.Utils;
import io.vertx.core.impl.VertxInternal;
import io.vertx.test.core.VertxTestBase;
import java.io.File;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.stream.Collectors;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Test;

public abstract class FileResolverTestBase
extends VertxTestBase {
    private final String cacheBaseDir = new File(System.getProperty("java.io.tmpdir", ".") + File.separator + "vertx-cache").getAbsolutePath();
    protected FileResolver resolver;
    private ClassLoader testCL;

    @Override
    public void setUp() throws Exception {
        super.setUp();
        this.testCL = Thread.currentThread().getContextClassLoader();
        File baseDir = new File(new File(new File(new File("src"), "test"), "classpath"), "fileresolver");
        this.assertTrue(baseDir.exists() && baseDir.isDirectory());
        ClassLoader resourcesLoader = this.resourcesLoader(baseDir);
        Thread.currentThread().setContextClassLoader(resourcesLoader);
        this.resolver = new FileResolver();
    }

    protected ClassLoader resourcesLoader(File baseDir) throws Exception {
        return Thread.currentThread().getContextClassLoader();
    }

    @Override
    protected void tearDown() throws Exception {
        Thread.currentThread().setContextClassLoader(this.testCL);
        this.resolver.close();
        super.tearDown();
    }

    @Test
    public void testReadFileInDirThenReadDir() {
        Buffer buff = this.vertx.fileSystem().readFileBlocking("webroot/subdir/subfile.html");
        this.assertEquals(buff.toString(), "<html><body>subfile</body></html>");
        Set names = this.vertx.fileSystem().readDirBlocking("webroot/subdir").stream().map(path -> {
            int idx = path.lastIndexOf(File.separator);
            return idx == -1 ? path : path.substring(idx + 1);
        }).collect(Collectors.toSet());
        this.assertEquals(names, new HashSet<String>(Arrays.asList("subdir2", "subfile.html")));
    }

    @Test
    public void testResolveNotExistFile() {
        File file = this.resolver.resolveFile("doesnotexist.txt");
        this.assertFalse(file.exists());
        this.assertEquals("doesnotexist.txt", file.getPath());
    }

    @Test
    public void testResolveNotExistDirectory() {
        File file = this.resolver.resolveFile("somedir");
        this.assertFalse(file.exists());
        this.assertEquals("somedir", file.getPath());
    }

    @Test
    public void testResolveNotExistFileInDirectory() {
        File file = this.resolver.resolveFile("somedir/doesnotexist.txt");
        this.assertFalse(file.exists());
        this.assertEquals("somedir" + File.separator + "doesnotexist.txt", file.getPath());
    }

    @Test
    public void testResolveFileFromClasspath() throws Exception {
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolveFile("afile.html");
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertFalse(file.isDirectory());
            this.assertEquals("<html><body>afile</body></html>", this.readFile(file));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testResolveFileFromClasspathDisableCaching() throws Exception {
        try (VertxInternal vertx = (VertxInternal)Vertx.vertx((VertxOptions)new VertxOptions().setFileSystemOptions(new FileSystemOptions().setFileCachingEnabled(true)));){
            for (int i = 0; i < 2; ++i) {
                File file = vertx.resolveFile("afile.html");
                this.assertTrue(file.exists());
                this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
                this.assertFalse(file.isDirectory());
                this.assertEquals("<html><body>afile</body></html>", this.readFile(file));
            }
        }
    }

    @Test
    public void testResolveFileWithSpacesFromClasspath() throws Exception {
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolveFile("afile with spaces.html");
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertFalse(file.isDirectory());
            this.assertEquals("<html><body>afile with spaces</body></html>", this.readFile(file));
        }
    }

    @Test
    public void testResolveFileWithSpaceAtEndFromClasspath() {
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolveFile("afilewithspaceatend ");
            if (Utils.isWindows()) {
                this.assertFalse(file.exists());
                continue;
            }
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertFalse(file.isDirectory());
            this.assertEquals("afilewithspaceatend ", this.readFile(file));
        }
    }

    @Test
    public void testCacheDirIsPosix0700() throws Exception {
        Assume.assumeFalse((boolean)Utils.isWindows());
        File file = this.resolver.resolveFile("afile with spaces.html");
        this.assertTrue(file.exists());
        this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
        File parent = file.getParentFile();
        this.assertEquals("rwx------", PosixFilePermissions.toString(Files.getPosixFilePermissions(parent.toPath(), new LinkOption[0])));
    }

    @Test
    public void testResolveDirectoryFromClasspath() throws Exception {
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolveFile("webroot");
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertTrue(file.isDirectory());
        }
    }

    @Test
    public void testResolveFileInDirectoryFromClasspath() throws Exception {
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolveFile("webroot/somefile.html");
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertFalse(file.isDirectory());
            this.assertEquals("<html><body>blah</body></html>", this.readFile(file));
        }
    }

    @Test
    public void testResolveSubDirectoryFromClasspath() throws Exception {
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolveFile("webroot/subdir");
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertTrue(file.isDirectory());
        }
    }

    @Test
    public void testResolveFileInSubDirectoryFromClasspath() throws Exception {
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolveFile("webroot/subdir/subfile.html");
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertFalse(file.isDirectory());
            this.assertEquals("<html><body>subfile</body></html>", this.readFile(file));
        }
    }

    @Test
    public void testRecursivelyUnpack() throws Exception {
        File file = this.resolver.resolveFile("webroot/subdir");
        this.assertTrue(file.exists());
        File sub = new File(file, "subfile.html");
        this.assertTrue(sub.exists());
        this.assertEquals("<html><body>subfile</body></html>", this.readFile(sub));
    }

    @Test
    public void testRecursivelyUnpack2() throws Exception {
        File file = this.resolver.resolveFile("webroot/subdir");
        this.assertTrue(file.exists());
        File sub = new File(new File(file, "subdir2"), "subfile2.html");
        this.assertTrue(sub.exists());
        this.assertEquals("<html><body>subfile2</body></html>", this.readFile(sub));
    }

    @Test
    public void testDeleteCacheDir() throws Exception {
        FileResolver resolver2 = new FileResolver();
        File file = resolver2.resolveFile("webroot/somefile.html");
        this.assertTrue(file.exists());
        File cacheDir = file.getParentFile().getParentFile();
        this.assertTrue(cacheDir.exists());
        resolver2.close();
        this.assertFalse(cacheDir.exists());
    }

    @Test
    public void testCacheDirDeletedOnVertxClose() {
        VertxInternal vertx2 = (VertxInternal)this.vertx();
        File file = vertx2.resolveFile("webroot/somefile.html");
        this.assertTrue(file.exists());
        File cacheDir = file.getParentFile().getParentFile();
        this.assertTrue(cacheDir.exists());
        vertx2.close(this.onSuccess(v -> {
            this.assertFalse(cacheDir.exists());
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testFileSystemReadFile() {
        this.assertTrue(this.vertx.fileSystem().existsBlocking("afile.html"));
        this.assertFalse(this.vertx.fileSystem().propsBlocking("afile.html").isDirectory());
        Buffer buffer = this.vertx.fileSystem().readFileBlocking("afile.html");
        this.assertNotNull(buffer);
        this.assertTrue(buffer.toString().startsWith("<html><body>afile</body></html>"));
    }

    @Test
    public void testFileSystemReadDirectory() {
        this.assertTrue(this.vertx.fileSystem().existsBlocking("webroot"));
        this.assertTrue(this.vertx.fileSystem().propsBlocking("webroot").isDirectory());
    }

    @Test
    public void testSendFileFromClasspath() {
        this.vertx.createHttpServer(new HttpServerOptions().setPort(8080)).requestHandler(res -> res.response().sendFile("webroot/somefile.html")).listen(this.onSuccess(res -> this.vertx.createHttpClient(new HttpClientOptions()).request(HttpMethod.GET, 8080, "localhost", "/").compose(HttpClientRequest::send).onComplete(this.onSuccess(resp -> resp.bodyHandler(buff -> {
            this.assertTrue(buff.toString().startsWith("<html><body>blah</body></html>"));
            this.testComplete();
        })))));
        this.await();
    }

    @Test
    public void testResolveFileFromDifferentThreads() throws Exception {
        int size = 0xA00000;
        byte[] content = new byte[size];
        new Random().nextBytes(content);
        File out = new File("target/test-classes/temp");
        if (out.exists()) {
            Files.delete(out.toPath());
        }
        Files.write(out.toPath(), content, StandardOpenOption.CREATE_NEW);
        int count = 100;
        CountDownLatch latch = new CountDownLatch(count);
        CountDownLatch start = new CountDownLatch(1);
        ArrayList errors = new ArrayList();
        for (int i = 0; i < count; ++i) {
            Runnable runnable = () -> {
                try {
                    start.await();
                    File file = this.resolver.resolveFile("temp");
                    byte[] data = Files.readAllBytes(file.toPath());
                    Assert.assertArrayEquals((byte[])content, (byte[])data);
                }
                catch (Exception e) {
                    errors.add(e);
                }
                finally {
                    latch.countDown();
                }
            };
            new Thread(runnable).start();
        }
        start.countDown();
        latch.await();
        this.assertTrue(errors.isEmpty());
    }

    @Test
    public void testEnableCaching() throws Exception {
        this.testCaching(true);
    }

    @Test
    public void testDisableCaching() throws Exception {
        this.testCaching(false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testCaching(boolean enabled) throws Exception {
        VertxInternal vertx = (VertxInternal)Vertx.vertx((VertxOptions)new VertxOptions().setFileSystemOptions(new FileSystemOptions().setFileCachingEnabled(enabled)));
        File tmp = File.createTempFile("vertx", ".bin");
        tmp.deleteOnExit();
        final URL url = tmp.toURI().toURL();
        Files.write(tmp.toPath(), "foo".getBytes(), new OpenOption[0]);
        ClassLoader old = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(new ClassLoader(){

                @Override
                public URL getResource(String name) {
                    if ("foo".equals(name)) {
                        return url;
                    }
                    return super.getResource(name);
                }
            });
            File f = vertx.resolveFile("foo");
            this.assertEquals("foo", new String(Files.readAllBytes(f.toPath())));
            Files.write(tmp.toPath(), "bar".getBytes(), new OpenOption[0]);
            f = vertx.resolveFile("foo");
            if (enabled) {
                this.assertEquals("foo", new String(Files.readAllBytes(f.toPath())));
            } else {
                this.assertEquals("bar", new String(Files.readAllBytes(f.toPath())));
            }
        }
        finally {
            Thread.currentThread().setContextClassLoader(old);
        }
    }

    private String readFile(File file) {
        return this.vertx.fileSystem().readFileBlocking(file.getAbsolutePath()).toString();
    }

    @Test
    public void testResolveAfterCloseThrowsISE() throws Exception {
        FileResolver resolver2 = new FileResolver();
        File file = resolver2.resolveFile("webroot/somefile.html");
        this.assertTrue(file.exists());
        File cacheDir = file.getParentFile().getParentFile();
        this.assertTrue(cacheDir.exists());
        resolver2.close();
        this.assertFalse(cacheDir.exists());
        try {
            resolver2.resolveFile("webroot/somefile.html");
            this.fail("Should fail");
        }
        catch (IllegalStateException illegalStateException) {
            // empty catch block
        }
    }
}

