/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.core.net;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.DeploymentOptions;
import io.vertx.core.Future;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.net.NetClient;
import io.vertx.core.net.NetServer;
import io.vertx.core.net.NetServerOptions;
import io.vertx.core.net.SocketAddress;
import io.vertx.core.net.TrafficShapingOptions;
import io.vertx.test.core.TestUtils;
import io.vertx.test.core.VertxTestBase;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class NetBandwidthLimitingTest
extends VertxTestBase {
    public static final String DEFAULT_HOST = "localhost";
    private static final int OUTBOUND_LIMIT = 65536;
    private static final int INBOUND_LIMIT = 65536;
    @Rule
    public TemporaryFolder testFolder = new TemporaryFolder();
    private SocketAddress testAddress;
    private NetClient client = null;

    @Override
    @Before
    public void setUp() throws Exception {
        super.setUp();
        if (USE_DOMAIN_SOCKETS) {
            this.assertTrue("Native transport not enabled", USE_NATIVE_TRANSPORT);
            File tmp = TestUtils.tmpFile(".sock");
            this.testAddress = SocketAddress.domainSocketAddress((String)tmp.getAbsolutePath());
        } else {
            this.testAddress = SocketAddress.inetSocketAddress((int)1234, (String)DEFAULT_HOST);
        }
        this.client = this.vertx.createNetClient();
    }

    @Override
    @After
    public void after() throws InterruptedException {
        CountDownLatch countDownLatch = new CountDownLatch(2);
        this.client.close().onComplete(v -> countDownLatch.countDown());
        this.vertx.close().onComplete(v -> countDownLatch.countDown());
        this.awaitLatch(countDownLatch);
    }

    @Test
    public void sendBufferThrottled() {
        long startTime = System.nanoTime();
        Buffer expected = TestUtils.randomBuffer(262144);
        Buffer received = Buffer.buffer();
        NetServer server = this.netServer(this.vertx);
        server.connectHandler(sock -> sock.handler(buf -> sock.write((Object)expected)));
        Future result = server.listen(this.testAddress);
        result.onComplete(this.onSuccess(resp -> {
            Future clientConnect = this.client.connect(this.testAddress);
            clientConnect.onComplete(this.onSuccess(sock -> {
                sock.handler(buff -> {
                    received.appendBuffer(buff);
                    if (received.length() == expected.length()) {
                        long expectedTimeInMillis = this.expectedTimeMillis(received.length(), 65536);
                        this.assertEquals(expected, received);
                        long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
                        this.assertTimeTakenFallsInRange(expectedTimeInMillis, elapsedMillis);
                        this.testComplete();
                    }
                });
                sock.write("foo");
            }));
        }));
        this.await();
    }

    @Test
    public void sendFileIsThrottled() throws Exception {
        long startTime = System.nanoTime();
        File fDir = this.testFolder.newFolder();
        String content = TestUtils.randomUnicodeString(60000);
        File file = this.setupFile(fDir.toString(), "some-file.txt", content);
        Buffer expected = Buffer.buffer((String)content);
        Buffer received = Buffer.buffer();
        NetServer server = this.netServer(this.vertx);
        server.connectHandler(sock -> sock.handler(buf -> sock.sendFile(file.getAbsolutePath())));
        Future result = server.listen(this.testAddress);
        result.onComplete(this.onSuccess(resp -> {
            Future clientConnect = this.client.connect(this.testAddress);
            clientConnect.onComplete(this.onSuccess(sock -> {
                sock.handler(buff -> {
                    received.appendBuffer(buff);
                    if (received.length() == expected.length()) {
                        long expectedTimeInMillis = this.expectedTimeMillis(received.length(), 65536);
                        this.assertEquals(expected, received);
                        long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
                        this.assertTimeTakenFallsInRange(expectedTimeInMillis, elapsedMillis);
                        this.testComplete();
                    }
                });
                sock.write("foo");
            }));
        }));
        this.await();
    }

    @Test
    public void dataUploadIsThrottled() {
        long startTime = System.nanoTime();
        Buffer expected = TestUtils.randomBuffer(262144);
        Buffer received = Buffer.buffer();
        NetServer server = this.netServer(this.vertx);
        server.connectHandler(sock -> {
            sock.handler(buff -> {
                received.appendBuffer(buff);
                if (received.length() == expected.length()) {
                    long expectedTimeInMillis = this.expectedTimeMillis(received.length(), 65536);
                    this.assertEquals(expected, received);
                    long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
                    this.assertTimeTakenFallsInRange(expectedTimeInMillis, elapsedMillis);
                    this.testComplete();
                }
            });
            sock.write("foo");
        });
        Future result = server.listen(this.testAddress);
        result.onComplete(this.onSuccess(resp -> {
            Future clientConnect = this.client.connect(this.testAddress);
            clientConnect.onComplete(this.onSuccess(sock -> sock.handler(buf -> sock.write((Object)expected))));
        }));
        this.await();
    }

    @Test
    public void fileUploadIsThrottled() throws Exception {
        long startTime = System.nanoTime();
        File fDir = this.testFolder.newFolder();
        String content = TestUtils.randomUnicodeString(60000);
        File file = this.setupFile(fDir.toString(), "some-file.txt", content);
        Buffer expected = Buffer.buffer((String)content);
        Buffer received = Buffer.buffer();
        NetServer server = this.netServer(this.vertx);
        server.connectHandler(sock -> {
            sock.handler(buff -> {
                received.appendBuffer(buff);
                if (received.length() == expected.length()) {
                    long expectedTimeInMillis = this.expectedTimeMillis(received.length(), 65536);
                    this.assertEquals(expected, received);
                    long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
                    this.assertTimeTakenFallsInRange(expectedTimeInMillis, elapsedMillis);
                    this.testComplete();
                }
            });
            sock.write("foo");
        });
        Future result = server.listen(this.testAddress);
        result.onComplete(this.onSuccess(resp -> {
            Future clientConnect = this.client.connect(this.testAddress);
            clientConnect.onComplete(this.onSuccess(sock -> sock.handler(buf -> sock.sendFile(file.getAbsolutePath()))));
        }));
        this.await();
    }

    @Test
    public void testSendBufferIsTrafficShapedWithSharedServers() throws Exception {
        final Buffer expected = TestUtils.randomBuffer(262144);
        int numEventLoops = 4;
        Future listenLatch = this.vertx.deployVerticle(() -> new AbstractVerticle(){

            public void start(Promise<Void> startPromise) {
                NetServer testServer = NetBandwidthLimitingTest.this.netServer(this.vertx);
                testServer.connectHandler(sock -> sock.handler(buf -> sock.write((Object)expected)));
                testServer.listen(NetBandwidthLimitingTest.this.testAddress).onComplete(v -> startPromise.complete());
            }
        }, new DeploymentOptions().setInstances(numEventLoops));
        AtomicLong startTime = new AtomicLong();
        CountDownLatch waitForResponse = new CountDownLatch(4);
        listenLatch.onComplete(this.onSuccess(v -> {
            startTime.set(System.nanoTime());
            for (int i = 0; i < 4; ++i) {
                Buffer received = Buffer.buffer();
                Future clientConnect = this.client.connect(this.testAddress);
                clientConnect.onComplete(this.onSuccess(sock -> {
                    sock.handler(buff -> {
                        received.appendBuffer(buff);
                        if (received.length() == expected.length()) {
                            this.assertEquals(expected, received);
                            waitForResponse.countDown();
                        }
                    });
                    sock.write("foo");
                }));
            }
        }));
        waitForResponse.await();
        long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime.get());
        long expectedTimeInMillis = this.expectedTimeMillis(expected.length() * 4, 65536);
        this.assertTimeTakenFallsInRange(expectedTimeInMillis, elapsedMillis);
    }

    private long expectedTimeMillis(int size, int rate) {
        return TimeUnit.MILLISECONDS.convert(size / rate, TimeUnit.SECONDS);
    }

    private void assertTimeTakenFallsInRange(long expectedTimeInMillis, long actualTimeInMillis) {
        Assert.assertTrue((actualTimeInMillis >= expectedTimeInMillis - 2000L ? 1 : 0) != 0);
        Assert.assertTrue((actualTimeInMillis <= expectedTimeInMillis + 2000L ? 1 : 0) != 0);
    }

    private NetServer netServer(Vertx vertx) {
        NetServerOptions options = new NetServerOptions().setHost(DEFAULT_HOST).setPort(0).setTrafficShapingOptions(new TrafficShapingOptions().setInboundGlobalBandwidth(65536L).setOutboundGlobalBandwidth(65536L));
        return vertx.createNetServer(options);
    }

    private File setupFile(String testDir, String fileName, String content) throws Exception {
        File file = new File(testDir, fileName);
        if (file.exists()) {
            file.delete();
        }
        BufferedWriter out = new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(file), "UTF-8"));
        out.write(content);
        out.close();
        return file;
    }
}

