/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.core.http;

import io.netty.handler.codec.http.HttpHeaderNames;
import io.vertx.core.AbstractVerticle;
import io.vertx.core.DeploymentOptions;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.Http2TestBase;
import io.vertx.core.http.HttpClient;
import io.vertx.core.http.HttpClientResponse;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.HttpServer;
import io.vertx.core.http.HttpServerOptions;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.http.HttpTestBase;
import io.vertx.core.net.TrafficShapingOptions;
import io.vertx.test.core.TestUtils;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public class HttpBandwidthLimitingTest
extends Http2TestBase {
    private static final int OUTBOUND_LIMIT = 65536;
    private static final int INBOUND_LIMIT = 65536;
    private static final int TEST_CONTENT_SIZE = 262144;
    private final File sampleF = new File(new File(TestUtils.MAVEN_TARGET_DIR, "test-classes"), "test_traffic.txt");
    private final Handlers HANDLERS = new Handlers();
    private Function<Vertx, HttpServer> serverFactory;
    private Function<Vertx, HttpClient> clientFactory;

    @Parameterized.Parameters(name="HTTP {0}")
    public static Iterable<Object[]> data() {
        Function<Vertx, HttpServer> http1ServerFactory = v -> Providers.http1Server(v, 65536, 65536);
        Function<Vertx, HttpServer> http2ServerFactory = v -> Providers.http2Server(v, 65536, 65536);
        Function<Vertx, HttpClient> http1ClientFactory = v -> v.createHttpClient();
        Function<Vertx, HttpClient> http2ClientFactory = v -> v.createHttpClient(HttpBandwidthLimitingTest.createHttp2ClientOptions());
        return Arrays.asList({1.1, http1ServerFactory, http1ClientFactory}, {2.0, http2ServerFactory, http2ClientFactory});
    }

    public HttpBandwidthLimitingTest(double protoVersion, Function<Vertx, HttpServer> serverFactory, Function<Vertx, HttpClient> clientFactory) {
        this.serverFactory = serverFactory;
        this.clientFactory = clientFactory;
    }

    @Override
    @Before
    public void setUp() throws Exception {
        super.setUp();
        this.server = this.serverFactory.apply(this.vertx);
        this.client = this.clientFactory.apply(this.vertx);
    }

    @Override
    @After
    public void after() throws InterruptedException {
        CountDownLatch waitForClose = new CountDownLatch(1);
        this.vertx.close().onComplete(this.onSuccess(resp -> waitForClose.countDown()));
        this.awaitLatch(waitForClose);
    }

    @Test
    public void sendBufferThrottled() throws Exception {
        Buffer expectedBuffer = TestUtils.randomBuffer(262144);
        HttpServer testServer = this.serverFactory.apply(this.vertx);
        testServer.requestHandler(this.HANDLERS.bufferRead(expectedBuffer));
        this.startServer(testServer);
        long startTime = System.nanoTime();
        HttpClient testClient = this.clientFactory.apply(this.vertx);
        this.read(expectedBuffer, testServer, testClient);
        this.await();
        long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
        Assert.assertTrue((elapsedMillis > this.expectedTimeMillis(262144L, 65536) ? 1 : 0) != 0);
    }

    @Test
    public void sendFileIsThrottled() throws Exception {
        HttpServer testServer = this.serverFactory.apply(this.vertx);
        testServer.requestHandler(this.HANDLERS.getFile(this.sampleF));
        this.startServer(testServer);
        long startTime = System.nanoTime();
        HttpClient testClient = this.clientFactory.apply(this.vertx);
        AtomicLong receivedLength = new AtomicLong();
        long expectedLength = Files.size(Paths.get(this.sampleF.getAbsolutePath(), new String[0]));
        testClient.request(HttpMethod.GET, testServer.actualPort(), "localhost", "/get-file").compose(req -> req.send().andThen(this.onSuccess(resp -> this.assertEquals(200L, resp.statusCode()))).compose(HttpClientResponse::body)).onComplete(this.onSuccess(body -> {
            receivedLength.set(body.getBytes().length);
            Assert.assertEquals((long)expectedLength, (long)receivedLength.get());
            this.testComplete();
        }));
        this.await();
        long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
        Assert.assertTrue((elapsedMillis > this.expectedTimeMillis(receivedLength.get(), 65536) ? 1 : 0) != 0);
    }

    @Test
    public void dataUploadIsThrottled() throws Exception {
        Buffer expectedBuffer = TestUtils.randomBuffer(262144);
        HttpServer testServer = this.serverFactory.apply(this.vertx);
        testServer.requestHandler(this.HANDLERS.bufferWrite(expectedBuffer));
        this.startServer(testServer);
        long startTime = System.nanoTime();
        HttpClient testClient = this.clientFactory.apply(this.vertx);
        this.write(expectedBuffer, testServer, testClient);
        this.await();
        long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
        Assert.assertTrue((elapsedMillis > this.expectedTimeMillis(262144L, 65536) ? 1 : 0) != 0);
    }

    @Test
    public void fileUploadIsThrottled() throws Exception {
        HttpServer testServer = this.serverFactory.apply(this.vertx);
        testServer.requestHandler(this.HANDLERS.uploadFile(this.sampleF));
        this.startServer(testServer);
        long startTime = System.nanoTime();
        HttpClient testClient = this.clientFactory.apply(this.vertx);
        this.upload(testServer, testClient, this.sampleF);
        this.await();
        long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
        Assert.assertTrue((elapsedMillis > this.expectedTimeMillis(Files.size(Paths.get(this.sampleF.getAbsolutePath(), new String[0])), 65536) ? 1 : 0) != 0);
    }

    @Test
    public void testSendFileTrafficShapedWithSharedServers() throws InterruptedException, IOException {
        int numEventLoops = 2;
        Future listenLatch = this.vertx.deployVerticle(() -> new AbstractVerticle(){

            public void start(Promise<Void> startPromise) {
                HttpServer testServer = (HttpServer)HttpBandwidthLimitingTest.this.serverFactory.apply(this.vertx);
                testServer.requestHandler(HttpBandwidthLimitingTest.this.HANDLERS.getFile(HttpBandwidthLimitingTest.this.sampleF)).listen(HttpTestBase.DEFAULT_HTTP_PORT, "localhost").mapEmpty().onComplete(startPromise);
            }
        }, new DeploymentOptions().setInstances(numEventLoops));
        HttpClient testClient = this.clientFactory.apply(this.vertx);
        CountDownLatch waitForResponse = new CountDownLatch(2);
        AtomicLong startTime = new AtomicLong();
        AtomicLong totalReceivedLength = new AtomicLong();
        long expectedLength = Files.size(Paths.get(this.sampleF.getAbsolutePath(), new String[0]));
        listenLatch.onComplete(this.onSuccess(v -> {
            startTime.set(System.nanoTime());
            for (int i = 0; i < 2; ++i) {
                testClient.request(HttpMethod.GET, DEFAULT_HTTP_PORT, "localhost", "/get-file").compose(req -> req.send().andThen(this.onSuccess(resp -> this.assertEquals(200L, resp.statusCode()))).compose(HttpClientResponse::body)).onComplete(this.onSuccess(body -> {
                    long receivedBytes = body.getBytes().length;
                    totalReceivedLength.addAndGet(receivedBytes);
                    Assert.assertEquals((long)expectedLength, (long)receivedBytes);
                    waitForResponse.countDown();
                }));
            }
        }));
        this.awaitLatch(waitForResponse);
        long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime.get());
        Assert.assertTrue((elapsedMillis > this.expectedTimeMillis(totalReceivedLength.get(), 65536) ? 1 : 0) != 0);
    }

    private long expectedTimeMillis(long size, int rate) {
        return (long)((double)TimeUnit.MILLISECONDS.convert(size / (long)rate, TimeUnit.SECONDS) * 0.5);
    }

    private void read(Buffer expected, HttpServer server, HttpClient client) {
        client.request(HttpMethod.GET, server.actualPort(), "localhost", "/buffer-read").compose(req -> req.send().andThen(this.onSuccess(resp -> this.assertEquals(200L, resp.statusCode()))).compose(HttpClientResponse::body)).onComplete(this.onSuccess(body -> {
            this.assertEquals(expected.getBytes().length, body.getBytes().length);
            this.testComplete();
        }));
    }

    private void write(Buffer buffer, HttpServer server, HttpClient client) {
        client.request(HttpMethod.POST, server.actualPort(), "localhost", "/buffer-write").compose(req -> req.putHeader((CharSequence)HttpHeaderNames.CONTENT_LENGTH, (CharSequence)String.valueOf(buffer.length())).end(buffer));
    }

    private void upload(HttpServer server, HttpClient client, File expected) {
        Buffer b = this.vertx.fileSystem().readFileBlocking(expected.getAbsolutePath());
        client.request(HttpMethod.PUT, server.actualPort(), "localhost", "/upload-file").compose(req -> req.putHeader((CharSequence)HttpHeaderNames.CONTENT_LENGTH, (CharSequence)String.valueOf(expected.length())).putHeader((CharSequence)HttpHeaderNames.CONTENT_TYPE, (CharSequence)"application/binary").end(b));
    }

    static class Providers {
        Providers() {
        }

        private static HttpServer http1Server(Vertx vertx, int inboundLimit, int outboundLimit) {
            HttpServerOptions options = new HttpServerOptions().setHost("localhost").setPort(HttpTestBase.DEFAULT_HTTP_PORT).setTrafficShapingOptions(new TrafficShapingOptions().setInboundGlobalBandwidth((long)inboundLimit).setOutboundGlobalBandwidth((long)outboundLimit));
            return vertx.createHttpServer(options);
        }

        private static HttpServer http2Server(Vertx vertx, int inboundLimit, int outboundLimit) {
            HttpServerOptions options = Http2TestBase.createHttp2ServerOptions(HttpTestBase.DEFAULT_HTTP_PORT, "localhost").setTrafficShapingOptions(new TrafficShapingOptions().setInboundGlobalBandwidth((long)inboundLimit).setOutboundGlobalBandwidth((long)outboundLimit));
            return vertx.createHttpServer(options);
        }
    }

    class Handlers {
        Handlers() {
        }

        public Handler<HttpServerRequest> bufferRead(Buffer expectedBuffer) {
            return req -> {
                int len;
                req.response().setChunked(true);
                int start = 0;
                int chunkSize = 32768;
                for (int size = expectedBuffer.length(); size > 0; size -= len) {
                    len = Math.min(chunkSize, size);
                    req.response().write((Object)expectedBuffer.getBuffer(start, start + len));
                    start += len;
                }
                req.response().end();
            };
        }

        public Handler<HttpServerRequest> getFile(File expected) {
            return req -> req.response().sendFile(expected.getAbsolutePath());
        }

        public Handler<HttpServerRequest> bufferWrite(Buffer expected) {
            return req -> req.bodyHandler(buffer -> {
                HttpBandwidthLimitingTest.this.assertEquals(expected.getByteBuf(), buffer.getByteBuf());
                HttpBandwidthLimitingTest.this.testComplete();
            });
        }

        public Handler<HttpServerRequest> uploadFile(File expected) {
            return req -> req.endHandler(r -> {
                HttpBandwidthLimitingTest.this.assertEquals(expected.length(), req.bytesRead());
                HttpBandwidthLimitingTest.this.testComplete();
            });
        }
    }
}

