/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.tests.dns;

import io.netty.resolver.dns.DnsNameResolverTimeoutException;
import io.vertx.core.Vertx;
import io.vertx.core.VertxException;
import io.vertx.core.VertxOptions;
import io.vertx.core.dns.DnsClient;
import io.vertx.core.dns.DnsClientOptions;
import io.vertx.core.dns.MxRecord;
import io.vertx.core.dns.SrvRecord;
import io.vertx.test.core.TestUtils;
import io.vertx.test.core.VertxTestBase;
import io.vertx.test.fakedns.FakeDNSServer;
import io.vertx.test.netty.TestLoggerFactory;
import java.net.InetSocketAddress;
import java.util.Collections;
import java.util.TreeMap;
import java.util.concurrent.CountDownLatch;
import java.util.function.Function;
import org.apache.directory.server.dns.messages.DnsMessage;
import org.apache.directory.server.dns.store.RecordStore;
import org.junit.Test;

public class DNSTest
extends VertxTestBase {
    private FakeDNSServer dnsServer;

    @Override
    public void setUp() throws Exception {
        this.dnsServer = new FakeDNSServer();
        this.dnsServer.start();
        super.setUp();
    }

    @Override
    protected void tearDown() throws Exception {
        this.dnsServer.stop();
        super.tearDown();
    }

    @Test
    public void testIllegalArguments() throws Exception {
        this.dnsServer.testResolveAAAA("::1");
        DnsClient dns = this.prepareDns();
        TestUtils.assertNullPointerException(() -> dns.lookup(null));
        TestUtils.assertNullPointerException(() -> dns.lookup4(null));
        TestUtils.assertNullPointerException(() -> dns.lookup6(null));
        TestUtils.assertNullPointerException(() -> dns.resolveA(null));
        TestUtils.assertNullPointerException(() -> dns.resolveAAAA(null));
        TestUtils.assertNullPointerException(() -> dns.resolveCNAME(null));
        TestUtils.assertNullPointerException(() -> dns.resolveMX(null));
        TestUtils.assertNullPointerException(() -> dns.resolveTXT(null));
        TestUtils.assertNullPointerException(() -> dns.resolvePTR(null));
        TestUtils.assertNullPointerException(() -> dns.resolveNS(null));
        TestUtils.assertNullPointerException(() -> dns.resolveSRV(null));
    }

    @Test
    public void testDefaultDnsClient() throws Exception {
        this.testDefaultDnsClient(vertx -> vertx.createDnsClient());
    }

    @Test
    public void testDefaultDnsClientWithOptions() throws Exception {
        this.testDefaultDnsClient(vertx -> vertx.createDnsClient(new DnsClientOptions()));
    }

    private void testDefaultDnsClient(Function<Vertx, DnsClient> clientProvider) throws Exception {
        String ip = "10.0.0.1";
        this.dnsServer.testLookup4("10.0.0.1");
        VertxOptions vertxOptions = new VertxOptions();
        InetSocketAddress fakeServerAddress = this.dnsServer.localAddress();
        vertxOptions.getAddressResolverOptions().addServer(fakeServerAddress.getHostString() + ":" + fakeServerAddress.getPort());
        Vertx vertxWithFakeDns = Vertx.vertx((VertxOptions)vertxOptions);
        DnsClient dnsClient = clientProvider.apply(vertxWithFakeDns);
        dnsClient.lookup4("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertEquals("10.0.0.1", result);
            this.testComplete();
        }));
        this.await();
        vertxWithFakeDns.close();
    }

    @Test
    public void testResolveA() throws Exception {
        String ip = "10.0.0.1";
        this.dnsServer.testResolveA("10.0.0.1");
        DnsClient dns = this.prepareDns();
        dns.resolveA("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertFalse(result.isEmpty());
            this.assertEquals(1L, result.size());
            this.assertEquals("10.0.0.1", result.get(0));
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testUnresolvedDnsServer() throws Exception {
        try {
            DnsClient dns = this.vertx.createDnsClient(new DnsClientOptions().setHost("iamanunresolvablednsserver.com").setPort(53));
            this.fail();
        }
        catch (Exception e) {
            this.assertTrue(e instanceof IllegalArgumentException);
            this.assertEquals("Cannot resolve the host to a valid ip address", e.getMessage());
        }
    }

    @Test
    public void testResolveAIpV6() throws Exception {
        String ip = "10.0.0.1";
        this.dnsServer.testResolveA("10.0.0.1").ipAddress("::1");
        DnsClient dns = this.prepareDns();
        dns.resolveA("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertFalse(result.isEmpty());
            this.assertEquals(1L, result.size());
            this.assertEquals("10.0.0.1", result.get(0));
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testResolveAAAA() throws Exception {
        this.dnsServer.testResolveAAAA("::1");
        DnsClient dns = this.prepareDns();
        dns.resolveAAAA("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertFalse(result.isEmpty());
            this.assertEquals(1L, result.size());
            this.assertEquals("0:0:0:0:0:0:0:1", result.get(0));
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testResolveMX() throws Exception {
        String mxRecord = "mail.vertx.io";
        int prio = 10;
        this.dnsServer.testResolveMX(10, "mail.vertx.io");
        DnsClient dns = this.prepareDns();
        dns.resolveMX("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertFalse(result.isEmpty());
            this.assertEquals(1L, result.size());
            MxRecord record = (MxRecord)result.get(0);
            this.assertEquals(100L, record.ttl());
            this.assertEquals(10L, record.priority());
            this.assertEquals(record.name(), "mail.vertx.io");
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testResolveTXT() throws Exception {
        String txt = "vertx is awesome";
        this.dnsServer.testResolveTXT("vertx is awesome");
        DnsClient dns = this.prepareDns();
        dns.resolveTXT("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertFalse(result.isEmpty());
            this.assertEquals(1L, result.size());
            this.assertEquals("vertx is awesome", result.get(0));
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testResolveNS2() throws Exception {
        String ns = "ns.vertx.io";
        this.dnsServer.testResolveNS("ns.vertx.io");
        DnsClient dns = this.prepareDns();
        dns.resolveNS("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertFalse(result.isEmpty());
            this.assertEquals(1L, result.size());
            this.assertEquals("ns.vertx.io", result.get(0));
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testResolveCNAME() throws Exception {
        String cname = "cname.vertx.io";
        this.dnsServer.testResolveCNAME("cname.vertx.io");
        DnsClient dns = this.prepareDns();
        dns.resolveCNAME("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertFalse(result.isEmpty());
            this.assertEquals(1L, result.size());
            String record = (String)result.get(0);
            this.assertFalse(record.isEmpty());
            this.assertEquals("cname.vertx.io", record);
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testResolvePTR() throws Exception {
        String ptr = "ptr.vertx.io";
        this.dnsServer.testResolvePTR("ptr.vertx.io");
        DnsClient dns = this.prepareDns();
        dns.resolvePTR("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertEquals("ptr.vertx.io", result);
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testResolveSRV() throws Exception {
        int priority = 10;
        boolean weight = true;
        int port = 80;
        this.dnsServer.testResolveSRV("_svc._tcp.vertx.io", 10, 1, 80, "svc.vertx.io");
        DnsClient dns = this.prepareDns();
        dns.resolveSRV("_svc._tcp.vertx.io").onComplete(this.onSuccess(result -> {
            this.assertFalse(result.isEmpty());
            this.assertEquals(1L, result.size());
            SrvRecord record = (SrvRecord)result.get(0);
            this.assertEquals("vertx.io.", record.name());
            this.assertEquals("_svc", record.service());
            this.assertEquals("_tcp", record.protocol());
            this.assertEquals(100L, record.ttl());
            this.assertEquals(10L, record.priority());
            this.assertEquals(1L, record.weight());
            this.assertEquals(80L, record.port());
            this.assertEquals("svc.vertx.io", record.target());
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testResolveSRV2() throws Exception {
        int priority = 10;
        boolean weight = true;
        int port = 80;
        this.dnsServer.testResolveSRV2(10, 1, 80, "_svc._tcp.vertx.io");
        DnsClient dns = this.prepareDns();
        dns.resolveSRV("_svc._tcp.vertx.io").onComplete(this.onSuccess(result -> {
            this.assertFalse(result.isEmpty());
            this.assertEquals(2L, result.size());
            TreeMap<Integer, SrvRecord> map = new TreeMap<Integer, SrvRecord>();
            map.put(((SrvRecord)result.get(0)).port(), (SrvRecord)result.get(0));
            map.put(((SrvRecord)result.get(1)).port(), (SrvRecord)result.get(1));
            SrvRecord record = (SrvRecord)map.get(80);
            this.assertEquals("vertx.io.", record.name());
            this.assertEquals("_tcp", record.protocol());
            this.assertEquals("_svc", record.service());
            this.assertEquals(100L, record.ttl());
            this.assertEquals(10L, record.priority());
            this.assertEquals(1L, record.weight());
            this.assertEquals(80L, record.port());
            this.assertEquals("svc0.vertx.io", record.target());
            record = (SrvRecord)map.get(81);
            this.assertEquals("vertx.io.", record.name());
            this.assertEquals("_tcp", record.protocol());
            this.assertEquals("_svc", record.service());
            this.assertEquals(100L, record.ttl());
            this.assertEquals(10L, record.priority());
            this.assertEquals(1L, record.weight());
            this.assertEquals(81L, record.port());
            this.assertEquals("svc1.vertx.io", record.target());
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testLookup4() throws Exception {
        String ip = "10.0.0.1";
        this.dnsServer.testLookup4("10.0.0.1");
        DnsClient dns = this.prepareDns();
        dns.lookup4("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertEquals("10.0.0.1", result);
            DnsMessage msg = this.dnsServer.pollMessage();
            this.assertTrue(msg.isRecursionDesired());
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testLookup6() throws Exception {
        this.dnsServer.testLookup6("::1");
        DnsClient dns = this.prepareDns();
        dns.lookup6("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertEquals("0:0:0:0:0:0:0:1", result);
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testLookupWithARecord() throws Exception {
        String ip = "10.0.0.1";
        this.dnsServer.testLookup4(ip);
        DnsClient dns = this.prepareDns();
        dns.lookup("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertEquals(ip, result);
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testLookupWithAAAARecord() throws Exception {
        this.dnsServer.testLookup6("::1");
        DnsClient dns = this.prepareDns();
        dns.lookup("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertEquals("0:0:0:0:0:0:0:1", result);
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testTimeout() throws Exception {
        DnsClient dns = this.prepareDns(new DnsClientOptions().setHost("127.0.0.1").setPort(10000).setQueryTimeout(5000L));
        dns.lookup("vertx.io").onComplete(this.onFailure(err -> {
            if (err instanceof VertxException) {
                this.assertEquals("DNS query timeout for vertx.io.", err.getMessage());
            } else if (err instanceof DnsNameResolverTimeoutException) {
                DnsNameResolverTimeoutException dnsNameResolverTimeoutException = (DnsNameResolverTimeoutException)err;
            }
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testLookupNonExisting() throws Exception {
        this.dnsServer.testLookupNonExisting();
        DnsClient dns = this.prepareDns();
        dns.lookup("gfegjegjf.sg1").onComplete(this.onFailure(err -> this.testComplete()));
        this.await();
    }

    @Test
    public void testReverseLookupIpv4() throws Exception {
        this.dnsServer.testReverseLookup("1.0.0.10.in-addr.arpa");
        DnsClient dns = this.prepareDns();
        dns.reverseLookup("10.0.0.1").onComplete(this.onSuccess(result -> {
            this.assertEquals("vertx.io", result);
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testReverseLookupIpv6() throws Exception {
        String address = "::1";
        this.dnsServer.testReverseLookup("1.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.in-addr.arpa");
        DnsClient dns = this.prepareDns();
        dns.reverseLookup(address).onComplete(this.onSuccess(result -> {
            this.assertEquals("vertx.io", result);
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testLookup4CNAME() throws Exception {
        String cname = "cname.vertx.io";
        String ip = "10.0.0.1";
        this.dnsServer.testLookup4CNAME("cname.vertx.io", "10.0.0.1");
        DnsClient dns = this.prepareDns();
        dns.lookup4("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertEquals("10.0.0.1", result);
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testResolveMXWhenDNSRepliesWithDNAMERecord() throws Exception {
        DnsClient dns = this.prepareDns();
        this.dnsServer.testResolveDNAME("mail.vertx.io");
        dns.resolveMX("vertx.io").onComplete(this.onSuccess(lst -> {
            this.assertEquals(Collections.emptyList(), lst);
            this.testComplete();
        }));
        this.await();
    }

    private TestLoggerFactory testLogging(DnsClientOptions options) {
        String ip = "10.0.0.1";
        this.dnsServer.testResolveA("10.0.0.1");
        return TestUtils.testLogging(() -> {
            try {
                this.prepareDns(options).resolveA("10.0.0.1").onComplete(fut -> this.testComplete());
                this.await();
            }
            catch (Exception e) {
                this.fail(e);
            }
        });
    }

    @Test
    public void testDoNotLogActivity() throws Exception {
        TestLoggerFactory factory = this.testLogging(new DnsClientOptions().setLogActivity(false));
        this.assertFalse(factory.hasName("io.netty.handler.logging.LoggingHandler"));
    }

    @Test
    public void testRecursionDesired() throws Exception {
        String ip = "10.0.0.1";
        this.dnsServer.testResolveA("10.0.0.1");
        DnsClient dns = this.prepareDns(new DnsClientOptions().setRecursionDesired(true));
        dns.resolveA("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertFalse(result.isEmpty());
            this.assertEquals(1L, result.size());
            this.assertEquals("10.0.0.1", result.get(0));
            DnsMessage msg = this.dnsServer.pollMessage();
            this.assertTrue(msg.isRecursionDesired());
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testRecursionNotDesired() throws Exception {
        String ip = "10.0.0.1";
        this.dnsServer.testResolveA("10.0.0.1");
        DnsClient dns = this.prepareDns(new DnsClientOptions().setRecursionDesired(false));
        dns.resolveA("vertx.io").onComplete(this.onSuccess(result -> {
            this.assertFalse(result.isEmpty());
            this.assertEquals(1L, result.size());
            this.assertEquals("10.0.0.1", result.get(0));
            DnsMessage msg = this.dnsServer.pollMessage();
            this.assertFalse(msg.isRecursionDesired());
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testClose() throws Exception {
        this.disableThreadChecks();
        this.waitFor(2);
        String ip = "10.0.0.1";
        RecordStore store = this.dnsServer.testResolveA(ip).store();
        CountDownLatch latch1 = new CountDownLatch(1);
        CountDownLatch latch2 = new CountDownLatch(1);
        this.dnsServer.store(question -> {
            latch1.countDown();
            try {
                this.awaitLatch(latch2);
            }
            catch (Exception e) {
                this.fail(e);
            }
            return store.getRecords(question);
        });
        DnsClient dns = this.prepareDns();
        dns.resolveA("vertx.io").onComplete(this.onFailure(timeout -> {
            this.assertTrue(timeout.getMessage().contains("closed"));
            this.complete();
        }));
        this.awaitLatch(latch1);
        dns.close().onComplete(this.onSuccess(v -> {
            latch2.countDown();
            this.complete();
        }));
        this.await();
    }

    private DnsClient prepareDns() throws Exception {
        return this.prepareDns(new DnsClientOptions().setQueryTimeout(15000L));
    }

    private DnsClient prepareDns(DnsClientOptions options) throws Exception {
        InetSocketAddress addr = this.dnsServer.localAddress();
        return this.vertx.createDnsClient(new DnsClientOptions(options).setPort(addr.getPort()).setHost(addr.getAddress().getHostAddress()));
    }
}

